package eu.shiftforward.adstax.scheduler.action

import scala.concurrent.duration.FiniteDuration

import spray.json._

import eu.shiftforward.apso.config.Implicits._
import eu.shiftforward.apso.config.LazyConfigFactory

/**
 * Represents an action capable to be scheduled.
 */
trait SchedulerAction {

  /**
   * Returns the id of the action.
   */
  def id: String

  /**
   * Returns the description of the action.
   */
  def action: String

  /**
   * Returns the duration in which to run the action.
   */
  def in: FiniteDuration

  /**
   * Returns the duration in which to repeat the scheduled action, or None if it's an one-shot task.
   */
  def repeat: Option[FiniteDuration]
}

/**
 * Conversions to and from JSON for [[SchedulerAction]].
 */
object SchedulerAction {

  object JsonProtocol {
    val jsonFormatRegistry: Map[String, JsonFormat[SchedulerAction]] =
      LazyConfigFactory.load.getConfig("adstax.sdk.scheduler").getMap[String]("json-formats").mapValues { jfClass =>
        getClass.getClassLoader.loadClass(jfClass).newInstance().asInstanceOf[JsonFormat[SchedulerAction]]
      }

    /**
     * The JsonProtocol object, with conversions to and from JSON.
     */
    implicit object SchedulerActionJsonProtocol extends RootJsonFormat[SchedulerAction] {
      def read(v: JsValue): SchedulerAction = {
        v.asJsObject.fields.get("action") match {
          case Some(JsString(action)) if jsonFormatRegistry.contains(action) =>
            v.convertTo[SchedulerAction](jsonFormatRegistry(action))
          case Some(JsString(action)) => deserializationError(s"Unknown or unsupported: 'action' field value: $action")
          case _ => deserializationError("JSON object not recognized: missing 'action' field for a valid SchedulerAction!")
        }
      }

      def write(a: SchedulerAction): JsValue = a match {
        case a: SchedulerAction if jsonFormatRegistry.contains(a.action) => a.toJson(jsonFormatRegistry(a.action))
        case other => serializationError(s"Cannot serialize unrecognized Scheduler Action: ${other.getClass}!" +
          s" Does the JSON Format Registry contain a '${a.action}' entry for that Scheduler Action!?")
      }
    }
  }
}
