package eu.shiftforward.adstax.productfeeder.rpc

import eu.shiftforward.adstax.productfeeder.ProductFeederClient
import scala.concurrent.{ ExecutionContext, Future }

import akka.actor.ActorRefFactory
import eu.shiftforward.adstax.productfeeder.ProductItem
import eu.shiftforward.adstax.productfeeder.rpc.ProductFeederOperation.JsonProtocol._
import eu.shiftforward.adstax.productfeeder.rpc.ProductFeederOperationResult.JsonProtocol._
import eu.shiftforward.adstax.util.{ RmqRpcJsonClient, RmqRpcJsonClientTypeDescriptor }
import eu.shiftforward.apso.config.LazyConfigFactory
import com.typesafe.config.Config

/**
 * The AMQP RPC Client to interact with for the Product Feeder.
 */
class ProductFeederRmqRpcClient(
  actorRefFactory: ActorRefFactory,
  val rmqConfig: Config = LazyConfigFactory.load.getConfig("adstax.product-feeder.rabbitmq"),
  val rpcClientConfig: Config = LazyConfigFactory.load.getConfig("adstax.product-feeder.rpc"))(implicit val ec: ExecutionContext)
    extends ProductFeederClient with RmqRpcJsonClient {

  def rmqActorRefFactory = actorRefFactory

  implicit val getTypeDescriptor = new RmqRpcJsonClientTypeDescriptor[GetProduct, GetProductResult] {}
  implicit val updateTypeDescriptor = new RmqRpcJsonClientTypeDescriptor[UpdateProduct, UpdateProductResult] {}
  implicit val deleteTypeDescriptor = new RmqRpcJsonClientTypeDescriptor[DeleteProduct, DeleteProductResult] {}

  implicit val getRequestRoutingKey = GetProductTypeRoutingKey
  implicit val updateRequestRoutingKey = UpdateProductTypeRoutingKey
  implicit val deleteRequestRoutingKey = DeleteProductTypeRoutingKey

  def getProduct(clientId: String, siteId: String, productId: String): Future[Option[ProductItem]] =
    for {
      req <- dispatchRequest[GetProduct, GetProductResult] {
        GetProduct(ProductQuery(clientId, siteId, productId))
      }
    } yield (req.productResult)

  def updateProduct(
    clientId: String, siteId: String, productId: String, productAttributes: Map[String, Any]): Future[Boolean] =
    for {
      req <- dispatchRequest[UpdateProduct, UpdateProductResult] {
        UpdateProduct(clientId, siteId, ProductItem(productId, productAttributes))
      }
    } yield true

  def deleteProduct(clientId: String, siteId: String, productId: String): Future[Boolean] =
    for {
      req <- dispatchRequest[DeleteProduct, DeleteProductResult] {
        DeleteProduct(clientId, siteId, productId)
      }
    } yield true
}
