package eu.shiftforward.adstax.productfeeder.rpc

import eu.shiftforward.adstax.productfeeder.ProductItem
import eu.shiftforward.adstax.productfeeder.ProductItem.JsonProtocol._
import eu.shiftforward.apso.json.ExtraMiscJsonProtocol
import spray.json._

/**
 * Represents the supported responses from a given [[eu.shiftforward.adstax.productfeeder.rpc.ProductFeederOperation]].
 */
sealed trait ProductFeederOperationResult

/**
 * Represents the result of a GetProduct operation.
 *
 * @param productResult the optional product item
 */
case class GetProductResult(productResult: Option[ProductItem]) extends ProductFeederOperationResult

/**
 * Represents the result of an UpdateProduct operation.
 *
 * @param clientId the client id
 * @param siteId the site id
 * @param product the product itself
 */
case class UpdateProductResult(clientId: String, siteId: String, product: ProductItem)
  extends ProductFeederOperationResult

/**
 * Represents the result of an DeleteProduct operation.
 *
 * @param clientId the client id
 * @param siteId the site id
 * @param productId the id of the product
 */
case class DeleteProductResult(clientId: String, siteId: String, productId: String)
  extends ProductFeederOperationResult

/**
 * Represents the result of a DownloadFeed operation
 * @param productCount the number of inserted products
 */
case class DownloadFeedResult(productCount: Int, msg: Option[String] = None) extends ProductFeederOperationResult

object ProductFeederOperationResult {
  object JsonProtocol extends DefaultJsonProtocol with ExtraMiscJsonProtocol {
    implicit val getProductResultFormat = jsonFormat1(GetProductResult)
    implicit val updateProductResultFormat = jsonFormat3(UpdateProductResult)
    implicit val deleteProductResultFormat = jsonFormat3(DeleteProductResult)
    implicit val downloadFeedResultFormat = jsonFormat2(DownloadFeedResult)
  }
}
