package eu.shiftforward.adstax.productfeeder

import scala.concurrent.Future

/**
 * Interface for clients capable of interacting with the product feeder, by fetching, updating or deleting products.
 */
trait ProductFeederClient {
  /**
   * Retrieves a product.
   *
   * @param clientId the client id
   * @param siteId the site id
   * @param productId the product id
   * @return a `Future` with the ProductItem wrapped in a `Some` if it exists, `None` otherwise.
   */
  def getProduct(clientId: String, siteId: String, productId: String): Future[Option[ProductItem]]

  /**
   * Updates the product details.
   *
   * @param clientId the client id
   * @param siteId the site id
   * @param productId the product id
   * @param productAttributes a map of keys to product attributes to use for updating the current product definition
   * @return a `Future ` with `true` if the product update was successful, `false` otherwise.
   */
  def updateProduct(clientId: String, siteId: String, productId: String, productAttributes: Map[String, Any]): Future[Boolean]

  /**
   * Deletes the product details.
   *
   * @param clientId the client id
   * @param siteId the site id
   * @param productId the product id
   * @return a `Future` with `true` if the product delete was successful, `false` otherwise.
   */
  def deleteProduct(clientId: String, siteId: String, productId: String): Future[Boolean]
}
