package eu.shiftforward.adstax.user

import scala.concurrent.{ ExecutionContext, Future }

import eu.shiftforward.adstax.storage.UserAttributes.AttributeValue
import eu.shiftforward.adstax.storage.mergers._
import eu.shiftforward.adstax.storage.{ UserAttributes, UserProfileStorageClient }

trait UserProfileStorageContext {
  private[user] def upsClient: UserProfileStorageClient
}

/**
 * Representation of an user for a given client and an optional siteId.
 *
 * @param userId the id of the user
 * @param clientId the client id of the user
 * @param siteId the optional site id of the user
 */
case class User(userId: String, clientId: String, siteId: Option[String] = None) {
  /**
   * Returns all the attributes of the user, or `None` if the user doesn't have attributes.
   *
   * @param context the user profile storage context to use when fetching the user attributes
   * @param ec the execution context required for future handling
   *
   * @return a Future with the optional user attributes of the user.
   */
  def getAttributes(
    implicit
    context: UserProfileStorageContext, ec: ExecutionContext): Future[Option[UserAttributes]] = {
    context.upsClient.get(userId, clientId)
  }

  /**
   * Returns the attribute of the user associated with the given key, or `None` if the user doesn't have an attribute
   * with that name.
   *
   * @param name the param name to fetch
   * @param context the user profile storage context to use when fetching the user attributes
   * @param ec the execution context required for future handling
   *
   * @return a Future with the optional attribute.
   */
  def getAttribute(name: String)(
    implicit
    context: UserProfileStorageContext, ec: ExecutionContext): Future[Option[AttributeValue]] = {
    context.upsClient.get(userId, clientId).map(_.flatMap(_.attributes.get(name)))
  }

  /**
   * Sets the attributes for the given user, replacing all previously available in the store.
   *
   * @param attributes the attributes to set
   * @param context the user profile storage context to use when fetching the user attributes
   * @param ec the execution context required for future handling
   *
   * @return a Future that completes when the attributes are set.
   */
  def setAttributes(attributes: UserAttributes)(
    implicit
    context: UserProfileStorageContext, ec: ExecutionContext): Future[Unit] = {
    context.upsClient.update(userId, clientId, attributes, ReplaceMergingStrategy).map(_ => ())
  }

  /**
   * Sets the provided attribute for the given user. The attribute is either added if it didn't exist before or its
   * previous value is replaced with the new one.
   *
   * @param name the name of the attribute to set
   * @param value the new value for the attribute
   * @param context the user profile storage context to use when fetching the user attributes
   * @param ec the execution context required for future handling
   *
   * @return a Future that completes when the attribute is set.
   */
  def setAttribute(name: String, value: AttributeValue)(
    implicit
    context: UserProfileStorageContext, ec: ExecutionContext): Future[Unit] = {
    context.upsClient.update(userId, clientId, UserAttributes(Map(name -> value)), JoinMergingStrategy).map(_ => ())
  }
}
