package eu.shiftforward.adstax.scheduler.rpc

import scala.concurrent.{ ExecutionContext, Future }

import akka.actor.ActorRefFactory
import com.typesafe.config.Config

import eu.shiftforward.adstax.scheduler.SchedulerOperationResult.JsonProtocol._
import eu.shiftforward.adstax.scheduler._
import eu.shiftforward.adstax.scheduler.action.SchedulerAction
import eu.shiftforward.adstax.scheduler.rpc.SchedulerOperation.JsonProtocol._
import eu.shiftforward.adstax.util.{ RmqRpcJsonClient, RmqRpcJsonClientTypeDescriptor }

/**
 * The AMQP RPC Client to interact with the scheduler.
 */
trait SchedulerRmqRpcClient extends RmqRpcJsonClient with SchedulerClient {
  def actorRefFactory: ActorRefFactory
  def rmqActorRefFactory = actorRefFactory

  implicit val scheduleTypeDescriptor =
    new RmqRpcJsonClientTypeDescriptor[Schedule, ScheduledReply] {}

  implicit val cancelTypeDescriptor =
    new RmqRpcJsonClientTypeDescriptor[Cancel, CancelledReply] {}

  implicit val getStatusTypeDescriptor =
    new RmqRpcJsonClientTypeDescriptor[GetStatus, JobsStatusReply] {}

  implicit val getJobStatusTypeDescriptor =
    new RmqRpcJsonClientTypeDescriptor[GetJobStatus, Option[JobStatusReply]] {}

  implicit val scheduleRequestRoutingKey = ScheduleTypeRoutingKey
  implicit val cancelRequestRoutingKey = CancelTypeRoutingKey
  implicit val getStatusRequestRoutingKey = GetStatusTypeRoutingKey
  implicit val getJobStatusRequestRoutingKey = GetJobStatusTypeRoutingKey

  def scheduleAction(action: SchedulerAction): Future[SchedulerAction] = {
    dispatchRequest[Schedule, ScheduledReply] {
      Schedule(action)
    }.map {
      case Scheduled(scheduledAction) => scheduledAction
      case ScheduledError(_, t) => throw t
    }
  }

  def cancelAction(id: String): Future[Boolean] = {
    dispatchRequest[Cancel, CancelledReply] {
      Cancel(id)
    }.map {
      case Cancelled(_) => true
      case CancelledError(_, t) => throw t
    }
  }

  def getJobsStatus: Future[Map[String, ScheduledReply]] = {
    dispatchRequest[GetStatus, JobsStatusReply] {
      GetStatus
    }.map {
      case JobsStatus(jobs) => jobs
      case JobsStatusError(t) => throw t
    }
  }

  def getJobStatus(id: String): Future[Option[JobStatus]] = {
    dispatchRequest[GetJobStatus, Option[JobStatusReply]] {
      GetJobStatus(id)
    }.map {
      case Some(JobStatusError(t)) => throw t
      case other: Option[JobStatus] => other
    }
  }
}

object SchedulerRmqRpcClientFactory {
  def apply(_actorRefFactory: ActorRefFactory, _ec: ExecutionContext, _rmqConfig: Config,
    _rpcClientConfig: Config): SchedulerRmqRpcClient = {
    new SchedulerRmqRpcClient {
      def actorRefFactory = _actorRefFactory
      implicit def ec: ExecutionContext = _ec
      def rmqConfig = _rmqConfig
      def rpcClientConfig = _rpcClientConfig
    }
  }
}
