package eu.shiftforward.adstax.scheduler

import spray.json._
import spray.json.lenses.JsonLenses._

import eu.shiftforward.adstax.scheduler.action.SchedulerAction
import eu.shiftforward.adstax.scheduler.action.SchedulerAction.JsonProtocol._

/**
 * The reply type for all Schedule requests
 */
sealed trait ScheduledReply {
  def action: SchedulerAction
}

/**
 * The reply type for all Cancel requests
 */
sealed trait CancelledReply {
  def actionId: String
}
/**
 * Signals a Scheduler Error
 */
sealed trait SchedulerReplyError {
  def error: Throwable
}

case class Scheduled(action: SchedulerAction) extends ScheduledReply
case class ScheduledError(action: SchedulerAction, error: Throwable)
  extends ScheduledReply with SchedulerReplyError

case class Cancelled(actionId: String) extends CancelledReply
case class CancelledError(actionId: String, error: Throwable)
  extends CancelledReply with SchedulerReplyError

/**
 * Represents the supported responses from a given [[eu.shiftforward.adstax.scheduler.rpc.SchedulingRequestMessage]].
 * The reply can be a successful one, or an error
 */
trait SchedulingResponseMessage

sealed trait JobsStatusReply extends SchedulingResponseMessage
case class JobsStatus(jobs: Map[String, ScheduledReply]) extends JobsStatusReply
case class JobsStatusError(error: Throwable)
  extends JobsStatusReply with SchedulerReplyError

sealed trait JobStatusReply extends SchedulingResponseMessage
case class JobStatus(status: String, jobDefinition: SchedulerAction) extends JobStatusReply
case class JobStatusError(error: Throwable)
  extends JobStatusReply with SchedulerReplyError

object SchedulerOperationResult {
  /**
   * The set of JSON formats required to serialized and deserialize the reply messages
   */
  object JsonProtocol extends DefaultJsonProtocol {

    implicit object ThrowableJsonFormat extends JsonFormat[Throwable] {
      override def read(json: JsValue): Throwable = json match {
        case JsString(value) => new Throwable(value)
        case _ => deserializationError("Invalid error field: impossible to read error message from JSON!")
      }

      override def write(obj: Throwable): JsValue = obj.getMessage.toJson
    }

    // reads and writes Scheduled
    implicit val scheduledFormat: RootJsonFormat[Scheduled] = new RootJsonFormat[Scheduled] {
      private val f = jsonFormat1(Scheduled)
      override def read(json: JsValue): Scheduled = f.read(json)
      override def write(obj: Scheduled): JsValue = f.write(obj).update('status ! set("scheduled"))
    }

    // reads and writes ScheduledError
    implicit val scheduledErrorFormat: RootJsonFormat[ScheduledError] = new RootJsonFormat[ScheduledError] {
      private val f = jsonFormat2(ScheduledError)
      override def read(json: JsValue): ScheduledError = f.read(json)
      override def write(obj: ScheduledError): JsValue = f.write(obj).update('status ! set("scheduled-error"))
    }

    implicit object ScheduledReplyJsonFormat extends RootJsonFormat[ScheduledReply] {
      override def read(json: JsValue): ScheduledReply = json.asJsObject.getFields("status") match {
        case Seq(JsString("scheduled")) => scheduledFormat.read(json)
        case Seq(JsString("scheduled-error")) => scheduledErrorFormat.read(json)
        case _ => deserializationError("Invalid status field: impossible to read ScheduledReply from JSON!")
      }

      override def write(obj: ScheduledReply): JsValue = obj match {
        case s: Scheduled => scheduledFormat.write(s)
        case s: ScheduledError => scheduledErrorFormat.write(s)
      }
    }

    // reads and writes Cancelled
    implicit val cancelledFormat = new RootJsonFormat[Cancelled] {
      private val f = jsonFormat1(Cancelled)
      override def read(json: JsValue): Cancelled = f.read(json)
      override def write(obj: Cancelled): JsValue = f.write(obj).update('status ! set("cancelled"))
    }

    // reads and writes CancelledError
    implicit val cancelledErrorFormat = new RootJsonFormat[CancelledError] {
      private val f = jsonFormat2(CancelledError)
      override def read(json: JsValue): CancelledError = f.read(json)
      override def write(obj: CancelledError): JsValue = f.write(obj).update('status ! set("cancelled-error"))
    }

    // reads and writes CancelledReply
    implicit object CancelledReplyJsonFormat extends RootJsonFormat[CancelledReply] {
      override def read(json: JsValue): CancelledReply = json.asJsObject.getFields("status") match {
        case Seq(JsString("cancelled")) => cancelledFormat.read(json)
        case Seq(JsString("cancelled-error")) => cancelledErrorFormat.read(json)
        case _ => deserializationError("Invalid status field: impossible to read CancelledReply from JSON!")
      }

      override def write(obj: CancelledReply): JsValue = obj match {
        case c: Cancelled => cancelledFormat.write(c)
        case c: CancelledError => cancelledErrorFormat.write(c)
      }
    }

    // reads and writes JobsStatusReply
    implicit object JobsStatusReplyJsonFormat extends RootJsonFormat[JobsStatusReply] {
      override def read(json: JsValue): JobsStatusReply = SchedulingResponseMessageJsonFormat.read(json).asInstanceOf[JobsStatusReply]
      override def write(obj: JobsStatusReply): JsValue = SchedulingResponseMessageJsonFormat.write(obj)
    }

    // reads and writes JobsStatus
    implicit val jobsStatusFormat = new RootJsonFormat[JobsStatus] {
      override def write(js: JobsStatus) = JsObject(
        "type" -> JsString("jobs-status"),
        "jobs" -> js.jobs.toJson)

      // case class JobsStatus(jobs: List[SchedulerStatus]) extends JobsStatusReply
      override def read(json: JsValue): JobsStatus = json.asJsObject.fields("jobs") match {
        case JsObject(jobs) => JobsStatus(jobs.mapValues(_.convertTo[ScheduledReply]))
        case somethingElse => deserializationError(
          s"JobsStatus mandatory 'jobs' parameter is missing or has an invalid type. Got: $somethingElse.")
      }
    }

    // reads and writes JobsStatusError
    implicit val jobsStatusErrorFormat: RootJsonFormat[JobsStatusError] = new RootJsonFormat[JobsStatusError] {
      private val f = jsonFormat1(JobsStatusError)
      override def read(json: JsValue): JobsStatusError = f.read(json)
      override def write(obj: JobsStatusError): JsValue = f.write(obj).update('type ! set("jobs-status-error"))
    }

    // reads and writes JobStatusReply
    implicit object JobStatusReplyJsonFormat extends RootJsonFormat[JobStatusReply] {
      override def read(json: JsValue): JobStatusReply = SchedulingResponseMessageJsonFormat.read(json).asInstanceOf[JobStatusReply]
      override def write(obj: JobStatusReply): JsValue = SchedulingResponseMessageJsonFormat.write(obj)
    }

    // reads and writes JobStatus
    implicit val jobStatusFormat: RootJsonFormat[JobStatus] = new RootJsonFormat[JobStatus] {
      def write(js: JobStatus) = JsObject(
        "type" -> JsString("job-status"),
        "status" -> JsString(js.status),
        "jobDefinition" -> js.jobDefinition.toJson
      )
      def read(json: JsValue): JobStatus = json.asJsObject.getFields("status", "jobDefinition") match {
        case Seq(JsString(status), definition: JsObject) =>
          JobStatus(status, definition.convertTo[SchedulerAction])
      }
    }

    // reads and writes JobStatusError
    implicit val jobStatusErrorFormat: RootJsonFormat[JobStatusError] = new RootJsonFormat[JobStatusError] {
      private val f = jsonFormat1(JobStatusError)
      override def read(json: JsValue): JobStatusError = f.read(json)
      override def write(obj: JobStatusError): JsValue = f.write(obj).update('type ! set("job-status-error"))
    }

    // reads and writes SchedulingResponseMessage
    implicit object SchedulingResponseMessageJsonFormat extends RootJsonFormat[SchedulingResponseMessage] {
      override def read(json: JsValue): SchedulingResponseMessage = json.asJsObject.getFields("type") match {
        case Seq(JsString("jobs-status")) => jobsStatusFormat.read(json)
        case Seq(JsString("jobs-status-error")) => jobsStatusErrorFormat.read(json)
        case Seq(JsString("job-status")) => jobStatusFormat.read(json)
        case Seq(JsString("job-status-error")) => jobStatusErrorFormat.read(json)
      }

      override def write(obj: SchedulingResponseMessage): JsValue = obj match {
        case j: JobsStatus => jobsStatusFormat.write(j)
        case j: JobsStatusError => jobsStatusErrorFormat.write(j)
        case j: JobStatus => jobStatusFormat.write(j)
        case j: JobStatusError => jobStatusErrorFormat.write(j)
      }
    }
  }
}
