package eu.shiftforward.adstax.productfeeder.rpc

import java.net.URI

import eu.shiftforward.adstax.productfeeder.ProductItem
import eu.shiftforward.adstax.productfeeder.rpc.ProductQuery.JsonProtocol._
import eu.shiftforward.adstax.util.RmqRpcJsonTypeRoutingKey
import eu.shiftforward.apso.json.ExtraMiscJsonProtocol
import spray.json._

/**
 * Represents a supported operation of the ProductActor.
 */
sealed trait ProductFeederOperation {
  /**
   * The unique operation type for a concrete ProductOperation.
   */
  def operation: String = this.getClass.getSimpleName
}

/**
 * Represents get product operation.
 *
 * @param query the query for the product (userId, siteId, ...)
 */
case class GetProduct(query: ProductQuery) extends ProductFeederOperation

object GetProductTypeRoutingKey extends RmqRpcJsonTypeRoutingKey[GetProduct] {
  final val value = "getproduct.json"
}

/**
 * Represents a update product operation, for a given clientId and siteId, if provided.
 *
 * @param clientId the id of the client
 * @param siteId the optional id of the site,
 *               if none the product will be updated at a client level, if set it will update both
 */
case class UpdateProduct(clientId: String, siteId: String, product: ProductItem)
  extends ProductFeederOperation

object UpdateProductTypeRoutingKey extends RmqRpcJsonTypeRoutingKey[UpdateProduct] {
  final val value = "updateproduct.json"
}

/**
 * Represents a delete product operation, for a given clientId and siteId, if provided.
 *
 * @param clientId the id of the client
 * @param siteId the optional id of the site,
 *               if none the product will be delete at a client level, if set it will delete both
 * @param productId the id of the product
 */
case class DeleteProduct(clientId: String, siteId: String, productId: String)
  extends ProductFeederOperation

object DeleteProductTypeRoutingKey extends RmqRpcJsonTypeRoutingKey[DeleteProduct] {
  final val value = "deleteproduct.json"
}

sealed trait SupportedMimes {
  def mime: String
}
case object JSON extends SupportedMimes {
  val mime = "JSON"
}
case object XML extends SupportedMimes {
  val mime = "XML"
}

/**
 * Represents a product feed download and import operation.
 *
 * @param xmlUrl a valid URL pointing to a valid XML product feed
 */
case class DownloadFeed(clientId: String, siteId: String, xmlUrl: URI, mimeType: SupportedMimes)
  extends ProductFeederOperation

object DownloadFeedTypeRoutingKey extends RmqRpcJsonTypeRoutingKey[DownloadFeed] {
  final val value = "downloadfeed.json"
}

object ProductFeederOperation {
  object JsonProtocol extends DefaultJsonProtocol with ExtraMiscJsonProtocol {
    import ProductItem.JsonProtocol._

    implicit object SupportedMimesFormat extends RootJsonFormat[SupportedMimes] {
      override def read(json: JsValue): SupportedMimes = json match {
        case JsString("JSON") => JSON
        case JsString("XML") => XML
        case _ =>
          throw new DeserializationException("Invalid mime type")
      }
      override def write(obj: SupportedMimes): JsValue = obj match {
        case JSON => JsString("JSON")
        case XML => JsString("XML")
      }
    }

    implicit object URIFormat extends JsonFormat[URI] {
      def write(obj: URI) = JsString(obj.toString)

      def read(json: JsValue) = json match {
        case JsString(uri) => new URI(uri)
        case other => deserializationError("Expected URI, got: " + other)
      }
    }

    implicit val getProductFormat = jsonFormat1(GetProduct)
    implicit val updateProductsFormat = jsonFormat3(UpdateProduct)
    implicit val deleteProductsFormat = jsonFormat3(DeleteProduct)
    implicit val downloadFeedFormat = jsonFormat4(DownloadFeed)
  }
}
