/*
 * Decompiled with CFR 0.152.
 */
package ac.simons.oembed;

import ac.simons.oembed.AutodiscoveredOembedEndpoint;
import ac.simons.oembed.DefaultOembedResponseRenderer;
import ac.simons.oembed.DefaultRequestProvider;
import ac.simons.oembed.OembedEndpoint;
import ac.simons.oembed.OembedException;
import ac.simons.oembed.OembedJsonParser;
import ac.simons.oembed.OembedParser;
import ac.simons.oembed.OembedResponse;
import ac.simons.oembed.OembedResponseRenderer;
import ac.simons.oembed.OembedXmlParser;
import ac.simons.oembed.RequestProvider;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.function.Function;
import java.util.stream.Collectors;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.util.EntityUtils;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Entities;
import org.jsoup.parser.Parser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OembedService {
    public static final Logger LOGGER = LoggerFactory.getLogger((String)OembedService.class.getPackage().getName());
    private final HttpClient httpClient;
    private final Optional<CacheManager> cacheManager;
    private final String userAgent;
    private String applicationName;
    private final Map<OembedResponse.Format, OembedParser> parsers;
    private final Map<OembedEndpoint, RequestProvider> endpoints;
    private final Map<List<String>, OembedResponseRenderer> renderers;
    private boolean autodiscovery = false;
    private String cacheName = OembedService.class.getName();
    private long defaultCacheAge = 3600L;
    private final RequestProvider defaultRequestProvider = new DefaultRequestProvider();
    private final OembedResponseRenderer defaultRenderer = new DefaultOembedResponseRenderer();

    public OembedService(HttpClient httpClient, CacheManager cacheManager, List<OembedEndpoint> endpoints, String applicationName) {
        this.httpClient = httpClient;
        this.cacheManager = Optional.ofNullable(cacheManager);
        Properties version = new Properties();
        try {
            version.load(OembedService.class.getResourceAsStream("/oembed.properties"));
        }
        catch (IOException iOException) {
            // empty catch block
        }
        this.userAgent = String.format("Java/%s java-oembed2/%s", System.getProperty("java.version"), version.getProperty("de.dailyfratze.text.oembed.version"));
        this.applicationName = applicationName;
        EnumMap<OembedResponse.Format, OembedParser> hlp = new EnumMap<OembedResponse.Format, OembedParser>(OembedResponse.Format.class);
        hlp.put(OembedResponse.Format.json, new OembedJsonParser());
        hlp.put(OembedResponse.Format.xml, new OembedXmlParser());
        this.parsers = Collections.unmodifiableMap(hlp);
        this.endpoints = endpoints.stream().collect(Collectors.toMap(Function.identity(), endpoint -> {
            LOGGER.debug("Endpoint {} will match the following patterns: {}", (Object)endpoint.getName(), endpoint.getUrlSchemes());
            LOGGER.debug("Configuring request provider of type {} for endpoint {}...", endpoint.getRequestProviderClass(), (Object)endpoint.getName());
            LOGGER.debug("Using properties: {}", endpoint.getRequestProviderProperties());
            RequestProvider requestProvider = null;
            try {
                requestProvider = endpoint.getRequestProviderClass().getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
                BeanUtils.populate((Object)requestProvider, endpoint.getRequestProviderProperties());
            }
            catch (IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException ex) {
                throw new OembedException(ex);
            }
            return requestProvider;
        }));
        this.renderers = endpoints.stream().collect(Collectors.toMap(OembedEndpoint::getUrlSchemes, endpoint -> {
            LOGGER.debug("Configuring response renderer of type {} for endpoint {}...", endpoint.getResponseRendererClass(), (Object)endpoint.getName());
            LOGGER.debug("Using properties: {}", endpoint.getResponseRendererProperties());
            OembedResponseRenderer oembedResponseRenderer = null;
            try {
                oembedResponseRenderer = endpoint.getResponseRendererClass().getDeclaredConstructor(new Class[0]).newInstance(new Object[0]);
                BeanUtils.populate((Object)oembedResponseRenderer, endpoint.getResponseRendererProperties());
            }
            catch (IllegalAccessException | InstantiationException | NoSuchMethodException | InvocationTargetException ex) {
                throw new OembedException(ex);
            }
            return oembedResponseRenderer;
        }));
        LOGGER.debug("Oembed has {} endpoints and autodiscovery {} enabled...", (Object)this.endpoints.size(), (Object)(this.autodiscovery ? "is" : "is not"));
        LOGGER.info("Oembed ({}) ready...", (Object)this.userAgent);
    }

    public boolean isAutodiscovery() {
        return this.autodiscovery;
    }

    public void setAutodiscovery(boolean autodiscovery) {
        this.autodiscovery = autodiscovery;
    }

    public String getCacheName() {
        return this.cacheName;
    }

    public void setCacheName(String cacheName) {
        if (this.cacheManager.isPresent() && this.cacheManager.get().cacheExists(this.cacheName)) {
            this.cacheManager.get().removeCache(this.cacheName);
        }
        this.cacheName = cacheName;
    }

    public long getDefaultCacheAge() {
        return this.defaultCacheAge;
    }

    public void setDefaultCacheAge(long defaultCacheAge) {
        this.defaultCacheAge = defaultCacheAge;
    }

    final Optional<OembedEndpoint> findEndpointFor(String url) {
        Optional<OembedEndpoint> rv = this.endpoints.keySet().stream().filter(endpoint -> endpoint.getUrlSchemes().stream().map(String::trim).anyMatch(url::matches)).findFirst();
        if (!rv.isPresent() && this.autodiscovery) {
            try {
                HttpResponse httpResponse = this.httpClient.execute((HttpUriRequest)new HttpGet(url));
                if (httpResponse.getStatusLine().getStatusCode() != 200) {
                    LOGGER.warn("Autodiscovery for {} failed, server returned error {}: {}", new Object[]{url, httpResponse.getStatusLine().getStatusCode(), EntityUtils.toString((HttpEntity)httpResponse.getEntity())});
                } else {
                    Document document = Jsoup.parse((String)EntityUtils.toString((HttpEntity)httpResponse.getEntity(), (String)"UTF-8"), (String)url);
                    rv = document.getElementsByAttributeValue("rel", "alternate").stream().map(alternate -> {
                        AutodiscoveredOembedEndpoint autodiscoveredEndpoint = null;
                        try {
                            if (alternate.attr("type").equalsIgnoreCase("application/json+oembed")) {
                                autodiscoveredEndpoint = new AutodiscoveredOembedEndpoint(new URI(alternate.absUrl("href")), OembedResponse.Format.json);
                            } else if (alternate.attr("type").equalsIgnoreCase("text/xml+oembed")) {
                                autodiscoveredEndpoint = new AutodiscoveredOembedEndpoint(new URI(alternate.absUrl("href")), OembedResponse.Format.xml);
                            }
                        }
                        catch (URISyntaxException uRISyntaxException) {
                            // empty catch block
                        }
                        return autodiscoveredEndpoint;
                    }).filter(Objects::nonNull).findFirst();
                }
            }
            catch (IOException e) {
                LOGGER.warn("Autodiscovery for {} failed: {}", (Object)url, (Object)e.getMessage());
            }
        }
        return rv;
    }

    final InputStream executeRequest(HttpGet request) {
        InputStream rv = null;
        try {
            HttpResponse httpResponse = this.httpClient.execute((HttpUriRequest)request);
            if (httpResponse.getStatusLine().getStatusCode() != 200) {
                LOGGER.warn("Skipping '{}', server returned error {}: {}", new Object[]{request.getURI().toString(), httpResponse.getStatusLine().getStatusCode(), EntityUtils.toString((HttpEntity)httpResponse.getEntity())});
            } else {
                rv = httpResponse.getEntity().getContent();
            }
        }
        catch (IOException ex) {
            LOGGER.warn("Skipping '{}', could not get a response: {}", (Object)request.getURI().toString(), (Object)ex.getMessage());
        }
        return rv;
    }

    public Optional<OembedResponse> getOembedResponseFor(String url) {
        String trimmedUrl = Optional.ofNullable(url).map(String::trim).orElse("");
        if (trimmedUrl.isEmpty()) {
            LOGGER.debug("Ignoring empty url...");
            return Optional.empty();
        }
        Optional<OembedResponse> rv = this.cacheManager.map(cm -> cm.addCacheIfAbsent(this.cacheName).get((Serializable)((Object)trimmedUrl))).map(element -> (OembedResponse)element.getObjectValue());
        if (rv.isPresent()) {
            LOGGER.debug("Using OembedResponse from cache for '{}'...", (Object)trimmedUrl);
            return rv;
        }
        Optional<OembedEndpoint> endPoint = this.findEndpointFor(trimmedUrl);
        LOGGER.debug("Found endpoint {} for '{}'...", endPoint, (Object)trimmedUrl);
        rv = endPoint.map(ep -> this.endpoints.getOrDefault(ep, this.defaultRequestProvider).createRequestFor(this.userAgent, this.applicationName, ep.toApiUrl(trimmedUrl))).map(this::executeRequest).map(content -> {
            OembedResponse oembedResponse = null;
            try {
                oembedResponse = this.parsers.get((Object)((OembedEndpoint)endPoint.get()).getFormat()).unmarshal((InputStream)content);
            }
            catch (OembedException ex) {
                LOGGER.warn("Server returned an invalid oembed format for url '{}': {}", (Object)trimmedUrl, (Object)ex.getMessage());
            }
            return oembedResponse;
        });
        if (this.cacheManager.isPresent()) {
            Ehcache cache = this.cacheManager.get().addCacheIfAbsent(this.cacheName);
            int cacheAge = (int)Math.min(Math.max(60L, rv.map(OembedResponse::getCacheAge).orElse(this.defaultCacheAge)), Integer.MAX_VALUE);
            cache.put(new Element((Object)trimmedUrl, rv.orElse(null), cacheAge, cacheAge));
            LOGGER.debug("Cached {} for {} seconds for url '{}'...", new Object[]{rv, cacheAge, trimmedUrl});
        }
        return rv;
    }

    public String embedUrls(String textWithEmbeddableUrls, String baseUrl) {
        return this.embedUrls(textWithEmbeddableUrls, baseUrl, String.class);
    }

    public <T> T embedUrls(String textWithEmbeddableUrls, String baseUrl, Class<? extends T> targetClass) {
        String rv;
        Optional<String> optionalBaseUrl = Optional.ofNullable(baseUrl);
        if (String.class.isAssignableFrom(targetClass)) {
            rv = textWithEmbeddableUrls;
        } else if (Document.class.isAssignableFrom(targetClass)) {
            rv = Document.createShell((String)optionalBaseUrl.orElse(""));
        } else {
            throw new OembedException(String.format("Invalid target class: %s", targetClass.getName()));
        }
        if (textWithEmbeddableUrls != null && !textWithEmbeddableUrls.trim().isEmpty()) {
            Document document = this.embedUrls(Jsoup.parseBodyFragment((String)textWithEmbeddableUrls, (String)optionalBaseUrl.orElse("")));
            if (Document.class.isAssignableFrom(targetClass)) {
                rv = document;
            } else {
                document.outputSettings().prettyPrint(false).escapeMode(Entities.EscapeMode.xhtml).charset(StandardCharsets.UTF_8);
                rv = Parser.unescapeEntities((String)document.body().html().trim(), (boolean)true);
            }
        }
        return (T)rv;
    }

    public Document embedUrls(Document document) {
        for (org.jsoup.nodes.Element a : document.getElementsByTag("a")) {
            String absUrl = a.absUrl("href");
            Optional<String> html = this.getOembedResponseFor(absUrl).map(response -> {
                OembedResponseRenderer renderer = this.renderers.entrySet().stream().filter(entry -> ((List)entry.getKey()).stream().anyMatch(absUrl::matches)).findFirst().map(Map.Entry::getValue).orElse(this.defaultRenderer);
                return renderer.render((OembedResponse)response, a.clone());
            });
            if (!html.isPresent() || html.get().trim().isEmpty()) continue;
            a.before(html.get().trim());
            a.remove();
        }
        return document;
    }

    public OembedParser getParser(OembedResponse.Format format) {
        return this.parsers.get((Object)format);
    }
}

