/*
 * Decompiled with CFR 0.152.
 */
package eu.ill.webx.relay;

import eu.ill.webx.WebXClientConfiguration;
import eu.ill.webx.WebXHostConfiguration;
import eu.ill.webx.exceptions.WebXConnectionException;
import eu.ill.webx.exceptions.WebXDisconnectedException;
import eu.ill.webx.model.ClientIdentifier;
import eu.ill.webx.model.SessionId;
import eu.ill.webx.model.SocketResponse;
import eu.ill.webx.relay.WebXClient;
import eu.ill.webx.relay.WebXSession;
import eu.ill.webx.transport.Transport;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WebXHost {
    private static final Logger logger = LoggerFactory.getLogger(WebXHost.class);
    private final WebXHostConfiguration configuration;
    private final Transport transport = new Transport();
    private List<WebXSession> sessions = new ArrayList<WebXSession>();

    WebXHost(WebXHostConfiguration configuration) {
        this.configuration = configuration;
    }

    public String getHostname() {
        return this.configuration.getHostname();
    }

    public int getPort() {
        return this.configuration.getPort();
    }

    void connect() throws WebXConnectionException {
        if (!this.transport.isConnected()) {
            try {
                logger.info("Connecting to WebX server at {}:{}...", (Object)this.configuration.getHostname(), (Object)this.configuration.getPort());
                this.transport.connect(this.configuration.getHostname(), this.configuration.getPort(), this.configuration.getSocketTimeoutMs(), this.configuration.isStandalone(), this::onMessage);
                logger.info("... connected to {}", (Object)this.configuration.getHostname());
            }
            catch (WebXDisconnectedException e) {
                throw new WebXConnectionException("Failed to connect to WebX host");
            }
        }
    }

    void disconnect() {
        this.transport.disconnect();
    }

    public WebXClient onClientConnection(WebXClientConfiguration clientConfiguration) throws WebXConnectionException {
        if (this.transport.isConnected()) {
            SessionId sessionId;
            if (clientConfiguration.getSessionId() == null) {
                logger.info("Connecting to WebX using password authentication");
                sessionId = this.startSession(clientConfiguration);
                logger.info("Authentication successful. Got session Id \"{}\"", (Object)sessionId.hexString());
            } else {
                logger.info("Connecting to existing WebX session using sessionId \"{}\"", (Object)clientConfiguration.getSessionId());
                sessionId = new SessionId(clientConfiguration.getSessionId());
            }
            ClientIdentifier clientIdentifier = this.connectClient(sessionId);
            WebXSession session = this.getSession(sessionId).orElseGet(() -> {
                WebXSession webXSession = new WebXSession(sessionId, this.transport);
                webXSession.start();
                this.addSession(webXSession);
                return webXSession;
            });
            return session.createClient(clientIdentifier);
        }
        logger.error("Trying to create client but transport to host is not connected");
        throw new WebXConnectionException("Transport to host not connected when creating client");
    }

    public void onClientDisconnected(WebXClient client) {
        this.disconnectClient(client);
        this.getSession(client.getSessionId()).ifPresent(session -> {
            session.onClientDisconnected(client);
            if (session.getClientCount() == 0) {
                logger.debug("Client removed from session with Id \"{}\". Session now has no clients: stopping it", (Object)session.getSessionId().hexString());
                session.stop();
                this.removeSession((WebXSession)session);
            }
        });
    }

    public synchronized int getClientCount() {
        return this.sessions.stream().mapToInt(WebXSession::getClientCount).reduce(0, Integer::sum);
    }

    public synchronized void cleanupSessions() {
        this.sessions = this.sessions.stream().filter(session -> {
            if (session.getClientCount() == 0) {
                logger.debug("Cleanup: Session with Id \"{}\" has no clients: stopping it", (Object)session.getSessionId().hexString());
                session.stop();
                return false;
            }
            return true;
        }).toList();
    }

    private synchronized void addSession(WebXSession session) {
        this.sessions.add(session);
    }

    private synchronized void removeSession(WebXSession session) {
        this.sessions.remove(session);
    }

    private synchronized Optional<WebXSession> getSession(SessionId sessionId) {
        return this.sessions.stream().filter(session -> sessionId.equals(session.getSessionId())).findFirst();
    }

    private void onMessage(byte[] messageData) {
        logger.trace("Got client message of length {} from {}", (Object)messageData.length, (Object)this.configuration.getHostname());
        SessionId sessionId = new SessionId(messageData);
        this.getSession(sessionId).ifPresent(session -> session.onMessage(messageData));
    }

    private SessionId startSession(WebXClientConfiguration clientConfiguration) throws WebXConnectionException {
        try {
            String response = this.transport.startSession(clientConfiguration);
            String[] responseData = response.split(",");
            int responseCode = Integer.parseInt(responseData[0]);
            String sessionIdString = responseData[1];
            if (responseCode == 0) {
                return new SessionId(sessionIdString);
            }
            logger.error("Couldn't create WebX session: {}", (Object)sessionIdString);
            throw new WebXConnectionException("Couldn't create WebX session: session response invalid");
        }
        catch (WebXDisconnectedException e) {
            logger.error("Cannot start session: WebX Server is disconnected");
            throw new WebXConnectionException("WebX Server disconnected when creating WebX session");
        }
    }

    private ClientIdentifier connectClient(SessionId sessionId) throws WebXConnectionException {
        try {
            String request = String.format("connect,%s", sessionId.hexString());
            String response = this.transport.sendRequest(request).toString();
            if (response == null) {
                throw new WebXConnectionException("WebX Server returned a null connection response");
            }
            if (response.isEmpty()) {
                throw new WebXConnectionException(String.format("WebX Server refused connection with sessionId %s", sessionId.hexString()));
            }
            String[] responseElements = response.split(",");
            if (responseElements.length != 2) {
                throw new WebXConnectionException("WebX Server returned an invalid connection response");
            }
            String clientIdString = responseElements[0];
            String clientIndexString = responseElements[1];
            int clientId = Integer.parseUnsignedInt(responseElements[0], 16);
            long clientIndex = Long.parseUnsignedLong(responseElements[1], 16);
            logger.info("Client connected to WebX session \"{}\":  Got client Id \"{}\" and index \"{}\"", new Object[]{sessionId.hexString(), clientIdString, clientIndexString});
            return new ClientIdentifier(clientIndex, clientId);
        }
        catch (NumberFormatException exception) {
            logger.error("Cannot connect client: Failed to parse client id and index");
            throw new WebXConnectionException("Failed to parse client id and index");
        }
        catch (WebXDisconnectedException e) {
            logger.error("Cannot connect client: WebX Server is disconnected");
            throw new WebXConnectionException("WebX Server disconnected when creating WebX session");
        }
    }

    private void disconnectClient(WebXClient client) {
        if (client != null && client.isConnected()) {
            try {
                String request = String.format("disconnect,%s,%s", client.getSessionId().hexString(), client.getClientIdentifier().clientIdString());
                SocketResponse response = this.transport.sendRequest(request);
                if (response == null) {
                    logger.error("Failed to get response from WebX server");
                } else {
                    logger.info("Client (Id \"{}\" and index \"{}\") disconnected from WebX session \"{}\"", new Object[]{client.getClientIdentifier().clientIdString(), client.getClientIdentifier().clientIndexString(), client.getSessionId().hexString()});
                }
            }
            catch (WebXDisconnectedException e) {
                logger.warn("Cannot disconnect client {}: WebX Server is disconnected", (Object)client.getClientIdentifier().clientIdString());
            }
        }
    }
}

