/*
 * Decompiled with CFR 0.152.
 */
package eu.fbk.utils.eval;

import com.google.common.base.Preconditions;
import com.google.common.collect.Ordering;
import java.io.Serializable;
import java.util.Objects;

public final class PrecisionRecall
implements Serializable {
    private static final long serialVersionUID = 1L;
    public static final PrecisionRecall EMPTY = new PrecisionRecall(0.0, 0.0, 0.0, 0.0, Double.NaN, Double.NaN, Double.NaN);
    private final double tp;
    private final double fp;
    private final double fn;
    private final double tn;
    private final double precision;
    private final double recall;
    private final double accuracy;

    private PrecisionRecall(double tp, double fp, double fn, double tn, double precision, double recall, double accuracy) {
        this.tp = tp;
        this.fp = fp;
        this.fn = fn;
        this.tn = tn;
        this.precision = precision;
        this.recall = recall;
        this.accuracy = accuracy;
    }

    public static PrecisionRecall forCounts(double tp, double fp, double fn) {
        return PrecisionRecall.forCounts(tp, fp, fn, 0.0);
    }

    public static PrecisionRecall forCounts(double tp, double fp, double fn, double tn) {
        if (tp < 0.0 || fp < 0.0 || fn < 0.0 || tn < 0.0) {
            throw new IllegalArgumentException("Invalid contingency table values: tp=" + tp + ", fp=" + fp + ", fn=" + fn + ", tn=" + tn);
        }
        if (tp == 0.0 && fp == 0.0 && fn == 0.0 && tn == 0.0) {
            return EMPTY;
        }
        double c = tp + fp + fn + tn;
        double a = c == 0.0 ? Double.NaN : (tp + tn) / c;
        double p = tp + fp == 0.0 ? Double.NaN : tp / (tp + fp);
        double r = tp + fn == 0.0 ? Double.NaN : tp / (tp + fn);
        return new PrecisionRecall(tp, fp, fn, tn, p, r, a);
    }

    public static PrecisionRecall forMeasures(double precision, double recall, double accuracy) {
        return PrecisionRecall.forMeasures(precision, recall, accuracy, Double.NaN);
    }

    public static PrecisionRecall forMeasures(double precision, double recall, double accuracy, double count) {
        Preconditions.checkArgument(Double.isNaN(precision) || precision >= 0.0 && precision <= 1.0);
        Preconditions.checkArgument(Double.isNaN(recall) || recall >= 0.0 && recall <= 1.0);
        Preconditions.checkArgument(Double.isNaN(accuracy) || accuracy >= 0.0 && accuracy <= 1.0);
        Preconditions.checkArgument(Double.isNaN(count) || count >= 0.0);
        if (count == Double.NaN || precision == 0.0 || recall == 0.0) {
            return new PrecisionRecall(Double.NaN, Double.NaN, Double.NaN, Double.NaN, precision, recall, accuracy);
        }
        if (precision == 1.0 && recall == 1.0 || accuracy == 1.0) {
            return new PrecisionRecall(Double.NaN, 0.0, 0.0, Double.NaN, precision, recall, accuracy);
        }
        double tp = count * (1.0 - accuracy) / (1.0 / precision + 1.0 / recall - 2.0);
        double fp = tp * (1.0 / precision - 1.0);
        double fn = tp * (1.0 / recall - 1.0);
        double tn = count - tp - fp - fn;
        if (tn > 0.0) {
            return new PrecisionRecall(tp, fp, fn, tn, precision, recall, accuracy);
        }
        return new PrecisionRecall(Double.NaN, Double.NaN, Double.NaN, Double.NaN, precision, recall, accuracy);
    }

    public static PrecisionRecall microAverage(Iterable<PrecisionRecall> prs) {
        double tp = 0.0;
        double fp = 0.0;
        double fn = 0.0;
        double tn = 0.0;
        int num = 0;
        for (PrecisionRecall pr : prs) {
            tp += pr.tp;
            fp += pr.fp;
            fn += pr.fn;
            tn += pr.tn;
            ++num;
        }
        return num == 0 ? EMPTY : PrecisionRecall.forCounts(tp / (double)num, fp / (double)num, fn / (double)num, tn / (double)num);
    }

    public static PrecisionRecall macroAverage(Iterable<PrecisionRecall> prs) {
        double p = 0.0;
        double r = 0.0;
        double a = 0.0;
        double count = 0.0;
        int num = 0;
        for (PrecisionRecall pr : prs) {
            p += pr.getPrecision();
            r += pr.getRecall();
            a += pr.getAccuracy();
            count += pr.getCount();
            ++num;
        }
        return PrecisionRecall.forMeasures(p, r, a, num == 0 ? 0.0 : count / (double)num);
    }

    public double getTP() {
        return this.tp;
    }

    public double getFP() {
        return this.fp;
    }

    public double getFN() {
        return this.fn;
    }

    public double getTN() {
        return this.tn;
    }

    public double getCount() {
        return this.tp + this.fp + this.tn + this.fn;
    }

    public double getAccuracy() {
        return this.accuracy;
    }

    public double getPrecision() {
        return this.precision;
    }

    public double getRecall() {
        return this.recall;
    }

    public double getF1() {
        return this.precision + this.recall == 0.0 ? Double.NaN : 2.0 * this.precision * this.recall / (this.precision + this.recall);
    }

    public double get(Measure measure) {
        switch (measure) {
            case ACCURACY: {
                return this.getAccuracy();
            }
            case PRECISION: {
                return this.getPrecision();
            }
            case RECALL: {
                return this.getRecall();
            }
            case F1: {
                return this.getF1();
            }
        }
        throw new IllegalArgumentException("Invalid measure " + (Object)((Object)measure));
    }

    public boolean equals(Object object) {
        if (object == this) {
            return true;
        }
        if (!(object instanceof PrecisionRecall)) {
            return false;
        }
        PrecisionRecall other = (PrecisionRecall)object;
        return this.tp == other.tp && this.fp == other.fp && this.tn == other.tn && this.fn == other.fn && this.precision == other.precision && this.recall == other.recall && this.accuracy == other.accuracy;
    }

    public int hashCode() {
        return Objects.hash(this.tp, this.fp, this.tn, this.fn, this.precision, this.recall, this.accuracy);
    }

    public String toString() {
        return String.format("p=%.3f r=%.3f f1=%.3f a=%.3f tp=%.2f fp=%.2f fn=%.2f tn=%.2f", this.precision, this.recall, this.getF1(), this.accuracy, this.tp, this.fp, this.fn, this.tn);
    }

    public static Ordering<PrecisionRecall> comparator(final Measure measure, final boolean higherIsBetter) {
        return new Ordering<PrecisionRecall>(){

            @Override
            public int compare(PrecisionRecall left, PrecisionRecall right) {
                double leftValue = left.get(measure);
                double rightValue = right.get(measure);
                int result = Double.compare(leftValue, rightValue);
                return higherIsBetter ? -result : result;
            }
        };
    }

    public static Evaluator evaluator() {
        return new Evaluator();
    }

    public static final class Evaluator {
        private double tp = 0.0;
        private double fp = 0.0;
        private double fn = 0.0;
        private double tn = 0.0;
        private PrecisionRecall pr = null;

        private Evaluator() {
        }

        private static void checkNotNegative(double value) {
            if (value < 0.0) {
                throw new IllegalArgumentException("Illegal negative value " + value);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Evaluator add(PrecisionRecall pr) {
            if (pr.getCount() > 0.0) {
                Evaluator evaluator = this;
                synchronized (evaluator) {
                    this.pr = null;
                    this.tp += pr.getTP();
                    this.fp += pr.getFP();
                    this.fn += pr.getFN();
                    this.tn += pr.getTN();
                }
            }
            return this;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Evaluator add(Evaluator evaluator) {
            Evaluator evaluator2 = evaluator;
            synchronized (evaluator2) {
                if (evaluator.tp > 0.0 || evaluator.fp > 0.0 || evaluator.fn > 0.0 || evaluator.tn > 0.0) {
                    Evaluator evaluator3 = this;
                    synchronized (evaluator3) {
                        this.pr = null;
                        this.tp += evaluator.tp;
                        this.fp += evaluator.fp;
                        this.fn += evaluator.fn;
                        this.tn += evaluator.tn;
                    }
                }
            }
            return this;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Evaluator add(double deltaTP, double deltaFP, double deltaFN) {
            Evaluator.checkNotNegative(deltaTP);
            Evaluator.checkNotNegative(deltaFP);
            Evaluator.checkNotNegative(deltaFN);
            if (deltaTP > 0.0 || deltaFP > 0.0 || deltaFN > 0.0) {
                Evaluator evaluator = this;
                synchronized (evaluator) {
                    this.pr = null;
                    this.tp += deltaTP;
                    this.fp += deltaFP;
                    this.fn += deltaFN;
                }
            }
            return this;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Evaluator add(double deltaTP, double deltaFP, double deltaFN, double deltaTN) {
            Evaluator.checkNotNegative(deltaTP);
            Evaluator.checkNotNegative(deltaFP);
            Evaluator.checkNotNegative(deltaFN);
            Evaluator.checkNotNegative(deltaTN);
            if (deltaTP > 0.0 || deltaFP > 0.0 || deltaFN > 0.0 || deltaTN > 0.0) {
                Evaluator evaluator = this;
                synchronized (evaluator) {
                    this.pr = null;
                    this.tp += deltaTP;
                    this.fp += deltaFP;
                    this.fn += deltaFN;
                    this.tn += deltaTN;
                }
            }
            return this;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Evaluator addTP(double deltaTP) {
            Evaluator.checkNotNegative(deltaTP);
            if (deltaTP != 0.0) {
                Evaluator evaluator = this;
                synchronized (evaluator) {
                    this.pr = null;
                    this.tp += deltaTP;
                }
            }
            return this;
        }

        public Evaluator addTP() {
            return this.addTP(1.0);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Evaluator addFP(double deltaFP) {
            Evaluator.checkNotNegative(deltaFP);
            if (deltaFP != 0.0) {
                Evaluator evaluator = this;
                synchronized (evaluator) {
                    this.pr = null;
                    this.fp += deltaFP;
                }
            }
            return this;
        }

        public Evaluator addFP() {
            return this.addFP(1.0);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Evaluator addFN(double deltaFN) {
            Evaluator.checkNotNegative(deltaFN);
            if (deltaFN != 0.0) {
                Evaluator evaluator = this;
                synchronized (evaluator) {
                    this.pr = null;
                    this.fn += deltaFN;
                }
            }
            return this;
        }

        public Evaluator addFN() {
            return this.addFN(1.0);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Evaluator addTN(double deltaTN) {
            Evaluator.checkNotNegative(deltaTN);
            if (deltaTN != 0.0) {
                Evaluator evaluator = this;
                synchronized (evaluator) {
                    this.pr = null;
                    this.tn += deltaTN;
                }
            }
            return this;
        }

        public Evaluator addTN() {
            return this.addTN(1.0);
        }

        public synchronized PrecisionRecall getResult() {
            if (this.pr == null) {
                this.pr = PrecisionRecall.forCounts(this.tp, this.fp, this.fn, this.tn);
            }
            return this.pr;
        }
    }

    public static enum Measure {
        ACCURACY,
        PRECISION,
        RECALL,
        F1;

    }
}

