/*
 * Decompiled with CFR 0.152.
 */
package eu.europeana.sitemap.service.update;

import eu.europeana.features.S3ObjectStorageClient;
import eu.europeana.sitemap.SitemapType;
import eu.europeana.sitemap.StorageFileName;
import eu.europeana.sitemap.config.PortalUrl;
import eu.europeana.sitemap.service.Deployment;
import java.util.Date;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class SitemapGenerator {
    private static final Logger LOG = LogManager.getLogger(SitemapGenerator.class);
    private static final String XML_HEADER = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
    private static final String SITEMAP_HEADER_OPENING = "<sitemapindex xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\">";
    private static final String SITEMAP_HEADER_CLOSING = "</sitemapindex>";
    private static final String SITEMAP_OPENING = "<sitemap>";
    private static final String SITEMAP_CLOSING = "</sitemap>";
    private static final String URLSET_HEADER = "<urlset xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\" xmlns:image=\"http://www.google.com/schemas/sitemap-image/1.1\" xmlns:geo=\"http://www.google.com/geo/schemas/sitemap/1.0\">";
    private static final String URLSET_HEADER_CLOSING = "</urlset>";
    private static final String URL_OPENING = "<url>";
    private static final String URL_CLOSING = "</url>";
    private static final String LOC_OPENING = "<loc>";
    private static final String LOC_CLOSING = "</loc>";
    private static final String PRIORITY_OPENING = "<priority>";
    private static final String PRIORITY_CLOSING = "</priority>";
    private static final String LASTMOD_OPENING = "<lastmod>";
    private static final String LASTMOD_CLOSING = "</lastmod>";
    private static final char LN = '\n';
    private static final String FROM_PARAM = "?from=";
    private static final String TO_PARAM = "&to=";
    private static final int MAX_SAVE_ATTEMPTS = 3;
    private static final long RETRY_SAVE_INTERVAL = 5000L;
    private static final int MS_PER_SEC = 1000;
    private final S3ObjectStorageClient objectStorage;
    private final SitemapType type;
    private Deployment deployment;
    private String websiteBaseUrl;
    private int itemsPerSitemap;
    private boolean generationStarted;
    private boolean generationFinished;
    private StringBuilder sitemapIndex;
    private StringBuilder sitemap;
    private long fileStartTime;
    private long nrRecords;
    private int nrSitemaps;
    private long from;

    public SitemapGenerator(SitemapType type, S3ObjectStorageClient objectStorage) {
        this.objectStorage = objectStorage;
        this.type = type;
    }

    public void init(Deployment desiredDeployment, String websiteBaseUrl, int itemsPerSitemap) {
        if (this.generationStarted) {
            throw new IllegalStateException("Cannot start " + this.type + "sitemap generation. It's already started.");
        }
        LOG.info("Starting {} sitemap generation. Location of sitemap files {}", (Object)this.type, (Object)websiteBaseUrl);
        this.deployment = desiredDeployment;
        this.websiteBaseUrl = websiteBaseUrl;
        this.itemsPerSitemap = itemsPerSitemap;
        this.generationStarted = true;
        this.nrRecords = 0L;
        this.nrSitemaps = 0;
        this.initSitemapIndex();
        this.initSitemapFile();
    }

    public void addItem(String url, String priority, Date dateLastModified) {
        if (this.generationFinished) {
            throw new IllegalStateException("Cannot add item; " + this.type + " sitemap generation is already finished.");
        }
        if (!this.generationStarted) {
            throw new IllegalStateException("Cannot add item; " + this.type + " sitemap generation is not started yet.");
        }
        this.appendItem(url, priority, dateLastModified);
        if (this.nrRecords % (long)this.itemsPerSitemap == 0L) {
            this.finishSitemapFile();
        }
    }

    public void finish() {
        if (!this.generationStarted) {
            throw new IllegalStateException("Cannot complete " + this.type + " sitemap generation. It hasn't started yet.");
        }
        if (this.generationFinished) {
            throw new IllegalStateException("Cannot complete " + this.type + " sitemap generation. It was already finished.");
        }
        this.finishSitemapFile();
        this.finishSitemapIndex();
        LOG.info("Items processed {}, written {} sitemap files and 1 sitemap index file", (Object)this.nrRecords, (Object)this.nrSitemaps);
        this.generationFinished = true;
    }

    private void initSitemapIndex() {
        LOG.debug("Starting new index...");
        this.sitemapIndex = new StringBuilder().append(XML_HEADER).append('\n').append(SITEMAP_HEADER_OPENING).append('\n');
    }

    private void finishSitemapIndex() {
        this.sitemapIndex.append(SITEMAP_HEADER_CLOSING);
        String fileName = StorageFileName.getSitemapIndexFileName((SitemapType)this.type, (Deployment)this.deployment);
        String fileContents = this.sitemapIndex.toString();
        LOG.debug("Generated contents for sitemap index\n{}", (Object)fileContents);
        if (this.saveToStorage(fileName, fileContents)) {
            LOG.info("Created sitemap file {}", (Object)fileName);
        }
    }

    private void initSitemapFile() {
        if (this.sitemap != null) {
            throw new IllegalStateException("Cannot start new sitemap file. Existing one isn't done yet");
        }
        LOG.debug("Starting new sitemap file...");
        this.fileStartTime = System.currentTimeMillis();
        this.from = this.nrRecords + 1L;
        this.sitemap = new StringBuilder().append(XML_HEADER).append('\n').append(URLSET_HEADER).append('\n');
    }

    private void finishSitemapFile() {
        if (this.sitemap == null) {
            throw new IllegalStateException("No sitemap file to finish!");
        }
        String fromToText = FROM_PARAM + this.from + TO_PARAM + this.nrRecords;
        String sitemapFileName = PortalUrl.getSitemapUrlEncoded((String)this.websiteBaseUrl, (SitemapType)this.type, (String)fromToText);
        LOG.debug("Add sitemap file {} to index", (Object)sitemapFileName);
        this.sitemapIndex.append(SITEMAP_OPENING).append('\n').append(LOC_OPENING).append(sitemapFileName).append(LOC_CLOSING).append('\n').append(SITEMAP_CLOSING).append('\n');
        this.sitemap.append(URLSET_HEADER_CLOSING);
        String fileName = StorageFileName.getSitemapFileName((SitemapType)this.type, (Deployment)this.deployment, (String)fromToText);
        String fileContents = this.sitemap.toString();
        LOG.debug("Generated contents for file {}\n{}", (Object)fileName, (Object)fileContents);
        ++this.nrSitemaps;
        if (this.saveToStorage(fileName, fileContents)) {
            LOG.info("Created sitemap file {} in {} ms", (Object)fileName, (Object)(System.currentTimeMillis() - this.fileStartTime));
        }
        this.sitemap = null;
        this.initSitemapFile();
    }

    private void appendItem(String url, String priority, Date dateLastModified) {
        this.sitemap.append(URL_OPENING).append('\n').append(LOC_OPENING).append(url).append(LOC_CLOSING).append('\n');
        if (!StringUtils.isEmpty((CharSequence)priority)) {
            this.sitemap.append(PRIORITY_OPENING).append(priority).append(PRIORITY_CLOSING).append('\n');
        }
        if (dateLastModified != null) {
            this.sitemap.append(LASTMOD_OPENING).append(DateFormatUtils.format((Date)dateLastModified, (String)DateFormatUtils.ISO_8601_EXTENDED_DATE_FORMAT.getPattern())).append(LASTMOD_CLOSING).append('\n');
        }
        this.sitemap.append(URL_CLOSING).append('\n');
        ++this.nrRecords;
    }

    private boolean saveToStorage(String key, String contents) {
        int nrSaveAttempts;
        boolean result = false;
        LOG.debug("Saving file with key {} and contents {}", (Object)key, (Object)contents);
        String eTag = this.objectStorage.putObject(key, contents);
        LOG.debug("Checking if file {} exists...", (Object)key);
        result = this.checkIfFileExists(key);
        for (nrSaveAttempts = 1; (StringUtils.isEmpty((CharSequence)eTag) || !result) && nrSaveAttempts < 3; ++nrSaveAttempts) {
            long timeout = (long)nrSaveAttempts * 5000L;
            LOG.warn("Failed to save file {} to storage provider (etag = {}, siteMapCacheFileExists={}). Waiting {} seconds before trying again...", (Object)key, (Object)eTag, (Object)result, (Object)(timeout / 1000L));
            try {
                Thread.sleep(timeout);
            }
            catch (InterruptedException e) {
                LOG.error("Waiting period to retry saving to storage was interrupted", (Throwable)e);
                Thread.currentThread().interrupt();
            }
            LOG.info("Retry saving the file...");
            eTag = this.objectStorage.putObject(key, contents);
            result = this.checkIfFileExists(key);
        }
        if (nrSaveAttempts >= 3) {
            LOG.error("Failed to save file {} to storage provider. Giving up because we retried it {} times.", (Object)key, (Object)nrSaveAttempts);
        }
        return result;
    }

    private boolean checkIfFileExists(String id) {
        return this.objectStorage.isObjectAvailable(id);
    }
}

