/*
 * Decompiled with CFR 0.152.
 */
package eu.europeana.features;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.http.IdleConnectionReaper;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.S3ClientOptions;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.Bucket;
import com.amazonaws.services.s3.model.DeleteObjectRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.PutObjectResult;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.util.BinaryUtils;
import com.amazonaws.util.IOUtils;
import eu.europeana.domain.ContentValidationException;
import eu.europeana.domain.ObjectMetadata;
import eu.europeana.domain.ObjectStorageClientException;
import eu.europeana.domain.StorageObject;
import eu.europeana.features.ObjectStorageClient;
import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.Properties;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jclouds.io.Payload;
import org.jclouds.io.payloads.ByteArrayPayload;

public class S3ObjectStorageClient
implements ObjectStorageClient {
    private static final Logger LOG = LogManager.getLogger(S3ObjectStorageClient.class);
    private static final String ERROR_MSG_RETRIEVE = "Error retrieving storage object ";
    private static final String OBJECT_STORAGE_PROPERTY_FILE = "objectstorage.properties";
    private static final String VALIDATE_AFTER_INACTIVITY_PROPERTY = "s3.validate.after.inactivity";
    private static final int VALIDATE_AFTER_INACTIVITY_DEFAULT_VALUE = 2000;
    private AmazonS3 client;
    private String bucketName;
    private boolean isIbmCloud = false;

    private static Properties loadProperties(String fileName, boolean required) {
        Properties prop = new Properties();
        try (InputStream inputStream = S3ObjectStorageClient.class.getClassLoader().getResourceAsStream(fileName);){
            if (inputStream == null) {
                if (required) {
                    throw new FileNotFoundException("Please provide " + fileName + " file");
                }
                LOG.warn("Property file {} not found", (Object)fileName);
            }
            prop.load(inputStream);
        }
        catch (IOException e) {
            LOG.error("Error reading the property file {} ", (Object)fileName, (Object)e);
        }
        return prop;
    }

    private static int getValidateAfterInactivity(Properties props) {
        String value = props.getProperty(VALIDATE_AFTER_INACTIVITY_PROPERTY);
        return value != null ? Integer.parseInt(value) : 2000;
    }

    public S3ObjectStorageClient(String clientKey, String secretKey, String region, String bucketName) {
        Properties props = S3ObjectStorageClient.loadProperties(OBJECT_STORAGE_PROPERTY_FILE, true);
        BasicAWSCredentials credentials = new BasicAWSCredentials(clientKey, secretKey);
        ClientConfiguration clientConfiguration = new ClientConfiguration().withValidateAfterInactivityMillis(S3ObjectStorageClient.getValidateAfterInactivity(props));
        this.client = (AmazonS3)((AmazonS3ClientBuilder)((AmazonS3ClientBuilder)((AmazonS3ClientBuilder)AmazonS3ClientBuilder.standard().withCredentials((AWSCredentialsProvider)new AWSStaticCredentialsProvider((AWSCredentials)credentials))).withClientConfiguration(clientConfiguration)).withRegion(region)).build();
        this.bucketName = bucketName;
        LOG.info("Connected to Amazon S3 bucket {}, region {}, validateAfterInactivity {} ms ", (Object)bucketName, (Object)region, (Object)clientConfiguration.getValidateAfterInactivityMillis());
    }

    public S3ObjectStorageClient(String clientKey, String secretKey, String region, String bucketName, String endpoint) {
        System.setProperty("com.amazonaws.sdk.disableDNSBuckets", "True");
        S3ClientOptions.Builder optionsBuilder = S3ClientOptions.builder().setPathStyleAccess(true);
        this.client = new AmazonS3Client((AWSCredentials)new BasicAWSCredentials(clientKey, secretKey));
        this.client.setS3ClientOptions(optionsBuilder.build());
        this.client.setEndpoint(endpoint);
        this.bucketName = bucketName;
        this.isIbmCloud = true;
        LOG.info("Connected to IBM Cloud S3 bucket {}, region {}", (Object)bucketName, (Object)region);
    }

    public S3ObjectStorageClient(String clientKey, String secretKey, String bucketName, String endpoint, S3ClientOptions s3ClientOptions) {
        this.client = new AmazonS3Client((AWSCredentials)new BasicAWSCredentials(clientKey, secretKey));
        this.client.setS3ClientOptions(s3ClientOptions);
        this.client.setEndpoint(endpoint);
        this.bucketName = bucketName;
        LOG.info("Connected to S3 bucket {}", (Object)bucketName);
    }

    @Override
    public String getName() {
        return this.isIbmCloud ? "IBM Cloud S3" : "Amazon S3";
    }

    @Override
    public String getBucketName() {
        return this.bucketName;
    }

    @Override
    public List<StorageObject> list() {
        ObjectListing objectListing = this.client.listObjects(this.bucketName);
        List results = objectListing.getObjectSummaries();
        ArrayList<StorageObject> storageObjects = new ArrayList<StorageObject>();
        for (S3ObjectSummary so : results) {
            storageObjects.add(this.toStorageObject(so));
        }
        while (objectListing.isTruncated()) {
            objectListing = this.client.listNextBatchOfObjects(objectListing);
            for (S3ObjectSummary so : objectListing.getObjectSummaries()) {
                storageObjects.add(this.toStorageObject(so));
            }
        }
        return storageObjects;
    }

    @Override
    public boolean isAvailable(String id) {
        return this.client.doesObjectExist(this.bucketName, id);
    }

    public void setEndpoint(String endpoint) {
        this.client.setEndpoint(endpoint);
    }

    private StorageObject toStorageObject(S3ObjectSummary so) {
        URI uri = this.getUri(so.getKey());
        ObjectMetadata metadata = new ObjectMetadata();
        metadata.setLastModified(so.getLastModified());
        metadata.setETag(so.getETag());
        metadata.setContentLength(so.getSize());
        return new StorageObject(so.getKey(), uri, metadata, null);
    }

    private URI getUri(String key) {
        if (this.isIbmCloud) {
            return URI.create(this.client.getUrl(this.bucketName, key).toString());
        }
        String bucketLocation = this.client.getRegionName();
        return URI.create(bucketLocation + "/" + key);
    }

    @Override
    public String put(StorageObject storageObject) {
        com.amazonaws.services.s3.model.ObjectMetadata metadata = new com.amazonaws.services.s3.model.ObjectMetadata();
        metadata.setContentType(storageObject.getMetadata().getContentType());
        metadata.setContentLength(storageObject.getMetadata().getContentLength());
        metadata.setContentMD5(storageObject.getMetadata().getContentMD5());
        PutObjectResult putObjectResult = null;
        try (InputStream inputStream = storageObject.getPayload().openStream();){
            putObjectResult = this.client.putObject(new PutObjectRequest(this.bucketName, storageObject.getName(), inputStream, this.checkMetaData(metadata)));
        }
        catch (IOException e) {
            LOG.error("Error storing object " + storageObject.getName(), (Throwable)e);
        }
        return putObjectResult == null ? null : putObjectResult.getETag();
    }

    @Override
    public String put(String key, Payload value) {
        com.amazonaws.services.s3.model.ObjectMetadata metadata = new com.amazonaws.services.s3.model.ObjectMetadata();
        byte[] content = new byte[]{};
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            try (DigestInputStream dis = new DigestInputStream(value.openStream(), md);){
                content = IOUtils.toByteArray((InputStream)dis);
                metadata.setContentMD5(BinaryUtils.toBase64((byte[])md.digest()));
            }
        }
        catch (NoSuchAlgorithmException e) {
            LOG.error("Cannot calculate MD5 hash of because no MD5 algorithm was found", (Throwable)e);
        }
        catch (IOException e) {
            LOG.error("Error reading payload for key " + key, (Throwable)e);
        }
        Integer intLength = content.length;
        metadata.setContentLength(intLength.longValue());
        PutObjectResult putObjectResult = null;
        try (ByteArrayInputStream is = new ByteArrayInputStream(content);){
            putObjectResult = this.client.putObject(new PutObjectRequest(this.bucketName, key, (InputStream)is, this.checkMetaData(metadata)));
        }
        catch (IOException e) {
            LOG.error("Error storing object with key " + key, (Throwable)e);
        }
        return putObjectResult == null ? null : putObjectResult.getETag();
    }

    private com.amazonaws.services.s3.model.ObjectMetadata checkMetaData(com.amazonaws.services.s3.model.ObjectMetadata metadata) {
        if (metadata.getContentLength() == 0L) {
            throw new ObjectStorageClientException("The metadata ContentLength is mandatory");
        }
        return metadata;
    }

    @Override
    public Optional<StorageObject> getWithoutBody(String objectName) {
        try {
            return Optional.ofNullable(this.retrieveAsStorageObject(objectName, false, false));
        }
        catch (AmazonS3Exception | ContentValidationException e) {
            throw new ObjectStorageClientException(ERROR_MSG_RETRIEVE + objectName + " without body", e);
        }
    }

    @Override
    public Optional<StorageObject> get(String objectName) {
        try {
            return Optional.ofNullable(this.retrieveAsStorageObject(objectName, true, false));
        }
        catch (AmazonS3Exception | ContentValidationException e) {
            throw new ObjectStorageClientException(ERROR_MSG_RETRIEVE + objectName, e);
        }
    }

    @Override
    public Optional<StorageObject> get(String objectName, boolean verify) throws ContentValidationException {
        try {
            return Optional.ofNullable(this.retrieveAsStorageObject(objectName, true, verify));
        }
        catch (AmazonS3Exception ex) {
            throw new ObjectStorageClientException(ERROR_MSG_RETRIEVE + objectName, ex);
        }
    }

    @Override
    public byte[] getContent(String objectName) {
        byte[] result;
        block2: {
            result = new byte[]{};
            try {
                result = this.retrieveAsBytes(objectName);
            }
            catch (AmazonS3Exception ex) {
                if (this.is404Exception((Exception)((Object)ex))) break block2;
                throw new ObjectStorageClientException(ERROR_MSG_RETRIEVE + objectName, ex);
            }
        }
        return result;
    }

    private ObjectMetadata getObjectMetaData(String id) {
        com.amazonaws.services.s3.model.ObjectMetadata s3data = this.client.getObjectMetadata(this.bucketName, id);
        return new ObjectMetadata(s3data.getRawMetadata());
    }

    @Override
    public ObjectMetadata getMetaData(String id) {
        try {
            return this.getObjectMetaData(id);
        }
        catch (AmazonS3Exception ex) {
            if (!this.is404Exception((Exception)((Object)ex))) {
                throw new ObjectStorageClientException(ERROR_MSG_RETRIEVE + id, ex);
            }
            return null;
        }
    }

    @Override
    public void delete(String objectName) {
        DeleteObjectRequest deleteObjectRequest = new DeleteObjectRequest(this.bucketName, objectName);
        this.client.deleteObject(deleteObjectRequest);
    }

    @Override
    public void close() {
        LOG.info("Shutting down connections to {} ...", (Object)this.getName());
        ((AmazonS3Client)this.client).shutdown();
        IdleConnectionReaper.shutdown();
    }

    private StorageObject retrieveAsStorageObject(String id, boolean getContent, boolean verify) throws ContentValidationException {
        StorageObject result;
        block19: {
            result = null;
            if (getContent) {
                try (S3Object object = this.client.getObject(this.bucketName, id);){
                    result = this.getContent(object, verify);
                    break block19;
                }
                catch (IOException e) {
                    LOG.error("Error reading object content", (Throwable)e);
                }
                catch (RuntimeException e) {
                    if (!this.is404Exception(e)) {
                        LOG.error("Error reading object content ", (Throwable)e);
                    }
                    break block19;
                }
            }
            try {
                ObjectMetadata metadata = this.getObjectMetaData(id);
                result = new StorageObject(id, this.getUri(id), metadata, null);
            }
            catch (RuntimeException e) {
                if (this.is404Exception(e)) break block19;
                LOG.error("Error reading object content ", (Throwable)e);
            }
        }
        return result;
    }

    private StorageObject getContent(S3Object object, boolean verify) throws IOException, ContentValidationException {
        ByteArrayPayload content;
        ObjectMetadata objectMetadata = new ObjectMetadata(object.getObjectMetadata().getRawMetadata());
        try (S3ObjectInputStream contentStream = object.getObjectContent();){
            content = verify ? this.readAndVerifyContent(contentStream, BinaryUtils.fromHex((String)objectMetadata.getETag())) : new ByteArrayPayload(IOUtils.toByteArray((InputStream)contentStream));
            content.close();
        }
        return new StorageObject(object.getKey(), this.getUri(object.getKey()), objectMetadata, (Payload)content);
    }

    private boolean is404Exception(Exception e) {
        return e instanceof AmazonServiceException && ((AmazonServiceException)e).getStatusCode() == 404;
    }

    private ByteArrayPayload readAndVerifyContent(S3ObjectInputStream contentStream, byte[] serverSideHash) throws ContentValidationException, IOException {
        ByteArrayPayload result = null;
        try {
            MessageDigest md = MessageDigest.getInstance("MD5");
            try (DigestInputStream dis = new DigestInputStream((InputStream)contentStream, md);){
                result = new ByteArrayPayload(IOUtils.toByteArray((InputStream)dis));
                result.close();
                byte[] streamHash = md.digest();
                if (streamHash == null || serverSideHash == null || !Arrays.equals(streamHash, serverSideHash)) {
                    throw new ContentValidationException("Error comparing retrieved content hash with server hash (content = " + Arrays.toString(streamHash) + ", server = " + Arrays.toString(serverSideHash));
                }
            }
        }
        catch (NoSuchAlgorithmException e) {
            throw new ContentValidationException("Cannot verify MD5 hash of downloaded content because no MD5 algorithm was found", e);
        }
        return result;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private byte[] retrieveAsBytes(String id) {
        try (S3Object object = this.client.getObject(this.bucketName, id);){
            byte[] byArray = IOUtils.toByteArray((InputStream)object.getObjectContent());
            return byArray;
        }
        catch (IOException e) {
            LOG.error(ERROR_MSG_RETRIEVE + id, (Throwable)e);
            return new byte[0];
        }
    }

    public Bucket createBucket(String bucketName) {
        Bucket result = this.client.createBucket(bucketName);
        this.bucketName = bucketName;
        return result;
    }

    public List<Bucket> listBuckets() {
        return this.client.listBuckets();
    }

    public void deleteBucket(String bucket) {
        this.client.deleteBucket(bucket);
    }

    public void setS3ClientOptions(S3ClientOptions s3ClientOptions) {
        this.client.setS3ClientOptions(s3ClientOptions);
    }
}

