/*
 * Decompiled with CFR 0.152.
 */
package eu.europeana.metis.repository.rest;

import eu.europeana.metis.harvesting.HarvesterException;
import eu.europeana.metis.harvesting.http.CompressedFileExtension;
import eu.europeana.metis.harvesting.http.HttpHarvesterImpl;
import eu.europeana.metis.repository.dao.Record;
import eu.europeana.metis.repository.dao.RecordDao;
import eu.europeana.metis.repository.rest.InsertionResult;
import eu.europeana.metis.repository.rest.RecordView;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.tags.Tags;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.Objects;
import java.util.regex.Pattern;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.format.annotation.DateTimeFormat;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RequestPart;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.server.ResponseStatusException;

/*
 * Exception performing whole class analysis ignored.
 */
@RestController
@Tags(value={@Tag(name="RecordController", description="Controller providing access to record management functionality.")})
@Api(tags={"RecordController"})
public class RecordController {
    public static final String CONTROLLER_TAG_NAME = "RecordController";
    private static final Logger LOGGER = LoggerFactory.getLogger(RecordController.class);
    private static final Pattern UNSUPPORTED_CHARACTERS_PATTERN = Pattern.compile("[^a-zA-Z0-9_]");
    private static final String REPLACEMENT_CHARACTER = "_";
    private RecordDao recordDao;

    @Autowired
    void setRecordDao(RecordDao recordDao) {
        this.recordDao = recordDao;
    }

    @PostMapping(value={"/repository/records/{recordId}"}, consumes={"application/xml"}, produces={"application/json"})
    @ResponseStatus(value=HttpStatus.OK)
    @ResponseBody
    @ApiOperation(value="The given record is put into the database. If the record ID already exists, the record is overwritten. Note that record IDs are normalized to contain only the characters a-z, A-Z, 0-9 and `_`. But contrary to the batch upload method, they are NOT prefixed by the dataset ID.")
    @ApiResponses(value={@ApiResponse(code=404, message="Illegal dataset or record ID"), @ApiResponse(code=500, message="Error processing the record")})
    public InsertionResult saveRecord(@ApiParam(value="Record ID (new or existing)", required=true) @PathVariable(value="recordId") String recordId, @ApiParam(value="Dataset ID (new or existing)", required=true) @RequestParam(value="datasetId") String datasetId, @ApiParam(value="Date stamp (in ISO format)") @RequestParam(name="dateStamp", required=false) @DateTimeFormat(iso=DateTimeFormat.ISO.DATE_TIME) Instant dateStamp, @ApiParam(value="Whether the record is to be marked as deleted", required=true) @RequestParam(value="markAsDeleted") boolean markAsDeleted, @ApiParam(value="The actual (EDM/RDF) record", required=true) @RequestBody String edmRecord) {
        RecordController.verifyDatasetId((String)datasetId);
        InsertionResult result = new InsertionResult(datasetId, Objects.requireNonNullElseGet(dateStamp, Instant::now));
        this.saveRecord(recordId, edmRecord, result, markAsDeleted);
        return result;
    }

    @PostMapping(value={"/repository/records"}, consumes={"multipart/form-data"}, produces={"application/json"})
    @ResponseStatus(value=HttpStatus.OK)
    @ResponseBody
    @ApiOperation(value="The given records are put into the database as non-deleted records. The record IDs are computed to be the file name (without the extension) prefixed by the dataset ID. If a record ID already exists, the record is overwritten. Note that record IDs are normalized to contain only the characters a-z, A-Z, 0-9 and `_`.")
    @ApiResponses(value={@ApiResponse(code=404, message="Illegal dataset or record ID"), @ApiResponse(code=500, message="Error processing the file archive")})
    public InsertionResult saveRecords(@ApiParam(value="Dataset ID (new or existing)", required=true) @RequestParam(value="datasetId") String datasetId, @ApiParam(value="Date stamp (in ISO format)") @RequestParam(name="dateStamp", required=false) @DateTimeFormat(iso=DateTimeFormat.ISO.DATE_TIME) Instant dateStamp, @ApiParam(value="The (EDM/RDF) records", required=true) @RequestPart MultipartFile recordsZipFile) {
        RecordController.verifyDatasetId((String)datasetId);
        InsertionResult result = new InsertionResult(datasetId, Objects.requireNonNullElseGet(dateStamp, Instant::now));
        try (InputStream inputStream = recordsZipFile.getInputStream();){
            new HttpHarvesterImpl().harvestRecords(inputStream, CompressedFileExtension.ZIP, entry -> {
                byte[] content = entry.getEntryContent().readAllBytes();
                String recordId = datasetId + "_" + FilenameUtils.getBaseName((String)entry.getEntryName());
                this.saveRecord(recordId, new String(content, StandardCharsets.UTF_8), result, false);
            });
        }
        catch (HarvesterException | IOException | RuntimeException e) {
            LOGGER.warn("A problem occurred while processing the file archive.", e);
            throw new ResponseStatusException(HttpStatus.INTERNAL_SERVER_ERROR, e.getMessage(), e);
        }
        return result;
    }

    @PutMapping(value={"/repository/records/{recordId}/header"}, produces={"application/json"})
    @ResponseStatus(value=HttpStatus.OK)
    @ResponseBody
    @ApiOperation(value="The header fields of the given record is updated.")
    @ApiResponses(value={@ApiResponse(code=404, message="Illegal dataset or unknown record ID"), @ApiResponse(code=500, message="Error processing the record")})
    public InsertionResult updateRecordHeader(@ApiParam(value="Record ID (existing)", required=true) @PathVariable(value="recordId") String recordId, @ApiParam(value="Dataset ID (new or existing)", required=true) @RequestParam(value="datasetId") String datasetId, @ApiParam(value="Date stamp (in ISO format)") @RequestParam(name="dateStamp", required=false) @DateTimeFormat(iso=DateTimeFormat.ISO.DATE_TIME) Instant dateStamp, @ApiParam(value="Whether the record is to be marked as deleted", required=true) @RequestParam(value="markAsDeleted") boolean markAsDeleted) {
        Record record = this.recordDao.getRecord(recordId);
        if (record == null) {
            throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No record found for this identifier.");
        }
        return this.saveRecord(recordId, datasetId, dateStamp, markAsDeleted, record.getEdmRecord());
    }

    private void saveRecord(String providedRecordId, String edmRecord, InsertionResult result, boolean markedAsDeleted) {
        String recordId = RecordController.normalizeRecordId((String)providedRecordId);
        try {
            Record recordToSave = new Record(recordId, result.getDatasetId(), result.getDateStamp(), markedAsDeleted, edmRecord);
            if (this.recordDao.createRecord(recordToSave)) {
                result.addInsertedRecord(recordId);
            } else {
                result.addUpdatedRecord(recordId);
            }
        }
        catch (RuntimeException e) {
            LOGGER.warn("A problem occurred while saving a record.", (Throwable)e);
            throw new ResponseStatusException(HttpStatus.INTERNAL_SERVER_ERROR, e.getMessage(), (Throwable)e);
        }
    }

    @GetMapping(value={"/repository/records/{recordId}"}, produces={"application/xml"})
    @ResponseStatus(value=HttpStatus.OK)
    @ResponseBody
    @ApiOperation(value="The record is retrieved from the database.")
    @ApiResponses(value={@ApiResponse(code=404, message="Record ID is invalid or unknown"), @ApiResponse(code=500, message="Error processing the request")})
    public RecordView getRecord(@ApiParam(value="Record ID", required=true) @PathVariable(value="recordId") String recordId) {
        Record record = this.recordDao.getRecord(recordId);
        if (record == null) {
            throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No record found for this identifier.");
        }
        return new RecordView(record.getRecordId(), record.getDatasetId(), record.getDateStamp(), record.isDeleted(), record.getEdmRecord());
    }

    @DeleteMapping(value={"/repository/records/{recordId}"})
    @ResponseStatus(value=HttpStatus.OK)
    @ResponseBody
    @ApiOperation(value="The record is deleted from the database. Note: this is not the same as marking a record as deleted.")
    @ApiResponses(value={@ApiResponse(code=404, message="Record ID is invalid or unknown"), @ApiResponse(code=500, message="Error processing the request")})
    public void deleteRecord(@ApiParam(value="Record ID", required=true) @PathVariable(value="recordId") String recordId) {
        if (!this.recordDao.deleteRecord(recordId)) {
            throw new ResponseStatusException(HttpStatus.NOT_FOUND, "No record found for this identifier.");
        }
    }

    private static void verifyDatasetId(String datasetId) {
        if (UNSUPPORTED_CHARACTERS_PATTERN.matcher(datasetId).matches()) {
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, "Invalid dataset ID.");
        }
    }

    private static String normalizeRecordId(String suggestedRecordId) {
        if (StringUtils.isEmpty((CharSequence)suggestedRecordId)) {
            throw new ResponseStatusException(HttpStatus.BAD_REQUEST, "Invalid record ID.");
        }
        return UNSUPPORTED_CHARACTERS_PATTERN.matcher(suggestedRecordId).replaceAll("_");
    }
}

