/*
 * Decompiled with CFR 0.152.
 */
package eu.europeana.metis.mediaprocessing.extraction;

import eu.europeana.metis.mediaprocessing.exception.CommandExecutionException;
import eu.europeana.metis.mediaprocessing.exception.MediaExtractionException;
import eu.europeana.metis.mediaprocessing.exception.MediaProcessorException;
import eu.europeana.metis.mediaprocessing.extraction.CommandExecutor;
import eu.europeana.metis.mediaprocessing.extraction.ImageMetadata;
import eu.europeana.metis.mediaprocessing.extraction.ResourceType;
import eu.europeana.metis.mediaprocessing.model.Thumbnail;
import eu.europeana.metis.mediaprocessing.model.ThumbnailImpl;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ThumbnailGenerator {
    private static final Logger LOGGER = LoggerFactory.getLogger(ThumbnailGenerator.class);
    private static final String PDF_MIME_TYPE = "application/pdf";
    private static final String PNG_MIME_TYPE = "image/png";
    private static final int MEDIUM_THUMBNAIL_SIZE = 200;
    private static final int LARGE_THUMBNAIL_SIZE = 400;
    private static final Map<Integer, String> THUMBNAIL_SIZES_AND_SUFFIXES = new HashMap<Integer, String>();
    private static final String COMMAND_RESULT_FORMAT = "%w\n%h\n%[colorspace]\n";
    private static final int COMMAND_RESULT_WIDTH_LINE = 0;
    private static final int COMMAND_RESULT_HEIGHT_LINE = 1;
    private static final int COMMAND_RESULT_COLORSPACE_LINE = 2;
    private static final int COMMAND_RESULT_COLORS_LINE = 3;
    private static final int COMMAND_RESULT_MAX_COLORS = 6;
    private static String globalMagickCommand;
    private static Path globalColormapFile;
    private final String magickCmd;
    private final String colormapFile;
    private final CommandExecutor commandExecutor;

    ThumbnailGenerator(CommandExecutor commandExecutor) throws MediaProcessorException {
        this(commandExecutor, ThumbnailGenerator.getGlobalImageMagickCommand(commandExecutor), ThumbnailGenerator.initColorMap().toString());
    }

    ThumbnailGenerator(CommandExecutor commandExecutor, String magickCommand, String colorMapFile) {
        this.commandExecutor = commandExecutor;
        this.magickCmd = magickCommand;
        this.colormapFile = colorMapFile;
    }

    private static synchronized Path initColorMap() throws MediaProcessorException {
        Path colormapTempFile;
        if (globalColormapFile != null) {
            return globalColormapFile;
        }
        try (InputStream colorMapInputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream("colormap.png");){
            colormapTempFile = Files.createTempFile("colormap", ".png", new FileAttribute[0]);
            Files.copy(colorMapInputStream, colormapTempFile, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            LOGGER.warn("Could not load color map file: {}.", (Object)"colormap.png", (Object)e);
            throw new MediaProcessorException("Could not load color map file.", e);
        }
        colormapTempFile.toFile().deleteOnExit();
        globalColormapFile = colormapTempFile;
        return globalColormapFile;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static synchronized String getGlobalImageMagickCommand(CommandExecutor commandExecutor) throws MediaProcessorException {
        Class<ThumbnailGenerator> clazz = ThumbnailGenerator.class;
        synchronized (ThumbnailGenerator.class) {
            if (globalMagickCommand == null) {
                globalMagickCommand = ThumbnailGenerator.discoverImageMagickCommand(commandExecutor);
            }
            // ** MonitorExit[var1_1] (shouldn't be in output)
            return globalMagickCommand;
        }
    }

    static String discoverImageMagickCommand(CommandExecutor commandExecutor) throws MediaProcessorException {
        List<Object> paths;
        try {
            List<String> lines = commandExecutor.execute(Arrays.asList("magick", "-version"), true);
            if (String.join((CharSequence)"", lines).startsWith("Version: ImageMagick 7")) {
                String result = "magick";
                LOGGER.info("Found ImageMagic 7. Command: {}", (Object)"magick");
                return "magick";
            }
        }
        catch (CommandExecutionException e) {
            LOGGER.info("Could not find ImageMagick 7 because of: {}.", (Object)e.getMessage());
            LOGGER.debug("Could not find ImageMagick 7 due to following problem.", (Throwable)e);
        }
        boolean isWindows = System.getProperty("os.name").toLowerCase(Locale.ENGLISH).contains("win");
        try {
            paths = commandExecutor.execute(Arrays.asList(isWindows ? "where" : "which", "convert"), true);
        }
        catch (CommandExecutionException e) {
            LOGGER.warn("Could not find ImageMagick 6 due to following problem.", (Throwable)e);
            paths = Collections.emptyList();
        }
        for (String path : paths) {
            try {
                List<String> lines = commandExecutor.execute(Arrays.asList(path, "-version"), true);
                if (!String.join((CharSequence)"", lines).startsWith("Version: ImageMagick 6")) continue;
                LOGGER.info("Found ImageMagic 6. Command: {}", (Object)path);
                return path;
            }
            catch (CommandExecutionException e) {
                LOGGER.info("Could not find ImageMagick 6 at path {} because of: {}.", (Object)path, (Object)e.getMessage());
                LOGGER.debug("Could not find ImageMagick 6 at path {} due to following problem.", (Object)path, (Object)e);
            }
        }
        LOGGER.error("Could not find ImageMagick 6 or 7. See previous log statements for details.");
        throw new MediaProcessorException("Could not find ImageMagick 6 or 7.");
    }

    Pair<ImageMetadata, List<Thumbnail>> generateThumbnails(String url, String detectedMimeType, File content) throws MediaExtractionException {
        ImageMetadata image;
        if (content == null) {
            throw new MediaExtractionException("File content is null");
        }
        List<ThumbnailWithSize> thumbnails = this.prepareThumbnailFiles(url);
        try {
            image = this.generateThumbnailsInternal(thumbnails, detectedMimeType, content);
        }
        catch (RuntimeException e) {
            ThumbnailGenerator.closeAllThumbnailsSilently(thumbnails);
            throw new MediaExtractionException("Unexpected error during processing", e);
        }
        catch (MediaExtractionException e) {
            ThumbnailGenerator.closeAllThumbnailsSilently(thumbnails);
            throw e;
        }
        List resultThumbnails = thumbnails.stream().map(ThumbnailWithSize::getThumbnail).collect(Collectors.toList());
        return new ImmutablePair((Object)image, resultThumbnails);
    }

    private static void closeAllThumbnailsSilently(List<ThumbnailWithSize> thumbnails) {
        for (ThumbnailWithSize thumbnail : thumbnails) {
            try {
                thumbnail.getThumbnail().close();
            }
            catch (IOException e) {
                LOGGER.warn("Could not close thumbnail: {}", (Object)thumbnail.getThumbnail().getResourceUrl(), (Object)e);
            }
        }
    }

    List<String> createThumbnailGenerationCommand(List<ThumbnailWithSize> thumbnails, String detectedMimeType, File content) {
        String fileTypePrefix = PDF_MIME_TYPE.equals(detectedMimeType) || PNG_MIME_TYPE.equals(detectedMimeType) ? "png:" : "jpeg:";
        ArrayList<String> command = new ArrayList<String>(Arrays.asList(this.magickCmd, content.getPath() + "[0]", "-format", COMMAND_RESULT_FORMAT, "-write", "info:"));
        if (PDF_MIME_TYPE.equals(detectedMimeType)) {
            command.addAll(Arrays.asList("-background", "white", "-alpha", "remove"));
        }
        int thumbnailCounter = thumbnails.size();
        for (int i = 0; i < thumbnailCounter; ++i) {
            if (i != thumbnailCounter - 1) {
                command.add("(");
                command.add("+clone");
            }
            ThumbnailWithSize thumbnail = thumbnails.get(i);
            command.addAll(Arrays.asList("-thumbnail", thumbnail.getImageSize() + "x", "-write", fileTypePrefix + thumbnail.getThumbnail().getContentPath().toString()));
            if (i == thumbnailCounter - 1) continue;
            command.add("+delete");
            command.add(")");
        }
        command.addAll(Arrays.asList("-colorspace", "sRGB", "-dither", "Riemersma", "-remap", this.colormapFile, "-format", "\n%c", "histogram:info:"));
        return command;
    }

    private ImageMetadata generateThumbnailsInternal(List<ThumbnailWithSize> thumbnails, String detectedMimeType, File content) throws MediaExtractionException {
        List<String> response;
        try {
            response = this.commandExecutor.execute(this.createThumbnailGenerationCommand(thumbnails, detectedMimeType, content), false);
        }
        catch (CommandExecutionException e) {
            throw new MediaExtractionException("Could not analyze content and generate thumbnails.", e);
        }
        ImageMetadata result = this.parseCommandResponse(response);
        for (ThumbnailWithSize thumbnail : thumbnails) {
            try {
                Path thumb = thumbnail.getThumbnail().getContentPath();
                if (this.getFileSize(thumb) == 0L) {
                    throw new MediaExtractionException("Thumbnail file empty: " + thumb);
                }
                boolean isImage = ResourceType.getResourceType(detectedMimeType) == ResourceType.IMAGE;
                if (!isImage || result.getWidth() >= thumbnail.getImageSize()) continue;
                this.copyFile(content, thumb);
            }
            catch (IOException e) {
                throw new MediaExtractionException("Could not access thumbnail file", e);
            }
        }
        return result;
    }

    long getFileSize(Path file) throws IOException {
        return Files.size(file);
    }

    void copyFile(File source, Path destination) throws IOException {
        Files.copy(source.toPath(), destination, StandardCopyOption.REPLACE_EXISTING);
    }

    List<ThumbnailWithSize> prepareThumbnailFiles(String url) throws MediaExtractionException {
        String md5 = ThumbnailGenerator.md5Hex(url);
        ArrayList<ThumbnailWithSize> result = new ArrayList<ThumbnailWithSize>(THUMBNAIL_SIZES_AND_SUFFIXES.size());
        try {
            for (Map.Entry<Integer, String> entry : THUMBNAIL_SIZES_AND_SUFFIXES.entrySet()) {
                ThumbnailImpl thumbnail = new ThumbnailImpl(url, md5 + entry.getValue());
                result.add(new ThumbnailWithSize(thumbnail, entry.getKey()));
            }
        }
        catch (IOException e) {
            throw new MediaExtractionException("Could not create thumbnail files.", e);
        }
        return result;
    }

    private static String md5Hex(String s) throws MediaExtractionException {
        try {
            byte[] bytes = s.getBytes(StandardCharsets.UTF_8.name());
            byte[] md5bytes = MessageDigest.getInstance("MD5").digest(bytes);
            return String.format("%032x", new BigInteger(1, md5bytes));
        }
        catch (UnsupportedEncodingException | NoSuchAlgorithmException e) {
            throw new MediaExtractionException("Could not compute md5 hash", e);
        }
    }

    ImageMetadata parseCommandResponse(List<String> response) throws MediaExtractionException {
        try {
            int width = Integer.parseInt(response.get(0));
            int height = Integer.parseInt(response.get(1));
            String colorSpace = response.get(2);
            List<String> dominantColors = ThumbnailGenerator.extractDominantColors(response, 3);
            return new ImageMetadata(width, height, colorSpace, dominantColors);
        }
        catch (RuntimeException e) {
            LOGGER.info("Could not parse ImageMagick response:\n" + StringUtils.join(response, (String)"\n"), (Throwable)e);
            throw new MediaExtractionException("File seems to be corrupted", e);
        }
    }

    private static List<String> extractDominantColors(List<String> results, int skipLines) {
        Pattern pattern = Pattern.compile("#([0-9A-F]{6})");
        return results.stream().skip(skipLines).sorted(Collections.reverseOrder()).limit(6L).map(pattern::matcher).peek(m -> {
            if (!m.find()) {
                throw new IllegalStateException("Invalid color line found.");
            }
        }).map(matcher -> matcher.group(1)).collect(Collectors.toList());
    }

    static {
        THUMBNAIL_SIZES_AND_SUFFIXES.put(200, "-MEDIUM");
        THUMBNAIL_SIZES_AND_SUFFIXES.put(400, "-LARGE");
    }

    static class ThumbnailWithSize {
        private final ThumbnailImpl thumbnail;
        private final int imageSize;

        ThumbnailWithSize(ThumbnailImpl thumbnail, int imageSize) {
            this.thumbnail = thumbnail;
            this.imageSize = imageSize;
        }

        ThumbnailImpl getThumbnail() {
            return this.thumbnail;
        }

        int getImageSize() {
            return this.imageSize;
        }
    }
}

