/*
 * Decompiled with CFR 0.152.
 */
package eu.europeana.metis.core.service;

import eu.europeana.metis.authentication.user.AccountRole;
import eu.europeana.metis.authentication.user.MetisUser;
import eu.europeana.metis.core.dao.DatasetDao;
import eu.europeana.metis.core.dao.WorkflowDao;
import eu.europeana.metis.core.dao.WorkflowExecutionDao;
import eu.europeana.metis.core.dataset.Dataset;
import eu.europeana.metis.core.dataset.DatasetExecutionInformation;
import eu.europeana.metis.core.exceptions.NoDatasetFoundException;
import eu.europeana.metis.core.exceptions.NoWorkflowExecutionFoundException;
import eu.europeana.metis.core.exceptions.NoWorkflowFoundException;
import eu.europeana.metis.core.exceptions.PluginExecutionNotAllowed;
import eu.europeana.metis.core.exceptions.WorkflowAlreadyExistsException;
import eu.europeana.metis.core.exceptions.WorkflowExecutionAlreadyExistsException;
import eu.europeana.metis.core.execution.ExecutionRules;
import eu.europeana.metis.core.execution.WorkflowExecutorManager;
import eu.europeana.metis.core.rest.VersionEvolution;
import eu.europeana.metis.core.rest.execution.overview.ExecutionAndDatasetView;
import eu.europeana.metis.core.service.Authorizer;
import eu.europeana.metis.core.service.OrchestratorHelper;
import eu.europeana.metis.core.workflow.OrderField;
import eu.europeana.metis.core.workflow.Workflow;
import eu.europeana.metis.core.workflow.WorkflowExecution;
import eu.europeana.metis.core.workflow.WorkflowStatus;
import eu.europeana.metis.core.workflow.plugins.AbstractExecutablePlugin;
import eu.europeana.metis.core.workflow.plugins.AbstractExecutablePluginMetadata;
import eu.europeana.metis.core.workflow.plugins.AbstractMetisPlugin;
import eu.europeana.metis.core.workflow.plugins.DataStatus;
import eu.europeana.metis.core.workflow.plugins.ExecutablePluginType;
import eu.europeana.metis.core.workflow.plugins.PluginStatus;
import eu.europeana.metis.core.workflow.plugins.PluginType;
import eu.europeana.metis.exception.BadContentException;
import eu.europeana.metis.exception.GenericMetisException;
import eu.europeana.metis.utils.DateUtils;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class OrchestratorService {
    private static final Logger LOGGER = LoggerFactory.getLogger(OrchestratorService.class);
    private static final String EXECUTION_FOR_DATASETID_SUBMITION_LOCK = "EXECUTION_FOR_DATASETID_SUBMITION_LOCK_%s";
    private final WorkflowExecutionDao workflowExecutionDao;
    private final WorkflowDao workflowDao;
    private final DatasetDao datasetDao;
    private final WorkflowExecutorManager workflowExecutorManager;
    private final RedissonClient redissonClient;
    private final Authorizer authorizer;
    private final OrchestratorHelper orchestratorHelper;
    private int solrCommitPeriodInMins;

    @Autowired
    public OrchestratorService(OrchestratorHelper orchestratorHelper, WorkflowDao workflowDao, WorkflowExecutionDao workflowExecutionDao, DatasetDao datasetDao, WorkflowExecutorManager workflowExecutorManager, RedissonClient redissonClient, Authorizer authorizer) {
        this.orchestratorHelper = orchestratorHelper;
        this.workflowDao = workflowDao;
        this.workflowExecutionDao = workflowExecutionDao;
        this.datasetDao = datasetDao;
        this.workflowExecutorManager = workflowExecutorManager;
        this.redissonClient = redissonClient;
        this.authorizer = authorizer;
    }

    public void createWorkflow(MetisUser metisUser, String datasetId, Workflow workflow) throws GenericMetisException {
        this.authorizer.authorizeWriteExistingDatasetById(metisUser, datasetId);
        try {
            this.orchestratorHelper.validateAndTrimHarvestParameters(workflow);
        }
        catch (MalformedURLException | URISyntaxException e) {
            throw new BadContentException("Harvesting parameters are invalid", (Throwable)e);
        }
        if (this.datasetDao.getDatasetByDatasetId(datasetId) == null) {
            throw new NoDatasetFoundException(String.format("Dataset with datasetId: %s does NOT exist", datasetId));
        }
        workflow.setDatasetId(datasetId);
        this.checkRestrictionsOnWorkflowCreate(datasetId, workflow);
        workflow.getMetisPluginsMetadata().forEach(abstractMetisPluginMetadata -> abstractMetisPluginMetadata.setEnabled(true));
        this.workflowDao.create(workflow);
    }

    public void updateWorkflow(MetisUser metisUser, String datasetId, Workflow workflow) throws GenericMetisException {
        this.authorizer.authorizeWriteExistingDatasetById(metisUser, datasetId);
        try {
            this.orchestratorHelper.validateAndTrimHarvestParameters(workflow);
        }
        catch (MalformedURLException | URISyntaxException e) {
            throw new BadContentException("Harvesting parameters are invalid", (Throwable)e);
        }
        if (this.datasetDao.getDatasetByDatasetId(datasetId) == null) {
            throw new NoDatasetFoundException(String.format("Dataset with datasetId: %s does NOT exist", datasetId));
        }
        workflow.setDatasetId(datasetId);
        Workflow storedWorkflow = this.checkRestrictionsOnWorkflowUpdate(datasetId, workflow);
        workflow.setId(storedWorkflow.getId());
        this.orchestratorHelper.overwriteNewPluginMetadataOnWorkflowAndDisableOtherPluginMetadata(workflow, storedWorkflow);
        this.workflowDao.update(workflow);
    }

    public void deleteWorkflow(MetisUser metisUser, String datasetId) throws GenericMetisException {
        this.authorizer.authorizeWriteExistingDatasetById(metisUser, datasetId);
        this.workflowDao.deleteWorkflow(datasetId);
    }

    public Workflow getWorkflow(MetisUser metisUser, String datasetId) throws GenericMetisException {
        this.authorizer.authorizeReadExistingDatasetById(metisUser, datasetId);
        return this.getWorkflow(datasetId);
    }

    private Workflow getWorkflow(String datasetId) {
        return this.workflowDao.getWorkflow(datasetId);
    }

    public WorkflowExecution getWorkflowExecutionByExecutionId(MetisUser metisUser, String executionId) throws GenericMetisException {
        WorkflowExecution result = this.workflowExecutionDao.getById(executionId);
        if (result != null) {
            this.authorizer.authorizeReadExistingDatasetById(metisUser, result.getDatasetId());
        }
        return result;
    }

    public WorkflowExecution addWorkflowInQueueOfWorkflowExecutionsWithoutAuthorization(String datasetId, ExecutablePluginType enforcedPluginType, int priority) throws GenericMetisException {
        Dataset dataset = this.datasetDao.getDatasetByDatasetId(datasetId);
        if (dataset == null) {
            throw new NoDatasetFoundException(String.format("No dataset found with datasetId: %s, in METIS", datasetId));
        }
        return this.addWorkflowInQueueOfWorkflowExecutions(dataset, enforcedPluginType, priority);
    }

    public WorkflowExecution addWorkflowInQueueOfWorkflowExecutions(MetisUser metisUser, String datasetId, ExecutablePluginType enforcedPluginType, int priority) throws GenericMetisException {
        Dataset dataset = this.authorizer.authorizeWriteExistingDatasetById(metisUser, datasetId);
        return this.addWorkflowInQueueOfWorkflowExecutions(dataset, enforcedPluginType, priority);
    }

    private WorkflowExecution addWorkflowInQueueOfWorkflowExecutions(Dataset dataset, ExecutablePluginType enforcedPluginType, int priority) throws GenericMetisException {
        Workflow workflow = this.checkWorkflowExistence(dataset.getDatasetId());
        List<AbstractExecutablePlugin> metisPlugins = this.createMetisPluginsList(dataset, workflow, enforcedPluginType);
        if (metisPlugins.isEmpty()) {
            throw new BadContentException("Workflow has either no plugins or are all disabled");
        }
        this.datasetDao.checkAndCreateDatasetInEcloud(dataset);
        WorkflowExecution workflowExecution = new WorkflowExecution(dataset, metisPlugins, priority);
        workflowExecution.setWorkflowStatus(WorkflowStatus.INQUEUE);
        RLock executionDatasetIdLock = this.redissonClient.getFairLock(String.format(EXECUTION_FOR_DATASETID_SUBMITION_LOCK, dataset.getDatasetId()));
        executionDatasetIdLock.lock();
        String storedWorkflowExecutionId = this.workflowExecutionDao.existsAndNotCompleted(dataset.getDatasetId());
        if (storedWorkflowExecutionId != null) {
            executionDatasetIdLock.unlock();
            throw new WorkflowExecutionAlreadyExistsException(String.format("Workflow execution already exists with id %s and is not completed", storedWorkflowExecutionId));
        }
        workflowExecution.setCreatedDate(new Date());
        String objectId = this.workflowExecutionDao.create(workflowExecution);
        executionDatasetIdLock.unlock();
        this.workflowExecutorManager.addWorkflowExecutionToQueue(objectId, priority);
        LOGGER.info("WorkflowExecution with id: {}, added to execution queue", (Object)objectId);
        return this.workflowExecutionDao.getById(objectId);
    }

    private List<AbstractExecutablePlugin> createMetisPluginsList(Dataset dataset, Workflow workflow, ExecutablePluginType enforcedPluginType) throws PluginExecutionNotAllowed {
        ArrayList<AbstractExecutablePlugin> metisPlugins = new ArrayList<AbstractExecutablePlugin>();
        boolean firstPluginDefined = this.orchestratorHelper.addHarvestingPlugin(dataset, workflow, metisPlugins);
        this.orchestratorHelper.addNonHarvestPlugins(dataset, workflow, enforcedPluginType, metisPlugins, firstPluginDefined);
        return metisPlugins;
    }

    public void cancelWorkflowExecution(MetisUser metisUser, String executionId) throws GenericMetisException {
        WorkflowExecution workflowExecution = this.workflowExecutionDao.getById(executionId);
        if (workflowExecution != null) {
            this.authorizer.authorizeWriteExistingDatasetById(metisUser, workflowExecution.getDatasetId());
        }
        if (workflowExecution == null || workflowExecution.getWorkflowStatus() != WorkflowStatus.RUNNING && workflowExecution.getWorkflowStatus() != WorkflowStatus.INQUEUE) {
            throw new NoWorkflowExecutionFoundException(String.format("Running workflowExecution with executionId: %s, does not exist or not active", executionId));
        }
        this.workflowExecutionDao.setCancellingState(workflowExecution, metisUser);
        LOGGER.info("Cancelling user workflow execution with id: {}", (Object)workflowExecution.getId());
    }

    private void checkRestrictionsOnWorkflowCreate(String datasetId, Workflow workflow) throws WorkflowAlreadyExistsException, PluginExecutionNotAllowed {
        if (StringUtils.isNotEmpty((CharSequence)this.workflowExists(workflow))) {
            throw new WorkflowAlreadyExistsException(String.format("Workflow with datasetId: %s, already exists", workflow.getDatasetId()));
        }
        this.workflowOrderValidator(datasetId, workflow);
    }

    private Workflow checkRestrictionsOnWorkflowUpdate(String datasetId, Workflow workflow) throws NoWorkflowFoundException, PluginExecutionNotAllowed {
        Workflow storedWorkflow = this.getWorkflow(workflow.getDatasetId());
        if (storedWorkflow == null) {
            throw new NoWorkflowFoundException(String.format("Workflow with datasetId: %s, not found", workflow.getDatasetId()));
        }
        this.workflowOrderValidator(datasetId, workflow);
        return storedWorkflow;
    }

    private void workflowOrderValidator(String datasetId, Workflow workflow) throws PluginExecutionNotAllowed {
        if (this.orchestratorHelper.listContainsDuplicates(workflow.getMetisPluginsMetadata())) {
            throw new PluginExecutionNotAllowed("Plugin Execution Not Allowed");
        }
        this.orchestratorHelper.getLatestFinishedPluginByDatasetIdIfPluginTypeAllowedForExecution(datasetId, ((AbstractExecutablePluginMetadata)workflow.getMetisPluginsMetadata().get(0)).getExecutablePluginType(), null);
        boolean valid = workflow.getMetisPluginsMetadata().stream().skip(1L).map(AbstractExecutablePluginMetadata::getExecutablePluginType).filter(pluginType -> !ExecutionRules.getHarvestPluginGroup().contains(pluginType)).allMatch(pluginType -> this.orchestratorHelper.checkWorkflowForPluginType(workflow, (ExecutablePluginType)pluginType));
        if (!valid) {
            throw new PluginExecutionNotAllowed("Plugin Execution Not Allowed");
        }
    }

    private String workflowExists(Workflow workflow) {
        return this.workflowDao.exists(workflow);
    }

    public int getWorkflowExecutionsPerRequest() {
        return this.workflowExecutionDao.getWorkflowExecutionsPerRequest();
    }

    public int getWorkflowsPerRequest() {
        return this.workflowDao.getWorkflowsPerRequest();
    }

    public AbstractExecutablePlugin getLatestFinishedPluginByDatasetIdIfPluginTypeAllowedForExecution(MetisUser metisUser, String datasetId, ExecutablePluginType pluginType, ExecutablePluginType enforcedPluginType) throws GenericMetisException {
        this.authorizer.authorizeReadExistingDatasetById(metisUser, datasetId);
        return this.orchestratorHelper.getLatestFinishedPluginByDatasetIdIfPluginTypeAllowedForExecution(datasetId, pluginType, enforcedPluginType);
    }

    public List<WorkflowExecution> getAllWorkflowExecutions(MetisUser metisUser, String datasetId, Set<WorkflowStatus> workflowStatuses, OrderField orderField, boolean ascending, int nextPage) throws GenericMetisException {
        if (datasetId == null) {
            this.authorizer.authorizeReadAllDatasets(metisUser);
        } else {
            this.authorizer.authorizeReadExistingDatasetById(metisUser, datasetId);
        }
        Set<String> datasetIds = datasetId == null ? this.getDatasetIdsToFilterOn(metisUser) : Collections.singleton(datasetId);
        return this.workflowExecutionDao.getAllWorkflowExecutions(datasetIds, workflowStatuses, orderField, ascending, nextPage);
    }

    public List<ExecutionAndDatasetView> getWorkflowExecutionsOverview(MetisUser metisUser, Set<PluginStatus> pluginStatuses, Set<PluginType> pluginTypes, Date fromDate, Date toDate, int nextPage, int pageCount) throws GenericMetisException {
        this.authorizer.authorizeReadAllDatasets(metisUser);
        Set<String> datasetIds = this.getDatasetIdsToFilterOn(metisUser);
        return this.workflowExecutionDao.getWorkflowExecutionsOverview(datasetIds, pluginStatuses, pluginTypes, fromDate, toDate, nextPage, pageCount).stream().map(result -> new ExecutionAndDatasetView(result.getExecution(), result.getDataset())).collect(Collectors.toList());
    }

    private Set<String> getDatasetIdsToFilterOn(MetisUser metisUser) {
        Set datasetIds = metisUser.getAccountRole() == AccountRole.METIS_ADMIN ? null : this.datasetDao.getAllDatasetsByOrganizationId(metisUser.getOrganizationId()).stream().map(Dataset::getDatasetId).collect(Collectors.toSet());
        return datasetIds;
    }

    public DatasetExecutionInformation getDatasetExecutionInformation(MetisUser metisUser, String datasetId) throws GenericMetisException {
        this.authorizer.authorizeReadExistingDatasetById(metisUser, datasetId);
        AbstractExecutablePlugin lastHarvestPlugin = this.workflowExecutionDao.getLastFinishedWorkflowExecutionPluginByDatasetIdAndPluginType(datasetId, EnumSet.of(ExecutablePluginType.HTTP_HARVEST, ExecutablePluginType.OAIPMH_HARVEST), false);
        AbstractExecutablePlugin firstPublishPlugin = this.workflowExecutionDao.getFirstFinishedWorkflowExecutionPluginByDatasetIdAndPluginType(datasetId, EnumSet.of(ExecutablePluginType.PUBLISH));
        AbstractExecutablePlugin lastPreviewPlugin = this.workflowExecutionDao.getLastFinishedWorkflowExecutionPluginByDatasetIdAndPluginType(datasetId, EnumSet.of(ExecutablePluginType.PREVIEW), false);
        AbstractExecutablePlugin lastPublishPlugin = this.workflowExecutionDao.getLastFinishedWorkflowExecutionPluginByDatasetIdAndPluginType(datasetId, EnumSet.of(ExecutablePluginType.PUBLISH), false);
        WorkflowExecution runningOrInQueueExecution = this.workflowExecutionDao.getRunningOrInQueueExecution(datasetId);
        boolean isPreviewCleaningOrRunning = this.isPluginInWorkflowCleaningOrRunning(runningOrInQueueExecution, PluginType.PREVIEW);
        boolean isPublishCleaningOrRunning = this.isPluginInWorkflowCleaningOrRunning(runningOrInQueueExecution, PluginType.PUBLISH);
        DatasetExecutionInformation datasetExecutionInformation = new DatasetExecutionInformation();
        if (lastHarvestPlugin != null) {
            datasetExecutionInformation.setLastHarvestedDate(lastHarvestPlugin.getFinishedDate());
            datasetExecutionInformation.setLastHarvestedRecords(lastHarvestPlugin.getExecutionProgress().getProcessedRecords() - lastHarvestPlugin.getExecutionProgress().getErrors());
        }
        datasetExecutionInformation.setFirstPublishedDate(firstPublishPlugin == null ? null : firstPublishPlugin.getFinishedDate());
        Date now = new Date();
        if (lastPreviewPlugin != null) {
            datasetExecutionInformation.setLastPreviewDate(lastPreviewPlugin.getFinishedDate());
            datasetExecutionInformation.setLastPreviewRecords(lastPreviewPlugin.getExecutionProgress().getProcessedRecords() - lastPreviewPlugin.getExecutionProgress().getErrors());
            datasetExecutionInformation.setLastPreviewRecordsReadyForViewing(!isPreviewCleaningOrRunning && this.isPreviewOrPublishReadyForViewing(lastPreviewPlugin, now));
        }
        if (lastPublishPlugin != null) {
            datasetExecutionInformation.setLastPublishedDate(lastPublishPlugin.getFinishedDate());
            datasetExecutionInformation.setLastPublishedRecords(lastPublishPlugin.getExecutionProgress().getProcessedRecords() - lastPublishPlugin.getExecutionProgress().getErrors());
            datasetExecutionInformation.setLastPublishedRecordsReadyForViewing(!isPublishCleaningOrRunning && this.isPreviewOrPublishReadyForViewing(lastPublishPlugin, now));
        }
        return datasetExecutionInformation;
    }

    private boolean isPreviewOrPublishReadyForViewing(AbstractExecutablePlugin plugin, Date now) {
        boolean dataIsValid = AbstractExecutablePlugin.getDataStatus((AbstractExecutablePlugin)plugin) == DataStatus.VALID;
        boolean enoughTimeHasPassed = (long)this.getSolrCommitPeriodInMins() < DateUtils.calculateDateDifference((Date)plugin.getFinishedDate(), (Date)now, (TimeUnit)TimeUnit.MINUTES);
        return dataIsValid && enoughTimeHasPassed;
    }

    private boolean isPluginInWorkflowCleaningOrRunning(WorkflowExecution runningOrInQueueExecution, PluginType pluginType) {
        return runningOrInQueueExecution != null && runningOrInQueueExecution.getMetisPlugins().stream().filter(metisPlugin -> metisPlugin.getPluginType() == pluginType).map(AbstractMetisPlugin::getPluginStatus).anyMatch(pluginStatus -> pluginStatus == PluginStatus.CLEANING || pluginStatus == PluginStatus.RUNNING);
    }

    private Workflow checkWorkflowExistence(String datasetId) throws NoWorkflowFoundException {
        Workflow workflow = this.workflowDao.getWorkflow(datasetId);
        if (workflow == null) {
            throw new NoWorkflowFoundException(String.format("No workflow found with datasetId: %s, in METIS", datasetId));
        }
        return workflow;
    }

    public VersionEvolution getRecordEvolutionForVersion(MetisUser metisUser, String workflowExecutionId, PluginType pluginType) throws GenericMetisException {
        WorkflowExecution workflowExecution = this.workflowExecutionDao.getById(workflowExecutionId);
        if (workflowExecution == null) {
            throw new NoWorkflowExecutionFoundException(String.format("No workflow execution found for workflowExecutionId: %s", workflowExecutionId));
        }
        this.authorizer.authorizeReadExistingDatasetById(metisUser, workflowExecution.getDatasetId());
        AbstractMetisPlugin plugin = (AbstractMetisPlugin)workflowExecution.getMetisPluginWithType(pluginType).orElseThrow(() -> new NoWorkflowExecutionFoundException(String.format("No plugin of type %s found for workflowExecution with id: %s", pluginType.name(), workflowExecutionId)));
        Pair<WorkflowExecution, AbstractMetisPlugin> currentExecutionAndPlugin = new Pair<WorkflowExecution, AbstractMetisPlugin>((Object)workflowExecution, (Object)plugin);
        ArrayDeque<VersionEvolution.VersionEvolutionStep> evolutionSteps = new ArrayDeque<VersionEvolution.VersionEvolutionStep>();
        while ((currentExecutionAndPlugin = this.orchestratorHelper.getPreviousExecutionAndPlugin((AbstractMetisPlugin)currentExecutionAndPlugin.getRight(), ((WorkflowExecution)currentExecutionAndPlugin.getLeft()).getDatasetId())) != null && currentExecutionAndPlugin.getRight() instanceof AbstractExecutablePlugin) {
            AbstractExecutablePlugin executablePlugin = (AbstractExecutablePlugin)currentExecutionAndPlugin.getRight();
            VersionEvolution.VersionEvolutionStep evolutionStep = new VersionEvolution.VersionEvolutionStep();
            evolutionStep.setWorkflowExecutionId(((WorkflowExecution)currentExecutionAndPlugin.getLeft()).getId().toString());
            evolutionStep.setPluginType(((AbstractExecutablePluginMetadata)executablePlugin.getPluginMetadata()).getExecutablePluginType());
            evolutionStep.setFinishedTime(executablePlugin.getFinishedDate());
            evolutionSteps.addFirst(evolutionStep);
        }
        VersionEvolution versionEvolution = new VersionEvolution();
        versionEvolution.setEvolutionSteps(evolutionSteps);
        return versionEvolution;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int getSolrCommitPeriodInMins() {
        OrchestratorService orchestratorService = this;
        synchronized (orchestratorService) {
            return this.solrCommitPeriodInMins;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void setSolrCommitPeriodInMins(int solrCommitPeriodInMins) {
        OrchestratorService orchestratorService = this;
        synchronized (orchestratorService) {
            this.solrCommitPeriodInMins = solrCommitPeriodInMins;
        }
    }
}

