/*
 * Decompiled with CFR 0.152.
 */
package eu.europeana.metis.authentication.service;

import com.zoho.crm.api.record.Record;
import eu.europeana.metis.authentication.dao.PsqlMetisUserDao;
import eu.europeana.metis.authentication.user.AccountRole;
import eu.europeana.metis.authentication.user.Credentials;
import eu.europeana.metis.authentication.user.MetisUser;
import eu.europeana.metis.authentication.user.MetisUserAccessToken;
import eu.europeana.metis.authentication.user.MetisUserView;
import eu.europeana.metis.authentication.utils.ZohoMetisUserUtils;
import eu.europeana.metis.exception.BadContentException;
import eu.europeana.metis.exception.GenericMetisException;
import eu.europeana.metis.exception.NoUserFoundException;
import eu.europeana.metis.exception.UserAlreadyExistsException;
import eu.europeana.metis.exception.UserUnauthorizedException;
import eu.europeana.metis.zoho.OrganizationRole;
import eu.europeana.metis.zoho.ZohoAccessClient;
import eu.europeana.metis.zoho.ZohoException;
import eu.europeana.metis.zoho.ZohoUtils;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.util.Base64;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.crypto.bcrypt.BCrypt;
import org.springframework.stereotype.Service;

@Service
public class AuthenticationService {
    private static final int LOG_ROUNDS = 13;
    private static final int CREDENTIAL_FIELDS_NUMBER = 2;
    private static final String ACCESS_TOKEN_CHARACTER_BASKET = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
    private static final int ACCESS_TOKEN_LENGTH = 32;
    private static final Pattern TOKEN_MATCHING_PATTERN = Pattern.compile("^[0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz]*$");
    public static final Supplier<BadContentException> COULD_NOT_CONVERT_EXCEPTION_SUPPLIER = () -> new BadContentException("Could not convert internal user");
    private final PsqlMetisUserDao psqlMetisUserDao;
    private final ZohoAccessClient zohoAccessClient;

    @Autowired
    public AuthenticationService(PsqlMetisUserDao psqlMetisUserDao, ZohoAccessClient zohoAccessClient) {
        this.psqlMetisUserDao = psqlMetisUserDao;
        this.zohoAccessClient = zohoAccessClient;
    }

    public void registerUser(String email, String password) throws GenericMetisException {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByEmail(email);
        if (Objects.nonNull(storedMetisUser)) {
            throw new UserAlreadyExistsException(String.format("User with email: %s already exists", email));
        }
        MetisUser metisUser = this.constructMetisUserFromZoho(email);
        String hashedPassword = this.generatePasswordHashing(password);
        metisUser.setPassword(hashedPassword);
        this.psqlMetisUserDao.createMetisUser(metisUser);
    }

    public MetisUserView updateUserFromZoho(String email) throws GenericMetisException {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByEmail(email);
        if (Objects.isNull(storedMetisUser)) {
            throw new NoUserFoundException(String.format("User with email: %s does not exist", email));
        }
        MetisUser metisUser = this.constructMetisUserFromZoho(email);
        metisUser.setPassword(storedMetisUser.getPassword());
        metisUser.setMetisUserAccessToken(storedMetisUser.getMetisUserAccessToken());
        if (storedMetisUser.getAccountRole() == AccountRole.METIS_ADMIN) {
            metisUser.setAccountRole(AccountRole.METIS_ADMIN);
        }
        this.psqlMetisUserDao.updateMetisUser(metisUser);
        return AuthenticationService.convert(storedMetisUser);
    }

    private MetisUser constructMetisUserFromZoho(String email) throws GenericMetisException {
        Optional zohoRecord;
        try {
            zohoRecord = this.zohoAccessClient.getZohoRecordContactByEmail(email);
        }
        catch (ZohoException e) {
            throw new GenericMetisException("Could not retrieve Zoho user", (Throwable)e);
        }
        if (zohoRecord.isEmpty()) {
            throw new NoUserFoundException("User was not found in Zoho");
        }
        MetisUser metisUser = ZohoMetisUserUtils.checkZohoFieldsAndPopulateMetisUser((Record)((Record)zohoRecord.get()));
        if (StringUtils.isBlank((CharSequence)metisUser.getOrganizationName()) || !metisUser.isMetisUserFlag() || metisUser.getAccountRole() == null) {
            throw new BadContentException("Bad content while constructing metisUser, user does not have all the required fields defined properly in Zoho(Organization Name, Metis user, Account Role)");
        }
        this.checkMetisUserOrganizationRole(metisUser);
        return metisUser;
    }

    private void checkMetisUserOrganizationRole(MetisUser metisUser) throws BadContentException {
        String organizationRoleString;
        Optional recordOrganization;
        try {
            recordOrganization = this.zohoAccessClient.getZohoRecordOrganizationByName(metisUser.getOrganizationName());
        }
        catch (ZohoException e) {
            throw new BadContentException("Could not retrieve Zoho organization", (Throwable)e);
        }
        if (recordOrganization.isEmpty()) {
            throw new BadContentException("Organization Role from Zoho is empty");
        }
        List organizationRoleStringList = ZohoUtils.stringListSupplier((Object)((Record)recordOrganization.get()).getKeyValue("Organisation_Role2"));
        OrganizationRole organizationRole = null;
        Iterator iterator = organizationRoleStringList.iterator();
        while (iterator.hasNext() && (organizationRole = OrganizationRole.getRoleFromName((String)(organizationRoleString = (String)iterator.next()))) == null) {
        }
        if (organizationRole == null) {
            throw new BadContentException("Organization Role from Zoho is empty");
        }
    }

    private String generatePasswordHashing(String password) {
        return BCrypt.hashpw((String)password, (String)BCrypt.gensalt((int)13));
    }

    private boolean isPasswordValid(MetisUser metisUser, String passwordToTry) {
        return BCrypt.checkpw((String)passwordToTry, (String)metisUser.getPassword());
    }

    public Credentials validateAuthorizationHeaderWithCredentials(String authorization) throws GenericMetisException {
        if (StringUtils.isBlank((CharSequence)authorization)) {
            throw new BadContentException("Authorization header was empty");
        }
        Credentials credentials = this.decodeAuthorizationHeaderWithCredentials(authorization);
        if (credentials == null) {
            throw new BadContentException("Username or password not provided, or not properly defined in the Authorization Header");
        }
        return credentials;
    }

    public String validateAuthorizationHeaderWithAccessToken(String authorization) throws GenericMetisException {
        if (StringUtils.isBlank((CharSequence)authorization)) {
            throw new UserUnauthorizedException("Authorization header was empty");
        }
        String accessToken = this.decodeAuthorizationHeaderWithAccessToken(authorization);
        if (StringUtils.isBlank((CharSequence)accessToken)) {
            throw new UserUnauthorizedException("Access token not provided properly");
        }
        if (accessToken.length() != 32 || !TOKEN_MATCHING_PATTERN.matcher(accessToken).matches()) {
            throw new UserUnauthorizedException("Access token invalid");
        }
        return accessToken;
    }

    private Credentials decodeAuthorizationHeaderWithCredentials(String authorization) {
        Credentials credentials = null;
        if (Objects.nonNull(authorization) && authorization.startsWith("Basic")) {
            String base64Credentials = authorization.substring("Basic".length()).trim();
            String credentialsString = new String(Base64.getDecoder().decode(base64Credentials), StandardCharsets.UTF_8);
            String[] splitCredentials = credentialsString.split(":", 2);
            if (splitCredentials.length == 2) {
                credentials = new Credentials(splitCredentials[0], splitCredentials[1]);
            }
        }
        return credentials;
    }

    private String decodeAuthorizationHeaderWithAccessToken(String authorization) {
        String accessToken = Objects.nonNull(authorization) && authorization.startsWith("Bearer") ? authorization.substring("Bearer".length()).trim() : "";
        return accessToken;
    }

    public MetisUserView loginUser(String email, String password) throws GenericMetisException {
        MetisUser storedMetisUser = this.authenticateUser(email, password);
        if (storedMetisUser.getMetisUserAccessToken() == null) {
            MetisUserAccessToken metisUserAccessToken = new MetisUserAccessToken(email, this.generateAccessToken(), new Date());
            this.psqlMetisUserDao.createUserAccessToken(metisUserAccessToken);
            storedMetisUser.setMetisUserAccessToken(metisUserAccessToken);
        } else {
            this.psqlMetisUserDao.updateAccessTokenTimestamp(email);
        }
        return AuthenticationService.convert(storedMetisUser);
    }

    public void updateUserPassword(String email, String newPassword) {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByEmail(email);
        String hashedPassword = this.generatePasswordHashing(newPassword);
        storedMetisUser.setPassword(hashedPassword);
        this.psqlMetisUserDao.updateMetisUser(storedMetisUser);
    }

    public void updateUserMakeAdmin(String userEmailToMakeAdmin) throws GenericMetisException {
        if (Objects.isNull(this.psqlMetisUserDao.getMetisUserByEmail(userEmailToMakeAdmin))) {
            throw new NoUserFoundException(String.format("User with email %s does not exist", userEmailToMakeAdmin));
        }
        this.psqlMetisUserDao.updateMetisUserToMakeAdmin(userEmailToMakeAdmin);
    }

    public boolean isUserAdmin(String accessToken) throws GenericMetisException {
        MetisUser storedMetisUser = this.authenticateUserInternal(accessToken);
        return storedMetisUser.getAccountRole() == AccountRole.METIS_ADMIN;
    }

    public boolean hasPermissionToRequestUserUpdate(String accessToken, String userEmailToUpdate) throws GenericMetisException {
        MetisUser storedMetisUserToUpdate = this.psqlMetisUserDao.getMetisUserByEmail(userEmailToUpdate);
        if (Objects.isNull(storedMetisUserToUpdate)) {
            throw new NoUserFoundException(String.format("User with email: %s does not exist", userEmailToUpdate));
        }
        MetisUser storedMetisUser = this.authenticateUserInternal(accessToken);
        return storedMetisUser.getAccountRole() == AccountRole.METIS_ADMIN || storedMetisUser.getEmail().equals(storedMetisUserToUpdate.getEmail());
    }

    String generateAccessToken() {
        SecureRandom rnd = new SecureRandom();
        StringBuilder sb = new StringBuilder(32);
        for (int i = 0; i < 32; ++i) {
            sb.append(ACCESS_TOKEN_CHARACTER_BASKET.charAt(rnd.nextInt(ACCESS_TOKEN_CHARACTER_BASKET.length())));
        }
        return sb.toString();
    }

    public void expireAccessTokens() {
        Date now = new Date();
        this.psqlMetisUserDao.expireAccessTokens(now);
    }

    public void deleteUser(String email) {
        this.psqlMetisUserDao.deleteMetisUser(email);
    }

    public MetisUser authenticateUser(String email, String password) throws UserUnauthorizedException {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByEmail(email);
        if (Objects.isNull(storedMetisUser) || !this.isPasswordValid(storedMetisUser, password)) {
            throw new UserUnauthorizedException("Wrong credentials");
        }
        return storedMetisUser;
    }

    public MetisUserView authenticateUser(String accessToken) throws GenericMetisException {
        return AuthenticationService.convert(this.authenticateUserInternal(accessToken));
    }

    private MetisUser authenticateUserInternal(String accessToken) throws GenericMetisException {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByAccessToken(accessToken);
        if (Objects.isNull(storedMetisUser)) {
            throw new UserUnauthorizedException("Wrong access token");
        }
        this.psqlMetisUserDao.updateAccessTokenTimestampByAccessToken(accessToken);
        return storedMetisUser;
    }

    public boolean hasPermissionToRequestAllUsers(String accessToken) throws GenericMetisException {
        MetisUser storedMetisUser = this.authenticateUserInternal(accessToken);
        return storedMetisUser.getAccountRole() == AccountRole.METIS_ADMIN || storedMetisUser.getAccountRole() == AccountRole.EUROPEANA_DATA_OFFICER;
    }

    public MetisUserView getMetisUserByUserIdOnlyWithPublicFields(String accessToken, String userIdToRetrieve) throws GenericMetisException {
        this.authenticateUser(accessToken);
        return AuthenticationService.convert(this.psqlMetisUserDao.getMetisUserByUserId(userIdToRetrieve));
    }

    public List<MetisUserView> getAllUsers() {
        return AuthenticationService.convert(this.psqlMetisUserDao.getAllMetisUsers());
    }

    private static MetisUserView convert(MetisUser record) throws BadContentException {
        return Optional.ofNullable(record).map(MetisUserView::new).orElseThrow(COULD_NOT_CONVERT_EXCEPTION_SUPPLIER);
    }

    private static List<MetisUserView> convert(List<MetisUser> records) {
        return Optional.ofNullable(records).stream().flatMap(Collection::stream).map(MetisUserView::new).collect(Collectors.toList());
    }
}

