/*
 * Decompiled with CFR 0.152.
 */
package eu.europeana.metis.authentication.service;

import eu.europeana.metis.authentication.dao.PsqlMetisUserDao;
import eu.europeana.metis.authentication.user.AccountRole;
import eu.europeana.metis.authentication.user.Credentials;
import eu.europeana.metis.authentication.user.MetisUser;
import eu.europeana.metis.authentication.user.MetisUserAccessToken;
import eu.europeana.metis.authentication.user.MetisUserView;
import eu.europeana.metis.exception.BadContentException;
import eu.europeana.metis.exception.GenericMetisException;
import eu.europeana.metis.exception.NoUserFoundException;
import eu.europeana.metis.exception.UserAlreadyRegisteredException;
import eu.europeana.metis.exception.UserUnauthorizedException;
import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.util.Base64;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.crypto.bcrypt.BCrypt;
import org.springframework.stereotype.Service;

@Service
public class AuthenticationService {
    public static final Supplier<BadContentException> COULD_NOT_CONVERT_EXCEPTION_SUPPLIER = () -> new BadContentException("Could not convert internal user");
    private static final int LOG_ROUNDS = 13;
    private static final int CREDENTIAL_FIELDS_NUMBER = 2;
    private static final String ACCESS_TOKEN_CHARACTER_BASKET = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
    private static final int ACCESS_TOKEN_LENGTH = 32;
    private static final Pattern TOKEN_MATCHING_PATTERN = Pattern.compile("^[0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz]*$");
    private final PsqlMetisUserDao psqlMetisUserDao;

    @Autowired
    public AuthenticationService(PsqlMetisUserDao psqlMetisUserDao) {
        this.psqlMetisUserDao = psqlMetisUserDao;
    }

    private static MetisUserView convert(MetisUser metisUser) throws BadContentException {
        return Optional.ofNullable(metisUser).map(MetisUserView::new).orElseThrow(COULD_NOT_CONVERT_EXCEPTION_SUPPLIER);
    }

    private static List<MetisUserView> convert(List<MetisUser> records) {
        return Optional.ofNullable(records).stream().flatMap(Collection::stream).map(MetisUserView::new).toList();
    }

    public void registerUser(String email, String password) throws GenericMetisException {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByEmail(email);
        if (Objects.isNull(storedMetisUser)) {
            throw new NoUserFoundException(String.format("User with email: %s doesn't exist", email));
        }
        if (storedMetisUser.getPassword() != null) {
            throw new UserAlreadyRegisteredException(String.format("User with email: %s already exists", email));
        }
        if (storedMetisUser.getUserId() == null) {
            long genId = UUID.randomUUID().getMostSignificantBits() & Long.MAX_VALUE;
            storedMetisUser.setUserId(Long.toString(genId));
        }
        storedMetisUser.setPassword(this.generatePasswordHashing(password));
        this.psqlMetisUserDao.updateMetisUser(storedMetisUser);
    }

    public Credentials validateAuthorizationHeaderWithCredentials(String authorization) throws GenericMetisException {
        if (StringUtils.isBlank((CharSequence)authorization)) {
            throw new BadContentException("Authorization header was empty");
        }
        Credentials credentials = this.decodeAuthorizationHeaderWithCredentials(authorization);
        if (credentials == null) {
            throw new BadContentException("Username or password not provided, or not properly defined in the Authorization Header");
        }
        return credentials;
    }

    public String validateAuthorizationHeaderWithAccessToken(String authorization) throws GenericMetisException {
        if (StringUtils.isBlank((CharSequence)authorization)) {
            throw new UserUnauthorizedException("Authorization header was empty");
        }
        String accessToken = this.decodeAuthorizationHeaderWithAccessToken(authorization);
        if (StringUtils.isBlank((CharSequence)accessToken)) {
            throw new UserUnauthorizedException("Access token not provided properly");
        }
        if (accessToken.length() != 32 || !TOKEN_MATCHING_PATTERN.matcher(accessToken).matches()) {
            throw new UserUnauthorizedException("Access token invalid");
        }
        return accessToken;
    }

    public MetisUserView loginUser(String email, String password) throws GenericMetisException {
        MetisUser storedMetisUser = this.authenticateUser(email, password);
        if (storedMetisUser.getMetisUserAccessToken() == null) {
            MetisUserAccessToken metisUserAccessToken = new MetisUserAccessToken(email, this.generateAccessToken(), new Date());
            this.psqlMetisUserDao.createUserAccessToken(metisUserAccessToken);
            storedMetisUser.setMetisUserAccessToken(metisUserAccessToken);
        } else {
            this.psqlMetisUserDao.updateAccessTokenTimestamp(email);
        }
        return AuthenticationService.convert(storedMetisUser);
    }

    public void updateUserPassword(String email, String newPassword) {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByEmail(email);
        String hashedPassword = this.generatePasswordHashing(newPassword);
        storedMetisUser.setPassword(hashedPassword);
        this.psqlMetisUserDao.updateMetisUser(storedMetisUser);
    }

    public void updateUserMakeAdmin(String userEmailToMakeAdmin) throws GenericMetisException {
        if (Objects.isNull(this.psqlMetisUserDao.getMetisUserByEmail(userEmailToMakeAdmin))) {
            throw new NoUserFoundException(String.format("User with email %s does not exist", userEmailToMakeAdmin));
        }
        this.psqlMetisUserDao.updateMetisUserToMakeAdmin(userEmailToMakeAdmin);
    }

    public boolean isUserAdmin(String accessToken) throws GenericMetisException {
        MetisUser storedMetisUser = this.authenticateUserInternal(accessToken);
        return storedMetisUser.getAccountRole() == AccountRole.METIS_ADMIN;
    }

    public boolean hasPermissionToRequestUserUpdate(String accessToken, String userEmailToUpdate) throws GenericMetisException {
        MetisUser storedMetisUserToUpdate = this.psqlMetisUserDao.getMetisUserByEmail(userEmailToUpdate);
        if (Objects.isNull(storedMetisUserToUpdate)) {
            throw new NoUserFoundException(String.format("User with email: %s does not exist", userEmailToUpdate));
        }
        MetisUser storedMetisUser = this.authenticateUserInternal(accessToken);
        return storedMetisUser.getAccountRole() == AccountRole.METIS_ADMIN || storedMetisUser.getEmail().equals(storedMetisUserToUpdate.getEmail());
    }

    public void expireAccessTokens() {
        Date now = new Date();
        this.psqlMetisUserDao.expireAccessTokens(now);
    }

    public void deleteUser(String email) {
        this.psqlMetisUserDao.deleteMetisUser(email);
    }

    public MetisUser authenticateUser(String email, String password) throws UserUnauthorizedException {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByEmail(email);
        if (Objects.isNull(storedMetisUser) || !this.isPasswordValid(storedMetisUser, password)) {
            throw new UserUnauthorizedException("Wrong credentials");
        }
        return storedMetisUser;
    }

    public MetisUserView authenticateUser(String accessToken) throws GenericMetisException {
        return AuthenticationService.convert(this.authenticateUserInternal(accessToken));
    }

    public boolean hasPermissionToRequestAllUsers(String accessToken) throws GenericMetisException {
        MetisUser storedMetisUser = this.authenticateUserInternal(accessToken);
        return storedMetisUser.getAccountRole() == AccountRole.METIS_ADMIN || storedMetisUser.getAccountRole() == AccountRole.EUROPEANA_DATA_OFFICER;
    }

    public MetisUserView getMetisUserByUserIdOnlyWithPublicFields(String accessToken, String userIdToRetrieve) throws GenericMetisException {
        this.authenticateUser(accessToken);
        return AuthenticationService.convert(this.psqlMetisUserDao.getMetisUserByUserId(userIdToRetrieve));
    }

    public List<MetisUserView> getAllUsers() {
        return AuthenticationService.convert(this.psqlMetisUserDao.getAllMetisUsers());
    }

    private String generatePasswordHashing(String password) {
        return BCrypt.hashpw((String)password, (String)BCrypt.gensalt((int)13));
    }

    private boolean isPasswordValid(MetisUser metisUser, String passwordToTry) {
        return BCrypt.checkpw((String)passwordToTry, (String)metisUser.getPassword());
    }

    private Credentials decodeAuthorizationHeaderWithCredentials(String authorization) {
        Credentials credentials = null;
        if (Objects.nonNull(authorization) && authorization.startsWith("Basic")) {
            String base64Credentials = authorization.substring("Basic".length()).trim();
            String credentialsString = new String(Base64.getDecoder().decode(base64Credentials), StandardCharsets.UTF_8);
            String[] splitCredentials = credentialsString.split(":", 2);
            if (splitCredentials.length == 2) {
                credentials = new Credentials(splitCredentials[0], splitCredentials[1]);
            }
        }
        return credentials;
    }

    private String decodeAuthorizationHeaderWithAccessToken(String authorization) {
        String accessToken = Objects.nonNull(authorization) && authorization.startsWith("Bearer") ? authorization.substring("Bearer".length()).trim() : "";
        return accessToken;
    }

    private MetisUser authenticateUserInternal(String accessToken) throws GenericMetisException {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByAccessToken(accessToken);
        if (Objects.isNull(storedMetisUser)) {
            throw new UserUnauthorizedException("Wrong access token");
        }
        this.psqlMetisUserDao.updateAccessTokenTimestampByAccessToken(accessToken);
        return storedMetisUser;
    }

    String generateAccessToken() {
        SecureRandom rnd = new SecureRandom();
        StringBuilder sb = new StringBuilder(32);
        for (int i = 0; i < 32; ++i) {
            sb.append(ACCESS_TOKEN_CHARACTER_BASKET.charAt(rnd.nextInt(ACCESS_TOKEN_CHARACTER_BASKET.length())));
        }
        return sb.toString();
    }
}

