/*
 * Decompiled with CFR 0.152.
 */
package eu.europeana.metis.authentication.service;

import com.fasterxml.jackson.databind.JsonNode;
import eu.europeana.metis.authentication.dao.PsqlMetisUserDao;
import eu.europeana.metis.authentication.dao.ZohoAccessClientDao;
import eu.europeana.metis.authentication.user.AccountRole;
import eu.europeana.metis.authentication.user.Credentials;
import eu.europeana.metis.authentication.user.MetisUser;
import eu.europeana.metis.authentication.user.MetisUserAccessToken;
import eu.europeana.metis.exception.BadContentException;
import eu.europeana.metis.exception.GenericMetisException;
import eu.europeana.metis.exception.NoUserFoundException;
import eu.europeana.metis.exception.UserAlreadyExistsException;
import eu.europeana.metis.exception.UserUnauthorizedException;
import java.nio.charset.Charset;
import java.security.SecureRandom;
import java.text.ParseException;
import java.util.Base64;
import java.util.Date;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.crypto.bcrypt.BCrypt;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

@Service
public class AuthenticationService {
    private static final int LOG_ROUNDS = 13;
    private static final int CREDENTIAL_FIELDS_NUMBER = 2;
    private static final String ACCESS_TOKEN_CHARACTER_BASKET = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz";
    private static final int ACCESS_TOKEN_LENGTH = 32;
    private final ZohoAccessClientDao zohoAccessClientDao;
    private final PsqlMetisUserDao psqlMetisUserDao;

    @Autowired
    public AuthenticationService(ZohoAccessClientDao zohoAccessClientDao, PsqlMetisUserDao psqlMetisUserDao) {
        this.zohoAccessClientDao = zohoAccessClientDao;
        this.psqlMetisUserDao = psqlMetisUserDao;
    }

    public void registerUser(String email, String password) throws GenericMetisException {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByEmail(email);
        if (storedMetisUser != null) {
            throw new UserAlreadyExistsException(String.format("User with email: %s already exists", email));
        }
        MetisUser metisUser = this.constructMetisUserFromZoho(email);
        String hashedPassword = this.generatePasswordHashing(password);
        metisUser.setPassword(hashedPassword);
        this.psqlMetisUserDao.createMetisUser(metisUser);
    }

    public MetisUser updateUserFromZoho(String email) throws GenericMetisException {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByEmail(email);
        if (storedMetisUser == null) {
            throw new NoUserFoundException(String.format("User with email: %s does not exist", email));
        }
        MetisUser metisUser = this.constructMetisUserFromZoho(email);
        metisUser.setPassword(storedMetisUser.getPassword());
        metisUser.setMetisUserAccessToken(storedMetisUser.getMetisUserAccessToken());
        if (storedMetisUser.getAccountRole() == AccountRole.METIS_ADMIN) {
            metisUser.setAccountRole(AccountRole.METIS_ADMIN);
        }
        this.psqlMetisUserDao.updateMetisUser(metisUser);
        return metisUser;
    }

    private MetisUser constructMetisUserFromZoho(String email) throws GenericMetisException {
        MetisUser metisUser;
        JsonNode userByEmailJsonNode = this.zohoAccessClientDao.getUserByEmail(email);
        if (userByEmailJsonNode == null) {
            throw new NoUserFoundException("User was not found in Zoho");
        }
        try {
            metisUser = new MetisUser(userByEmailJsonNode);
        }
        catch (ParseException e) {
            throw new BadContentException("Bad content while constructing metisUser", (Throwable)e);
        }
        if (StringUtils.isEmpty((Object)metisUser.getOrganizationName()) || !metisUser.isMetisUserFlag() || metisUser.getAccountRole() == null) {
            throw new BadContentException("Bad content while constructing metisUser, user does not have all the required fields defined properly in Zoho(Organization Name, Metis user, Account Role)");
        }
        String organizationId = this.zohoAccessClientDao.getOrganizationIdByOrganizationName(metisUser.getOrganizationName());
        metisUser.setOrganizationId(organizationId);
        return metisUser;
    }

    private String generatePasswordHashing(String password) {
        return BCrypt.hashpw((String)password, (String)BCrypt.gensalt((int)13));
    }

    private boolean isPasswordValid(MetisUser metisUser, String passwordToTry) {
        return BCrypt.checkpw((String)passwordToTry, (String)metisUser.getPassword());
    }

    public Credentials validateAuthorizationHeaderWithCredentials(String authorization) throws GenericMetisException {
        if (StringUtils.isEmpty((Object)authorization)) {
            throw new BadContentException("Authorization header was empty");
        }
        Credentials credentials = this.decodeAuthorizationHeaderWithCredentials(authorization);
        if (credentials == null) {
            throw new BadContentException("Username or password not provided, or not properly defined in the Authorization Header");
        }
        return credentials;
    }

    public String validateAuthorizationHeaderWithAccessToken(String authorization) throws GenericMetisException {
        if (StringUtils.isEmpty((Object)authorization)) {
            throw new UserUnauthorizedException("Authorization header was empty");
        }
        String accessToken = this.decodeAuthorizationHeaderWithAccessToken(authorization);
        if (StringUtils.isEmpty((Object)accessToken)) {
            throw new UserUnauthorizedException("Access token not provided properly");
        }
        if (accessToken.length() != 32 || !accessToken.matches("^[0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz]*$")) {
            throw new UserUnauthorizedException("Access token invalid");
        }
        return accessToken;
    }

    private Credentials decodeAuthorizationHeaderWithCredentials(String authorization) {
        if (authorization != null && authorization.startsWith("Basic")) {
            String base64Credentials = authorization.substring("Basic".length()).trim();
            String credentialsString = new String(Base64.getDecoder().decode(base64Credentials), Charset.forName("UTF-8"));
            String[] splittedCredentials = credentialsString.split(":", 2);
            if (splittedCredentials.length != 2) {
                return null;
            }
            return new Credentials(splittedCredentials[0], splittedCredentials[1]);
        }
        return null;
    }

    private String decodeAuthorizationHeaderWithAccessToken(String authorization) {
        if (authorization != null && authorization.startsWith("Bearer")) {
            return authorization.substring("Bearer".length()).trim();
        }
        return "";
    }

    public MetisUser loginUser(String email, String password) throws GenericMetisException {
        MetisUser storedMetisUser = this.authenticateUser(email, password);
        if (storedMetisUser.getMetisUserAccessToken() != null) {
            this.psqlMetisUserDao.updateAccessTokenTimestamp(email);
            return storedMetisUser;
        }
        MetisUserAccessToken metisUserAccessToken = new MetisUserAccessToken(email, this.generateAccessToken(), new Date());
        this.psqlMetisUserDao.createUserAccessToken(metisUserAccessToken);
        storedMetisUser.setMetisUserAccessToken(metisUserAccessToken);
        return storedMetisUser;
    }

    public void updateUserPassword(MetisUser metisUser, String newPassword) {
        String hashedPassword = this.generatePasswordHashing(newPassword);
        metisUser.setPassword(hashedPassword);
        this.psqlMetisUserDao.updateMetisUser(metisUser);
    }

    public void updateUserMakeAdmin(String userEmailToMakeAdmin) throws GenericMetisException {
        if (this.psqlMetisUserDao.getMetisUserByEmail(userEmailToMakeAdmin) == null) {
            throw new NoUserFoundException(String.format("User with email %s does not exist", userEmailToMakeAdmin));
        }
        this.psqlMetisUserDao.updateMetisUserToMakeAdmin(userEmailToMakeAdmin);
    }

    public boolean isUserAdmin(String accessToken) throws GenericMetisException {
        MetisUser storedMetisUser = this.authenticateUser(accessToken);
        return storedMetisUser.getAccountRole() == AccountRole.METIS_ADMIN;
    }

    public boolean hasPermissionToRequestUserUpdate(String accessToken, String userEmailToUpdate) throws GenericMetisException {
        MetisUser storedMetisUserToUpdate = this.psqlMetisUserDao.getMetisUserByEmail(userEmailToUpdate);
        if (storedMetisUserToUpdate == null) {
            throw new NoUserFoundException(String.format("User with email: %s does not exist", userEmailToUpdate));
        }
        MetisUser storedMetisUser = this.authenticateUser(accessToken);
        return storedMetisUser.getAccountRole() == AccountRole.METIS_ADMIN || storedMetisUser.getEmail().equals(storedMetisUserToUpdate.getEmail());
    }

    String generateAccessToken() {
        SecureRandom rnd = new SecureRandom();
        StringBuilder sb = new StringBuilder(32);
        for (int i = 0; i < 32; ++i) {
            sb.append(ACCESS_TOKEN_CHARACTER_BASKET.charAt(rnd.nextInt(ACCESS_TOKEN_CHARACTER_BASKET.length())));
        }
        return sb.toString();
    }

    public void expireAccessTokens() {
        Date now = new Date();
        this.psqlMetisUserDao.expireAccessTokens(now);
    }

    public void deleteUser(String email) {
        this.psqlMetisUserDao.deleteMetisUser(email);
    }

    public MetisUser authenticateUser(String email, String password) throws UserUnauthorizedException {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByEmail(email);
        if (storedMetisUser == null || !this.isPasswordValid(storedMetisUser, password)) {
            throw new UserUnauthorizedException("Wrong credentials");
        }
        return storedMetisUser;
    }

    public MetisUser authenticateUser(String accessToken) throws GenericMetisException {
        MetisUser storedMetisUser = this.psqlMetisUserDao.getMetisUserByAccessToken(accessToken);
        if (storedMetisUser == null) {
            throw new UserUnauthorizedException("Wrong access token");
        }
        this.psqlMetisUserDao.updateAccessTokenTimestampByAccessToken(accessToken);
        return storedMetisUser;
    }

    public boolean hasPermissionToRequestAllUsers(String accessToken) throws GenericMetisException {
        MetisUser storedMetisUser = this.authenticateUser(accessToken);
        return storedMetisUser.getAccountRole() == AccountRole.METIS_ADMIN || storedMetisUser.getAccountRole() == AccountRole.EUROPEANA_DATA_OFFICER;
    }

    public MetisUser getMetisUserByUserIdOnlyWithPublicFields(String accessToken, String userIdToRetrieve) throws GenericMetisException {
        this.authenticateUser(accessToken);
        return this.psqlMetisUserDao.getMetisUserByUserId(userIdToRetrieve);
    }

    public List<MetisUser> getAllUsers(String accessToken) {
        List<MetisUser> allMetisUsers = this.psqlMetisUserDao.getAllMetisUsers();
        MetisUser metisUserByEmail = this.psqlMetisUserDao.getMetisUserByAccessToken(accessToken);
        if (metisUserByEmail.getAccountRole() != AccountRole.METIS_ADMIN) {
            for (MetisUser metisUser : allMetisUsers) {
                metisUser.setMetisUserAccessToken(null);
            }
        }
        return allMetisUsers;
    }
}

