/*
 * Decompiled with CFR 0.152.
 */
package eu.europeana.fulltext.api.service;

import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Date;
import java.util.Objects;
import javax.servlet.http.HttpServletRequest;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.utils.DateUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.MultiValueMap;

public final class CacheUtils {
    private static final Logger LOG = LogManager.getLogger(CacheUtils.class);
    private static final String IFNONEMATCH = "If-None-Match";
    private static final String IFMATCH = "If-Match";
    private static final String IFMODIFIEDSINCE = "If-Modified-Since";
    private static final String ANY = "*";
    private static final String ALLOWED = "GET, HEAD";
    private static final String ALLOWHEADERS = "If-Match, If-None-Match, If-Modified-Since";
    private static final String EXPOSEHEADERS = "Allow, ETag, Last-Modified, Link";
    private static final String CACHE_CONTROL = "public, max-age=";
    private static final String DEFAULT_MAX_AGE = "86400";
    private static final String ACCEPT = "Accept";
    private static final String MAX_AGE_600 = "600";

    private CacheUtils() {
    }

    public static String generateETag(String alldata, ZonedDateTime modified, String version, boolean weakETag) {
        String data = alldata + CacheUtils.zonedDateTimeToString(modified) + version;
        String eTag = "\"" + CacheUtils.getSHA256Hash(data) + "\"";
        if (weakETag) {
            return "W/" + eTag;
        }
        return eTag;
    }

    public static String generateSimpleETag(String data, boolean weakETag) {
        String eTag = "\"" + CacheUtils.getSHA256Hash(data) + "\"";
        if (weakETag) {
            return "W/" + eTag;
        }
        return eTag;
    }

    public static String zonedDateTimeToString(ZonedDateTime zonedDateTime) {
        return zonedDateTime.format(DateTimeFormatter.RFC_1123_DATE_TIME);
    }

    public static ZonedDateTime dateToZonedUTC(Date date) {
        return date.toInstant().atOffset(ZoneOffset.UTC).toZonedDateTime().withNano(0);
    }

    public static ZonedDateTime januarificator() {
        return ZonedDateTime.of(1990, 1, 11, 0, 0, 0, 0, ZoneId.of("UTC"));
    }

    public static ResponseEntity<String> checkCached(HttpServletRequest request, ZonedDateTime modified, String eTag) {
        if (StringUtils.isNotBlank((CharSequence)request.getHeader(IFNONEMATCH))) {
            if (CacheUtils.doesAnyIfNoneMatch(request, eTag)) {
                HttpHeaders headers = CacheUtils.generateHeaders(request, eTag, CacheUtils.zonedDateTimeToString(modified));
                return new ResponseEntity((MultiValueMap)headers, HttpStatus.NOT_MODIFIED);
            }
        } else if (StringUtils.isNotBlank((CharSequence)request.getHeader(IFMATCH))) {
            if (CacheUtils.doesPreconditionFail(request, eTag)) {
                return new ResponseEntity(HttpStatus.PRECONDITION_FAILED);
            }
        } else if (CacheUtils.isNotModifiedSince(request, modified)) {
            return new ResponseEntity(HttpStatus.NOT_MODIFIED);
        }
        return null;
    }

    public static HttpHeaders generateHeaders(HttpServletRequest request, String eTag, String modified) {
        return CacheUtils.generateHeaders(request, eTag, modified, null);
    }

    public static HttpHeaders generateHeaders(HttpServletRequest request, String eTag, String modified, Integer maxAge) {
        HttpHeaders headers = new HttpHeaders();
        if (StringUtils.isNotBlank((CharSequence)request.getHeader("Origin"))) {
            headers.add("Access-Control-Allow-Methods", ALLOWED);
            headers.add("Access-Control-Allow-Headers", ALLOWHEADERS);
            headers.add("Access-Control-Expose-Headers", EXPOSEHEADERS);
        }
        if (StringUtils.isNotBlank((CharSequence)eTag)) {
            headers.add("ETag", eTag);
        }
        if (StringUtils.isNotBlank((CharSequence)modified)) {
            headers.add("Last-Modified", modified);
        }
        headers.add("Access-Control-Max-Age", MAX_AGE_600);
        headers.add("Allow", ALLOWED);
        headers.add("Cache-Control", CACHE_CONTROL + (Serializable)(maxAge == null ? DEFAULT_MAX_AGE : maxAge));
        headers.add("Vary", ACCEPT);
        return headers;
    }

    private static boolean doesAnyIfNoneMatch(HttpServletRequest request, String eTag) {
        return StringUtils.isNotBlank((CharSequence)request.getHeader(IFNONEMATCH)) && CacheUtils.doesAnyETagMatch(request.getHeader(IFNONEMATCH), eTag);
    }

    private static boolean isNotModifiedSince(HttpServletRequest request, ZonedDateTime modified) {
        return StringUtils.isNotBlank((CharSequence)request.getHeader(IFMODIFIEDSINCE)) && Objects.requireNonNull(CacheUtils.stringToZonedUTC(request.getHeader(IFMODIFIEDSINCE))).compareTo(modified) >= 0;
    }

    private static boolean doesPreconditionFail(HttpServletRequest request, String eTag) {
        return StringUtils.isNotBlank((CharSequence)request.getHeader(IFMATCH)) && !CacheUtils.doesAnyETagMatch(request.getHeader(IFMATCH), eTag);
    }

    private static boolean doesAnyETagMatch(String eTags, String eTagToMatch) {
        if (StringUtils.equals((CharSequence)ANY, (CharSequence)eTags)) {
            return true;
        }
        if (StringUtils.isNoneBlank((CharSequence[])new CharSequence[]{eTags, eTagToMatch})) {
            for (String eTag : StringUtils.stripAll((String[])StringUtils.split((String)eTags, (String)","))) {
                if (!StringUtils.equalsIgnoreCase((CharSequence)CacheUtils.spicAndSpan(eTag), (CharSequence)CacheUtils.spicAndSpan(eTagToMatch))) continue;
                return true;
            }
        }
        return false;
    }

    private static String spicAndSpan(String header) {
        return StringUtils.remove((String)StringUtils.stripStart((String)header, (String)"W/"), (String)"\"");
    }

    private static String getSHA256Hash(String data) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] encodedhash = digest.digest(data.getBytes(StandardCharsets.UTF_8));
            return CacheUtils.bytesToHex(encodedhash);
        }
        catch (NoSuchAlgorithmException e) {
            LogManager.getLogger(CacheUtils.class).error("Error generating SHA-265 hash from record timestamp_update", (Throwable)e);
            return null;
        }
    }

    private static String bytesToHex(byte[] hash) {
        StringBuilder hexString = new StringBuilder();
        for (byte bt : hash) {
            String hex = Integer.toHexString(0xFF & bt);
            if (hex.length() == 1) {
                hexString.append('0');
            }
            hexString.append(hex);
        }
        return hexString.toString();
    }

    private static ZonedDateTime stringToZonedUTC(String dateString) {
        if (StringUtils.isEmpty((CharSequence)dateString)) {
            return null;
        }
        Date date = DateUtils.parseDate((String)dateString);
        if (date == null) {
            LOG.error("Error parsing request header Date string: {}", (Object)dateString);
            return null;
        }
        return CacheUtils.dateToZonedUTC(date);
    }
}

