/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.boot.jdbc;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.Locale;
import java.util.stream.Stream;
import javax.sql.DataSource;
import org.springframework.boot.jdbc.DatabaseDriver;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.ConnectionCallback;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.datasource.embedded.EmbeddedDatabaseType;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;

public enum EmbeddedDatabaseConnection {
    NONE(null),
    H2("jdbc:h2:mem:%s;DB_CLOSE_DELAY=-1;DB_CLOSE_ON_EXIT=FALSE"),
    DERBY("jdbc:derby:memory:%s;create=true"),
    HSQLDB("org.hsqldb.jdbcDriver", "jdbc:hsqldb:mem:%s");

    private final String alternativeDriverClass;
    private final String url;

    private EmbeddedDatabaseConnection(String url) {
        this(null, url);
    }

    private EmbeddedDatabaseConnection(String fallbackDriverClass, String url) {
        this.alternativeDriverClass = fallbackDriverClass;
        this.url = url;
    }

    public String getDriverClassName() {
        return switch (this) {
            default -> throw new IncompatibleClassChangeError();
            case NONE -> null;
            case H2 -> DatabaseDriver.H2.getDriverClassName();
            case DERBY -> DatabaseDriver.DERBY.getDriverClassName();
            case HSQLDB -> DatabaseDriver.HSQLDB.getDriverClassName();
        };
    }

    public EmbeddedDatabaseType getType() {
        return switch (this) {
            default -> throw new IncompatibleClassChangeError();
            case NONE -> null;
            case H2 -> EmbeddedDatabaseType.H2;
            case DERBY -> EmbeddedDatabaseType.DERBY;
            case HSQLDB -> EmbeddedDatabaseType.HSQL;
        };
    }

    public String getUrl(String databaseName) {
        Assert.hasText(databaseName, "DatabaseName must not be empty");
        return this.url != null ? String.format(this.url, databaseName) : null;
    }

    boolean isEmbeddedUrl(String url) {
        return switch (this) {
            default -> throw new IncompatibleClassChangeError();
            case NONE -> false;
            case H2 -> url.contains(":h2:mem");
            case DERBY -> true;
            case HSQLDB -> url.contains(":hsqldb:mem:");
        };
    }

    boolean isDriverCompatible(String driverClass) {
        return driverClass != null && (driverClass.equals(this.getDriverClassName()) || driverClass.equals(this.alternativeDriverClass));
    }

    public static boolean isEmbedded(String driverClass, String url) {
        if (driverClass == null) {
            return false;
        }
        EmbeddedDatabaseConnection connection = EmbeddedDatabaseConnection.getEmbeddedDatabaseConnection(driverClass);
        if (connection == NONE) {
            return false;
        }
        return url == null || connection.isEmbeddedUrl(url);
    }

    private static EmbeddedDatabaseConnection getEmbeddedDatabaseConnection(String driverClass) {
        return Stream.of(H2, HSQLDB, DERBY).filter(connection -> connection.isDriverCompatible(driverClass)).findFirst().orElse(NONE);
    }

    public static boolean isEmbedded(DataSource dataSource) {
        try {
            return new JdbcTemplate(dataSource).execute(new IsEmbedded());
        }
        catch (DataAccessException ex) {
            return false;
        }
    }

    public static EmbeddedDatabaseConnection get(ClassLoader classLoader) {
        for (EmbeddedDatabaseConnection candidate : EmbeddedDatabaseConnection.values()) {
            if (candidate == NONE || !ClassUtils.isPresent(candidate.getDriverClassName(), classLoader)) continue;
            return candidate;
        }
        return NONE;
    }

    private static final class IsEmbedded
    implements ConnectionCallback<Boolean> {
        private IsEmbedded() {
        }

        @Override
        public Boolean doInConnection(Connection connection) throws SQLException, DataAccessException {
            EmbeddedDatabaseConnection[] candidates;
            DatabaseMetaData metaData = connection.getMetaData();
            String productName = metaData.getDatabaseProductName();
            if (productName == null) {
                return false;
            }
            productName = productName.toUpperCase(Locale.ENGLISH);
            for (EmbeddedDatabaseConnection candidate : candidates = EmbeddedDatabaseConnection.values()) {
                if (candidate == NONE || !productName.contains(candidate.getType().name())) continue;
                String url = metaData.getURL();
                return url == null || candidate.isEmbeddedUrl(url);
            }
            return false;
        }
    }
}

