/*
 * Decompiled with CFR 0.152.
 */
package eu.europeana.metis.mediaprocessing.extraction;

import eu.europeana.metis.mediaprocessing.exception.MediaExtractionException;
import eu.europeana.metis.mediaprocessing.exception.MediaProcessorException;
import eu.europeana.metis.mediaprocessing.extraction.CommandExecutor;
import eu.europeana.metis.mediaprocessing.extraction.ImageMetadata;
import eu.europeana.metis.mediaprocessing.model.Thumbnail;
import eu.europeana.metis.mediaprocessing.model.ThumbnailImpl;
import eu.europeana.metis.mediaprocessing.model.ThumbnailKind;
import eu.europeana.metis.schema.model.MediaType;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.UUID;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ThumbnailGenerator {
    private static final Logger LOGGER = LoggerFactory.getLogger(ThumbnailGenerator.class);
    private static final String PNG_MIME_TYPE = "image/png";
    private static final String JPEG_MIME_TYPE = "image/jpeg";
    private static final String COMMAND_RESULT_FORMAT = "\n%w\n%h\n%[colorspace]\n";
    private static final int COMMAND_RESULT_WIDTH_LINE = 0;
    private static final int COMMAND_RESULT_HEIGHT_LINE = 1;
    private static final int COMMAND_RESULT_COLORSPACE_LINE = 2;
    private static final int COMMAND_RESULT_MAX_COLORS = 6;
    public static final String COLORMAP_PNG = "colormap.png";
    private static String globalMagickCommand;
    private static Path globalColormapFile;
    private final String magickCmd;
    private final String colormapFile;
    private final CommandExecutor commandExecutor;

    ThumbnailGenerator(CommandExecutor commandExecutor) throws MediaProcessorException {
        this(commandExecutor, ThumbnailGenerator.getGlobalImageMagickCommand(commandExecutor), ThumbnailGenerator.initColorMap().toString());
    }

    ThumbnailGenerator(CommandExecutor commandExecutor, String magickCommand, String colorMapFile) {
        this.commandExecutor = commandExecutor;
        this.magickCmd = magickCommand;
        this.colormapFile = colorMapFile;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Path initColorMap() throws MediaProcessorException {
        Class<ThumbnailGenerator> clazz = ThumbnailGenerator.class;
        synchronized (ThumbnailGenerator.class) {
            Path colormapTempFile;
            if (globalColormapFile != null) {
                // ** MonitorExit[var0] (shouldn't be in output)
                return globalColormapFile;
            }
            try (InputStream colorMapInputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream(COLORMAP_PNG);){
                if (colorMapInputStream == null) {
                    throw new MediaProcessorException("Could not load color map file: could not find file.");
                }
                colormapTempFile = Files.createTempFile("colormap", ".png", new FileAttribute[0]);
                Files.copy(colorMapInputStream, colormapTempFile, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException e) {
                throw new MediaProcessorException(String.format("Could not load color map file: %s", COLORMAP_PNG), e);
            }
            colormapTempFile.toFile().deleteOnExit();
            globalColormapFile = colormapTempFile;
            // ** MonitorExit[var0] (shouldn't be in output)
            return globalColormapFile;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static String getGlobalImageMagickCommand(CommandExecutor commandExecutor) throws MediaProcessorException {
        Class<ThumbnailGenerator> clazz = ThumbnailGenerator.class;
        synchronized (ThumbnailGenerator.class) {
            if (globalMagickCommand == null) {
                globalMagickCommand = ThumbnailGenerator.discoverImageMagickCommand(commandExecutor);
            }
            // ** MonitorExit[var1_1] (shouldn't be in output)
            return globalMagickCommand;
        }
    }

    static String discoverImageMagickCommand(CommandExecutor commandExecutor) throws MediaProcessorException {
        List<Object> paths;
        try {
            String im7Response = commandExecutor.execute(Arrays.asList("magick", "-version"), true, MediaProcessorException::new);
            if (im7Response.startsWith("Version: ImageMagick 7")) {
                String result = "magick";
                LOGGER.info("Found ImageMagic 7. Command: {}", (Object)"magick");
                return "magick";
            }
        }
        catch (MediaProcessorException e) {
            LOGGER.info("Could not find ImageMagick 7 because of: {}.", (Object)e.getMessage());
            LOGGER.debug("Could not find ImageMagick 7 due to following problem.", e);
        }
        boolean isWindows = System.getProperty("os.name").toLowerCase(Locale.ENGLISH).contains("win");
        try {
            paths = ThumbnailGenerator.splitByNewLine(commandExecutor.execute(Arrays.asList(isWindows ? "where" : "which", "convert"), true, MediaProcessorException::new));
        }
        catch (MediaProcessorException e) {
            LOGGER.warn("Could not find ImageMagick 6 due to following problem.", e);
            paths = Collections.emptyList();
        }
        for (String path : paths) {
            try {
                String pathResult = commandExecutor.execute(Arrays.asList(path, "-version"), true, MediaProcessorException::new);
                if (!pathResult.startsWith("Version: ImageMagick 6")) continue;
                LOGGER.info("Found ImageMagic 6. Command: {}", (Object)path);
                return path;
            }
            catch (MediaProcessorException e) {
                LOGGER.info("Could not find ImageMagick 6 at path {} because of: {}.", (Object)path, (Object)e.getMessage());
                LOGGER.debug("Could not find ImageMagick 6 at path {} due to following problem.", (Object)path, (Object)e);
            }
        }
        LOGGER.error("Could not find ImageMagick 6 or 7. See previous log statements for details.");
        throw new MediaProcessorException("Could not find ImageMagick 6 or 7.");
    }

    private static List<String> splitByNewLine(String input) {
        return Stream.of(input.split("\\R")).filter(StringUtils::isNotBlank).collect(Collectors.toList());
    }

    Pair<ImageMetadata, List<Thumbnail>> generateThumbnails(String url, String detectedMimeType, File content, boolean removeAlpha) throws MediaExtractionException {
        ImageMetadata image;
        if (content == null) {
            throw new MediaExtractionException("File content is null");
        }
        if (MediaType.getMediaType(detectedMimeType) != MediaType.IMAGE) {
            throw new MediaExtractionException("Cannot perform thumbnail generation on mime type '" + detectedMimeType + "'.");
        }
        if (detectedMimeType.startsWith("image/vnd.djvu") || detectedMimeType.startsWith("image/x-djvu") || detectedMimeType.startsWith("image/x.djvu")) {
            throw new MediaExtractionException("Cannot generate thumbnails for DjVu file.");
        }
        List<ThumbnailWithSize> thumbnails = this.prepareThumbnailFiles(url, detectedMimeType);
        try {
            image = this.generateThumbnailsInternal(thumbnails, removeAlpha, content);
        }
        catch (RuntimeException e) {
            ThumbnailGenerator.closeAllThumbnailsSilently(thumbnails);
            throw new MediaExtractionException("Unexpected error during processing", e);
        }
        catch (MediaExtractionException e) {
            ThumbnailGenerator.closeAllThumbnailsSilently(thumbnails);
            throw e;
        }
        finally {
            thumbnails.forEach(ThumbnailWithSize::deleteTempFileSilently);
        }
        List resultThumbnails = thumbnails.stream().map(ThumbnailWithSize::getThumbnail).collect(Collectors.toList());
        return new ImmutablePair<ImageMetadata, List<Thumbnail>>(image, resultThumbnails);
    }

    private static void closeAllThumbnailsSilently(List<ThumbnailWithSize> thumbnails) {
        for (ThumbnailWithSize thumbnail : thumbnails) {
            thumbnail.getThumbnail().close();
        }
    }

    List<String> createThumbnailGenerationCommand(List<ThumbnailWithSize> thumbnails, boolean removeAlpha, File content, String contentMarker) {
        String commandResultFormat = contentMarker + COMMAND_RESULT_FORMAT + contentMarker + "\n";
        ArrayList<String> command = new ArrayList<String>(Arrays.asList(this.magickCmd, "-quiet", content.getPath() + "[0]", "-format", commandResultFormat, "-write", "info:"));
        if (removeAlpha) {
            command.addAll(Arrays.asList("-background", "white", "-alpha", "remove"));
        }
        int thumbnailCounter = thumbnails.size();
        for (int i = 0; i < thumbnailCounter; ++i) {
            if (i != thumbnailCounter - 1) {
                command.add("(");
                command.add("+clone");
            }
            ThumbnailWithSize thumbnail = thumbnails.get(i);
            command.addAll(Arrays.asList("-thumbnail", thumbnail.getImageSize() + "x", "-write", thumbnail.getImageMagickTypePrefix() + thumbnail.getTempFileForThumbnail().toString()));
            if (i == thumbnailCounter - 1) continue;
            command.add("+delete");
            command.add(")");
        }
        String colorResultFormat = "\n" + contentMarker + "\n%c\n" + contentMarker;
        command.addAll(Arrays.asList("-colorspace", "sRGB", "-dither", "Riemersma", "-remap", this.colormapFile, "-format", colorResultFormat, "histogram:info:"));
        return command;
    }

    private ImageMetadata generateThumbnailsInternal(List<ThumbnailWithSize> thumbnails, boolean removeAlpha, File content) throws MediaExtractionException {
        String contentMarker = UUID.randomUUID().toString();
        List<String> command = this.createThumbnailGenerationCommand(thumbnails, removeAlpha, content, contentMarker);
        String response = this.commandExecutor.execute(command, false, message -> new MediaExtractionException("Could not analyze content and generate thumbnails: " + message));
        ImageMetadata result = this.parseCommandResponse(response, contentMarker);
        for (ThumbnailWithSize thumbnail : thumbnails) {
            try {
                boolean shouldUseOriginal;
                Path tempFileForThumbnail = thumbnail.getTempFileForThumbnail();
                if (this.getFileSize(tempFileForThumbnail) == 0L) {
                    throw new MediaExtractionException("Thumbnail file empty: " + tempFileForThumbnail);
                }
                boolean bl = shouldUseOriginal = result.getWidth() < thumbnail.getImageSize();
                if (shouldUseOriginal) {
                    this.copyFile(content, thumbnail);
                    continue;
                }
                this.copyFile(thumbnail.getTempFileForThumbnail(), thumbnail);
            }
            catch (IOException e) {
                throw new MediaExtractionException("Could not access thumbnail file", e);
            }
        }
        return result;
    }

    long getFileSize(Path file) throws IOException {
        return Files.size(file);
    }

    void copyFile(Path source, ThumbnailWithSize destination) throws IOException {
        try (InputStream thumbnailStream = Files.newInputStream(source, new OpenOption[0]);){
            destination.getThumbnail().markAsWithContent(thumbnailStream);
        }
    }

    void copyFile(File source, ThumbnailWithSize destination) throws IOException {
        this.copyFile(source.toPath(), destination);
    }

    List<ThumbnailWithSize> prepareThumbnailFiles(String url, String detectedMimeType) throws MediaExtractionException {
        String thumbnailMimeType;
        String imageMagickThumbnailTypePrefix;
        if (PNG_MIME_TYPE.equals(detectedMimeType)) {
            imageMagickThumbnailTypePrefix = "png:";
            thumbnailMimeType = PNG_MIME_TYPE;
        } else {
            imageMagickThumbnailTypePrefix = "jpeg:";
            thumbnailMimeType = JPEG_MIME_TYPE;
        }
        String md5 = ThumbnailGenerator.md5Hex(url);
        ArrayList<ThumbnailWithSize> result = new ArrayList<ThumbnailWithSize>(ThumbnailKind.values().length);
        try {
            for (ThumbnailKind thumbnailKind : ThumbnailKind.values()) {
                String targetName = md5 + thumbnailKind.getNameSuffix();
                ThumbnailImpl thumbnail = new ThumbnailImpl(url, thumbnailMimeType, targetName);
                result.add(new ThumbnailWithSize(thumbnail, thumbnailKind.getImageSize(), imageMagickThumbnailTypePrefix));
            }
        }
        catch (IOException | RuntimeException e) {
            ThumbnailGenerator.closeAllThumbnailsSilently(result);
            throw new MediaExtractionException("Could not create temporary thumbnail files.", e);
        }
        return result;
    }

    private static String md5Hex(String s2) throws MediaExtractionException {
        try {
            byte[] bytes = s2.getBytes(StandardCharsets.UTF_8.name());
            byte[] md5bytes = MessageDigest.getInstance("MD5").digest(bytes);
            return String.format("%032x", new BigInteger(1, md5bytes));
        }
        catch (UnsupportedEncodingException | NoSuchAlgorithmException e) {
            throw new MediaExtractionException("Could not compute md5 hash", e);
        }
    }

    ImageMetadata parseCommandResponse(String response, String contentMarker) throws MediaExtractionException {
        try {
            String[] segments = response.split(Pattern.quote(contentMarker), 6);
            if (segments.length < 5) {
                throw new MediaExtractionException(String.format("Could not parse ImageMagick response(there are not enough content markers):%s%s", System.lineSeparator(), response));
            }
            if (segments.length > 5) {
                throw new MediaExtractionException(String.format("Could not parse ImageMagick response(there are too many content markers):%s%s", System.lineSeparator(), response));
            }
            String unexpectedContent = IntStream.range(0, segments.length).filter(index -> index % 2 == 0).mapToObj(index -> segments[index]).collect(Collectors.joining(System.lineSeparator()));
            if (StringUtils.isNotBlank(unexpectedContent)) {
                throw new MediaExtractionException(String.format("Unexpected content found in ImageMagick response: %s%s", System.lineSeparator(), unexpectedContent.trim()));
            }
            Pattern pattern = Pattern.compile("#([0-9A-F]{6})");
            List colorStrings = ThumbnailGenerator.splitByNewLine(segments[3]).stream().sorted(Collections.reverseOrder()).limit(6L).collect(Collectors.toList());
            Supplier<Stream> streamMatcherSupplier = () -> colorStrings.stream().map(pattern::matcher);
            if (!streamMatcherSupplier.get().allMatch(Matcher::find)) {
                throw new IllegalStateException("Invalid color line found.");
            }
            List<String> dominantColors = streamMatcherSupplier.get().filter(Matcher::find).map(matcher -> matcher.group(1)).collect(Collectors.toList());
            List<String> metadata = ThumbnailGenerator.splitByNewLine(segments[1]);
            int width = Integer.parseInt(metadata.get(0));
            int height = Integer.parseInt(metadata.get(1));
            String colorSpace = metadata.get(2);
            return new ImageMetadata(width, height, colorSpace, dominantColors);
        }
        catch (RuntimeException e) {
            throw new MediaExtractionException(String.format("Could not parse ImageMagick response:%s%s", System.lineSeparator(), response), e);
        }
    }

    static class ThumbnailWithSize {
        private final ThumbnailImpl thumbnail;
        private final int imageSize;
        private final Path tempFileForThumbnail;
        private final String imageMagickTypePrefix;

        ThumbnailWithSize(ThumbnailImpl thumbnail, int imageSize, Path tempFileForThumbnail, String imageMagickTypePrefix) {
            this.thumbnail = thumbnail;
            this.imageSize = imageSize;
            this.tempFileForThumbnail = tempFileForThumbnail;
            this.imageMagickTypePrefix = imageMagickTypePrefix;
        }

        ThumbnailWithSize(ThumbnailImpl thumbnail, int imageSize, String imageMagickTypePrefix) throws IOException {
            this(thumbnail, imageSize, Files.createTempFile("thumbnail_", null, new FileAttribute[0]), imageMagickTypePrefix);
        }

        ThumbnailImpl getThumbnail() {
            return this.thumbnail;
        }

        int getImageSize() {
            return this.imageSize;
        }

        Path getTempFileForThumbnail() {
            return this.tempFileForThumbnail;
        }

        String getImageMagickTypePrefix() {
            return this.imageMagickTypePrefix;
        }

        void deleteTempFileSilently() {
            try {
                Files.delete(this.getTempFileForThumbnail());
            }
            catch (IOException e) {
                LOGGER.warn("Could not close thumbnail: {}", (Object)this.getTempFileForThumbnail(), (Object)e);
            }
        }
    }
}

