/*
 * Decompiled with CFR 0.152.
 */
package eu.europeana.metis.mediaprocessing.extraction;

import eu.europeana.metis.mediaprocessing.exception.CommandExecutionException;
import eu.europeana.metis.mediaprocessing.exception.MediaExtractionException;
import eu.europeana.metis.mediaprocessing.exception.MediaProcessorException;
import eu.europeana.metis.mediaprocessing.extraction.CommandExecutor;
import eu.europeana.metis.mediaprocessing.extraction.ImageMetadata;
import eu.europeana.metis.mediaprocessing.model.Thumbnail;
import eu.europeana.metis.mediaprocessing.model.ThumbnailImpl;
import eu.europeana.metis.utils.MediaType;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ThumbnailGenerator {
    private static final Logger LOGGER = LoggerFactory.getLogger(ThumbnailGenerator.class);
    private static final String PDF_MIME_TYPE = "application/pdf";
    private static final String PNG_MIME_TYPE = "image/png";
    private static final String JPEG_MIME_TYPE = "image/jpeg";
    private static final String COMMAND_RESULT_FORMAT = "%w\n%h\n%[colorspace]\n";
    private static final int COMMAND_RESULT_WIDTH_LINE = 0;
    private static final int COMMAND_RESULT_HEIGHT_LINE = 1;
    private static final int COMMAND_RESULT_COLORSPACE_LINE = 2;
    private static final int COMMAND_RESULT_COLORS_LINE = 3;
    private static final int COMMAND_RESULT_MAX_COLORS = 6;
    private static String globalMagickCommand;
    private static Path globalColormapFile;
    private final String magickCmd;
    private final String colormapFile;
    private final CommandExecutor commandExecutor;

    ThumbnailGenerator(CommandExecutor commandExecutor) throws MediaProcessorException {
        this(commandExecutor, ThumbnailGenerator.getGlobalImageMagickCommand(commandExecutor), ThumbnailGenerator.initColorMap().toString());
    }

    ThumbnailGenerator(CommandExecutor commandExecutor, String magickCommand, String colorMapFile) {
        this.commandExecutor = commandExecutor;
        this.magickCmd = magickCommand;
        this.colormapFile = colorMapFile;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Path initColorMap() throws MediaProcessorException {
        Class<ThumbnailGenerator> clazz = ThumbnailGenerator.class;
        synchronized (ThumbnailGenerator.class) {
            Path colormapTempFile;
            if (globalColormapFile != null) {
                // ** MonitorExit[var0] (shouldn't be in output)
                return globalColormapFile;
            }
            try (InputStream colorMapInputStream = Thread.currentThread().getContextClassLoader().getResourceAsStream("colormap.png");){
                colormapTempFile = Files.createTempFile("colormap", ".png", new FileAttribute[0]);
                Files.copy(colorMapInputStream, colormapTempFile, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException e) {
                LOGGER.warn("Could not load color map file: {}.", (Object)"colormap.png", (Object)e);
                throw new MediaProcessorException("Could not load color map file.", e);
            }
            colormapTempFile.toFile().deleteOnExit();
            globalColormapFile = colormapTempFile;
            // ** MonitorExit[var0] (shouldn't be in output)
            return globalColormapFile;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static String getGlobalImageMagickCommand(CommandExecutor commandExecutor) throws MediaProcessorException {
        Class<ThumbnailGenerator> clazz = ThumbnailGenerator.class;
        synchronized (ThumbnailGenerator.class) {
            if (globalMagickCommand == null) {
                globalMagickCommand = ThumbnailGenerator.discoverImageMagickCommand(commandExecutor);
            }
            // ** MonitorExit[var1_1] (shouldn't be in output)
            return globalMagickCommand;
        }
    }

    static String discoverImageMagickCommand(CommandExecutor commandExecutor) throws MediaProcessorException {
        List<Object> paths;
        try {
            List<String> lines = commandExecutor.execute(Arrays.asList("magick", "-version"), true);
            if (String.join((CharSequence)"", lines).startsWith("Version: ImageMagick 7")) {
                String result2 = "magick";
                LOGGER.info("Found ImageMagic 7. Command: {}", (Object)"magick");
                return "magick";
            }
        }
        catch (CommandExecutionException e) {
            LOGGER.info("Could not find ImageMagick 7 because of: {}.", (Object)e.getMessage());
            LOGGER.debug("Could not find ImageMagick 7 due to following problem.", e);
        }
        boolean isWindows = System.getProperty("os.name").toLowerCase(Locale.ENGLISH).contains("win");
        try {
            paths = commandExecutor.execute(Arrays.asList(isWindows ? "where" : "which", "convert"), true);
        }
        catch (CommandExecutionException e) {
            LOGGER.warn("Could not find ImageMagick 6 due to following problem.", e);
            paths = Collections.emptyList();
        }
        for (String path : paths) {
            try {
                List<String> lines = commandExecutor.execute(Arrays.asList(path, "-version"), true);
                if (!String.join((CharSequence)"", lines).startsWith("Version: ImageMagick 6")) continue;
                LOGGER.info("Found ImageMagic 6. Command: {}", (Object)path);
                return path;
            }
            catch (CommandExecutionException e) {
                LOGGER.info("Could not find ImageMagick 6 at path {} because of: {}.", (Object)path, (Object)e.getMessage());
                LOGGER.debug("Could not find ImageMagick 6 at path {} due to following problem.", (Object)path, (Object)e);
            }
        }
        LOGGER.error("Could not find ImageMagick 6 or 7. See previous log statements for details.");
        throw new MediaProcessorException("Could not find ImageMagick 6 or 7.");
    }

    Pair<ImageMetadata, List<Thumbnail>> generateThumbnails(String url, String detectedMimeType, File content) throws MediaExtractionException {
        ImageMetadata image;
        if (content == null) {
            throw new MediaExtractionException("File content is null");
        }
        if (detectedMimeType.startsWith("image/vnd.djvu") || detectedMimeType.startsWith("image/x-djvu") || detectedMimeType.startsWith("image/x.djvu")) {
            throw new MediaExtractionException("Cannot generate thumbnails for DjVu file.");
        }
        List<ThumbnailWithSize> thumbnails = this.prepareThumbnailFiles(url, detectedMimeType);
        try {
            image = this.generateThumbnailsInternal(thumbnails, detectedMimeType, content);
        }
        catch (RuntimeException e) {
            ThumbnailGenerator.closeAllThumbnailsSilently(thumbnails);
            throw new MediaExtractionException("Unexpected error during processing", e);
        }
        catch (MediaExtractionException e) {
            ThumbnailGenerator.closeAllThumbnailsSilently(thumbnails);
            throw e;
        }
        finally {
            thumbnails.forEach(ThumbnailWithSize::deleteTempFileSilently);
        }
        List resultThumbnails = thumbnails.stream().map(ThumbnailWithSize::getThumbnail).collect(Collectors.toList());
        return new ImmutablePair<ImageMetadata, List<Thumbnail>>(image, resultThumbnails);
    }

    private static void closeAllThumbnailsSilently(List<ThumbnailWithSize> thumbnails) {
        for (ThumbnailWithSize thumbnail : thumbnails) {
            thumbnail.getThumbnail().close();
        }
    }

    List<String> createThumbnailGenerationCommand(List<ThumbnailWithSize> thumbnails, String detectedMimeType, File content) {
        ArrayList<String> command = new ArrayList<String>(Arrays.asList(this.magickCmd, content.getPath() + "[0]", "-format", COMMAND_RESULT_FORMAT, "-write", "info:"));
        if (PDF_MIME_TYPE.equals(detectedMimeType)) {
            command.addAll(Arrays.asList("-background", "white", "-alpha", "remove"));
        }
        int thumbnailCounter = thumbnails.size();
        for (int i = 0; i < thumbnailCounter; ++i) {
            if (i != thumbnailCounter - 1) {
                command.add("(");
                command.add("+clone");
            }
            ThumbnailWithSize thumbnail = thumbnails.get(i);
            command.addAll(Arrays.asList("-thumbnail", thumbnail.getImageSize() + "x", "-write", thumbnail.getImageMagickTypePrefix() + thumbnail.getTempFileForThumbnail().toString()));
            if (i == thumbnailCounter - 1) continue;
            command.add("+delete");
            command.add(")");
        }
        command.addAll(Arrays.asList("-colorspace", "sRGB", "-dither", "Riemersma", "-remap", this.colormapFile, "-format", "\n%c", "histogram:info:"));
        return command;
    }

    private ImageMetadata generateThumbnailsInternal(List<ThumbnailWithSize> thumbnails, String detectedMimeType, File content) throws MediaExtractionException {
        List<String> response;
        try {
            response = this.commandExecutor.execute(this.createThumbnailGenerationCommand(thumbnails, detectedMimeType, content), false);
        }
        catch (CommandExecutionException e) {
            throw new MediaExtractionException("Could not analyze content and generate thumbnails.", e);
        }
        ImageMetadata result2 = this.parseCommandResponse(response);
        for (ThumbnailWithSize thumbnail : thumbnails) {
            try {
                boolean shouldUseOriginal;
                Path tempFileForThumbnail = thumbnail.getTempFileForThumbnail();
                if (this.getFileSize(tempFileForThumbnail) == 0L) {
                    throw new MediaExtractionException("Thumbnail file empty: " + tempFileForThumbnail);
                }
                boolean isImage = MediaType.getMediaType(detectedMimeType) == MediaType.IMAGE;
                boolean bl = shouldUseOriginal = isImage && result2.getWidth() < thumbnail.getImageSize();
                if (shouldUseOriginal) {
                    this.copyFile(content, thumbnail);
                    continue;
                }
                this.copyFile(thumbnail.getTempFileForThumbnail(), thumbnail);
            }
            catch (IOException e) {
                throw new MediaExtractionException("Could not access thumbnail file", e);
            }
        }
        return result2;
    }

    long getFileSize(Path file) throws IOException {
        return Files.size(file);
    }

    void copyFile(Path source, ThumbnailWithSize destination) throws IOException {
        try (InputStream thumbnailStream = Files.newInputStream(source, new OpenOption[0]);){
            destination.getThumbnail().setContent(thumbnailStream);
        }
    }

    void copyFile(File source, ThumbnailWithSize destination) throws IOException {
        this.copyFile(source.toPath(), destination);
    }

    List<ThumbnailWithSize> prepareThumbnailFiles(String url, String detectedMimeType) throws MediaExtractionException {
        String thumbnailMimeType;
        String imageMagickThumbnailTypePrefix;
        if (PDF_MIME_TYPE.equals(detectedMimeType) || PNG_MIME_TYPE.equals(detectedMimeType)) {
            imageMagickThumbnailTypePrefix = "png:";
            thumbnailMimeType = PNG_MIME_TYPE;
        } else {
            imageMagickThumbnailTypePrefix = "jpeg:";
            thumbnailMimeType = JPEG_MIME_TYPE;
        }
        String md5 = ThumbnailGenerator.md5Hex(url);
        ArrayList<ThumbnailWithSize> result2 = new ArrayList<ThumbnailWithSize>(ThumbnailKind.values().length);
        try {
            for (ThumbnailKind thumbnailKind : ThumbnailKind.values()) {
                String targetName = md5 + thumbnailKind.suffix;
                ThumbnailImpl thumbnail = new ThumbnailImpl(url, thumbnailMimeType, targetName);
                result2.add(new ThumbnailWithSize(thumbnail, thumbnailKind.size, imageMagickThumbnailTypePrefix));
            }
        }
        catch (IOException e) {
            throw new MediaExtractionException("Could not create temporary thumbnail files.", e);
        }
        return result2;
    }

    private static String md5Hex(String s) throws MediaExtractionException {
        try {
            byte[] bytes = s.getBytes(StandardCharsets.UTF_8.name());
            byte[] md5bytes = MessageDigest.getInstance("MD5").digest(bytes);
            return String.format("%032x", new BigInteger(1, md5bytes));
        }
        catch (UnsupportedEncodingException | NoSuchAlgorithmException e) {
            throw new MediaExtractionException("Could not compute md5 hash", e);
        }
    }

    ImageMetadata parseCommandResponse(List<String> response) throws MediaExtractionException {
        try {
            Pattern pattern = Pattern.compile("#([0-9A-F]{6})");
            List colorStrings = response.stream().skip(3L).sorted(Collections.reverseOrder()).limit(6L).collect(Collectors.toList());
            Supplier<Stream> streamMatcherSupplier = () -> colorStrings.stream().map(pattern::matcher);
            if (!streamMatcherSupplier.get().allMatch(Matcher::find)) {
                throw new IllegalStateException("Invalid color line found.");
            }
            List<String> dominantColors = streamMatcherSupplier.get().filter(Matcher::find).map(matcher -> matcher.group(1)).collect(Collectors.toList());
            int width = Integer.parseInt(response.get(0));
            int height = Integer.parseInt(response.get(1));
            String colorSpace = response.get(2);
            return new ImageMetadata(width, height, colorSpace, dominantColors);
        }
        catch (RuntimeException e) {
            LOGGER.info("Could not parse ImageMagick response:\n" + StringUtils.join(response, "\n"), e);
            throw new MediaExtractionException("File seems to be corrupted", e);
        }
    }

    static class ThumbnailWithSize {
        private final ThumbnailImpl thumbnail;
        private final int imageSize;
        private final Path tempFileForThumbnail;
        private final String imageMagickTypePrefix;

        ThumbnailWithSize(ThumbnailImpl thumbnail, int imageSize, Path tempFileForThumbnail, String imageMagickTypePrefix) {
            this.thumbnail = thumbnail;
            this.imageSize = imageSize;
            this.tempFileForThumbnail = tempFileForThumbnail;
            this.imageMagickTypePrefix = imageMagickTypePrefix;
        }

        ThumbnailWithSize(ThumbnailImpl thumbnail, int imageSize, String imageMagickTypePrefix) throws IOException {
            this(thumbnail, imageSize, Files.createTempFile("thumbnail_", null, new FileAttribute[0]), imageMagickTypePrefix);
        }

        ThumbnailImpl getThumbnail() {
            return this.thumbnail;
        }

        int getImageSize() {
            return this.imageSize;
        }

        Path getTempFileForThumbnail() {
            return this.tempFileForThumbnail;
        }

        String getImageMagickTypePrefix() {
            return this.imageMagickTypePrefix;
        }

        void deleteTempFileSilently() {
            try {
                Files.delete(this.getTempFileForThumbnail());
            }
            catch (IOException e) {
                LOGGER.warn("Could not close thumbnail: {}", (Object)this.getTempFileForThumbnail(), (Object)e);
            }
        }
    }

    private static enum ThumbnailKind {
        MEDIUM(200, "-MEDIUM"),
        LARGE(400, "-LARGE");

        protected final int size;
        protected final String suffix;

        private ThumbnailKind(int size, String suffix) {
            this.size = size;
            this.suffix = suffix;
        }
    }
}

