/*
 * Copyright 2007-2019 The Europeana Foundation
 *
 *  Licenced under the EUPL, Version 1.1 (the "Licence") and subsequent versions as approved
 *  by the European Commission;
 *  You may not use this work except in compliance with the Licence.
 *
 *  You may obtain a copy of the Licence at:
 *  http://joinup.ec.europa.eu/software/page/eupl
 *
 *  Unless required by applicable law or agreed to in writing, software distributed under
 *  the Licence is distributed on an "AS IS" basis, without warranties or conditions of
 *  any kind, either express or implied.
 *  See the Licence for the specific language governing permissions and limitations under
 *  the Licence.
 */

package eu.europeana.api2.v2.utils.technicalfacets;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * Generates the common pure tags (mimetype and mediatype)
 */
public class CommonTagExtractor {

    private static final Logger LOG = LogManager.getLogger(CommonTagExtractor.class.getName());

    public static final BiMap<String, Integer> mimeTypes = HashBiMap.create(776);

    static {
        mimeTypes.put("", 0);
        mimeTypes.put("application/andrew-inset",1);
        mimeTypes.put("application/applixware",2);
        mimeTypes.put("application/atom+xml",3);
        mimeTypes.put("application/atomcat+xml",4);
        mimeTypes.put("application/atomsvc+xml",5);
        mimeTypes.put("application/ccxml+xml",6);
        mimeTypes.put("application/cdmi-capability",7);
        mimeTypes.put("application/cdmi-container",8);
        mimeTypes.put("application/cdmi-domain",9);
        mimeTypes.put("application/cdmi-object",10);
        mimeTypes.put("application/cdmi-queue",11);
        mimeTypes.put("application/cu-seeme",12);
        mimeTypes.put("application/davmount+xml",13);
        mimeTypes.put("application/docbook+xml",14);
        mimeTypes.put("application/dssc+der",15);
        mimeTypes.put("application/dssc+xml",16);
        mimeTypes.put("application/ecmascript",17);
        mimeTypes.put("application/emma+xml",18);
        mimeTypes.put("application/epub+zip",19);
        mimeTypes.put("application/exi",20);
        mimeTypes.put("application/font-tdpfr",21);
        mimeTypes.put("application/gml+xml",22);
        mimeTypes.put("application/gpx+xml",23);
        mimeTypes.put("application/gxf",24);
        mimeTypes.put("application/hyperstudio",25);
        mimeTypes.put("application/inkml+xml",26);
        mimeTypes.put("application/ipfix",27);
        mimeTypes.put("application/java-archive",28);
        mimeTypes.put("application/java-serialized-object",29);
        mimeTypes.put("application/java-vm",30);
        mimeTypes.put("application/javascript",31);
        mimeTypes.put("application/json",32);
        mimeTypes.put("application/jsonml+json",33);
        mimeTypes.put("application/lost+xml",34);
        mimeTypes.put("application/mac-binhex40",35);
        mimeTypes.put("application/mac-compactpro",36);
        mimeTypes.put("application/mads+xml",37);
        mimeTypes.put("application/marc",38);
        mimeTypes.put("application/marcxml+xml",39);
        mimeTypes.put("application/mathematica",40);
        mimeTypes.put("application/mathml+xml",41);
        mimeTypes.put("application/mbox",42);
        mimeTypes.put("application/mediaservercontrol+xml",43);
        mimeTypes.put("application/metalink+xml",44);
        mimeTypes.put("application/metalink4+xml",45);
        mimeTypes.put("application/mets+xml",46);
        mimeTypes.put("application/mods+xml",47);
        mimeTypes.put("application/mp21",48);
        mimeTypes.put("application/mp4",49);
        mimeTypes.put("application/msword",50);
        mimeTypes.put("application/mxf",51);
        mimeTypes.put("application/octet-stream",52);
        mimeTypes.put("application/oda",53);
        mimeTypes.put("application/oebps-package+xml",54);
        mimeTypes.put("application/ogg",55);
        mimeTypes.put("application/omdoc+xml",56);
        mimeTypes.put("application/onenote",57);
        mimeTypes.put("application/oxps",58);
        mimeTypes.put("application/patch-ops-error+xml",59);
        mimeTypes.put("application/pdf",60);
        mimeTypes.put("application/pgp-encrypted",61);
        mimeTypes.put("application/pgp-signature",62);
        mimeTypes.put("application/pics-rules",63);
        mimeTypes.put("application/pkcs10",64);
        mimeTypes.put("application/pkcs7-mime",65);
        mimeTypes.put("application/pkcs7-signature",66);
        mimeTypes.put("application/pkcs8",67);
        mimeTypes.put("application/pkix-attr-cert",68);
        mimeTypes.put("application/pkix-cert",69);
        mimeTypes.put("application/pkix-crl",70);
        mimeTypes.put("application/pkix-pkipath",71);
        mimeTypes.put("application/pkixcmp",72);
        mimeTypes.put("application/pls+xml",73);
        mimeTypes.put("application/postscript",74);
        mimeTypes.put("application/prs.cww",75);
        mimeTypes.put("application/pskc+xml",76);
        mimeTypes.put("application/rdf+xml",77);
        mimeTypes.put("application/reginfo+xml",78);
        mimeTypes.put("application/relax-ng-compact-syntax",79);
        mimeTypes.put("application/resource-lists+xml",80);
        mimeTypes.put("application/resource-lists-diff+xml",81);
        mimeTypes.put("application/rls-services+xml",82);
        mimeTypes.put("application/rpki-ghostbusters",83);
        mimeTypes.put("application/rpki-manifest",84);
        mimeTypes.put("application/rpki-roa",85);
        mimeTypes.put("application/rsd+xml",86);
        mimeTypes.put("application/rss+xml",87);
        mimeTypes.put("application/rtf",88);
        mimeTypes.put("application/sbml+xml",89);
        mimeTypes.put("application/scvp-cv-request",90);
        mimeTypes.put("application/scvp-cv-response",91);
        mimeTypes.put("application/scvp-vp-request",92);
        mimeTypes.put("application/scvp-vp-response",93);
        mimeTypes.put("application/sdp",94);
        mimeTypes.put("application/set-payment-initiation",95);
        mimeTypes.put("application/set-registration-initiation",96);
        mimeTypes.put("application/shf+xml",97);
        mimeTypes.put("application/smil+xml",98);
        mimeTypes.put("application/sparql-query",99);
        mimeTypes.put("application/sparql-results+xml",100);
        mimeTypes.put("application/srgs",101);
        mimeTypes.put("application/srgs+xml",102);
        mimeTypes.put("application/sru+xml",103);
        mimeTypes.put("application/ssdl+xml",104);
        mimeTypes.put("application/ssml+xml",105);
        mimeTypes.put("application/tei+xml",106);
        mimeTypes.put("application/thraud+xml",107);
        mimeTypes.put("application/timestamped-data",108);
        mimeTypes.put("application/vnd.3gpp.pic-bw-large",109);
        mimeTypes.put("application/vnd.3gpp.pic-bw-small",110);
        mimeTypes.put("application/vnd.3gpp.pic-bw-var",111);
        mimeTypes.put("application/vnd.3gpp2.tcap",112);
        mimeTypes.put("application/vnd.3m.post-it-notes",113);
        mimeTypes.put("application/vnd.accpac.simply.aso",114);
        mimeTypes.put("application/vnd.accpac.simply.imp",115);
        mimeTypes.put("application/vnd.acucobol",116);
        mimeTypes.put("application/vnd.acucorp",117);
        mimeTypes.put("application/vnd.adobe.air-application-installer-package+zip",118);
        mimeTypes.put("application/vnd.adobe.formscentral.fcdt",119);
        mimeTypes.put("application/vnd.adobe.fxp",120);
        mimeTypes.put("application/vnd.adobe.xdp+xml",121);
        mimeTypes.put("application/vnd.adobe.xfdf",122);
        mimeTypes.put("application/vnd.ahead.space",123);
        mimeTypes.put("application/vnd.airzip.filesecure.azf",124);
        mimeTypes.put("application/vnd.airzip.filesecure.azs",125);
        mimeTypes.put("application/vnd.amazon.ebook",126);
        mimeTypes.put("application/vnd.americandynamics.acc",127);
        mimeTypes.put("application/vnd.amiga.ami",128);
        mimeTypes.put("application/vnd.android.package-archive",129);
        mimeTypes.put("application/vnd.anser-web-certificate-issue-initiation",130);
        mimeTypes.put("application/vnd.anser-web-funds-transfer-initiation",131);
        mimeTypes.put("application/vnd.antix.game-component",132);
        mimeTypes.put("application/vnd.apple.installer+xml",133);
        mimeTypes.put("application/vnd.apple.mpegurl",134);
        mimeTypes.put("application/vnd.aristanetworks.swi",135);
        mimeTypes.put("application/vnd.astraea-software.iota",136);
        mimeTypes.put("application/vnd.audiograph",137);
        mimeTypes.put("application/vnd.blueice.multipass",138);
        mimeTypes.put("application/vnd.bmi",139);
        mimeTypes.put("application/vnd.businessobjects",140);
        mimeTypes.put("application/vnd.chemdraw+xml",141);
        mimeTypes.put("application/vnd.chipnuts.karaoke-mmd",142);
        mimeTypes.put("application/vnd.cinderella",143);
        mimeTypes.put("application/vnd.claymore",144);
        mimeTypes.put("application/vnd.cloanto.rp9",145);
        mimeTypes.put("application/vnd.clonk.c4group",146);
        mimeTypes.put("application/vnd.cluetrust.cartomobile-config",147);
        mimeTypes.put("application/vnd.cluetrust.cartomobile-config-pkg",148);
        mimeTypes.put("application/vnd.commonspace",149);
        mimeTypes.put("application/vnd.contact.cmsg",150);
        mimeTypes.put("application/vnd.cosmocaller",151);
        mimeTypes.put("application/vnd.crick.clicker",152);
        mimeTypes.put("application/vnd.crick.clicker.keyboard",153);
        mimeTypes.put("application/vnd.crick.clicker.palette",154);
        mimeTypes.put("application/vnd.crick.clicker.template",155);
        mimeTypes.put("application/vnd.crick.clicker.wordbank",156);
        mimeTypes.put("application/vnd.criticaltools.wbs+xml",157);
        mimeTypes.put("application/vnd.ctc-posml",158);
        mimeTypes.put("application/vnd.cups-ppd",159);
        mimeTypes.put("application/vnd.curl.car",160);
        mimeTypes.put("application/vnd.curl.pcurl",161);
        mimeTypes.put("application/vnd.dart",162);
        mimeTypes.put("application/vnd.data-vision.rdz",163);
        mimeTypes.put("application/vnd.dece.data",164);
        mimeTypes.put("application/vnd.dece.ttml+xml",165);
        mimeTypes.put("application/vnd.dece.unspecified",166);
        mimeTypes.put("application/vnd.dece.zip",167);
        mimeTypes.put("application/vnd.denovo.fcselayout-link",168);
        mimeTypes.put("application/vnd.dna",169);
        mimeTypes.put("application/vnd.dolby.mlp",170);
        mimeTypes.put("application/vnd.dpgraph",171);
        mimeTypes.put("application/vnd.dreamfactory",172);
        mimeTypes.put("application/vnd.ds-keypoint",173);
        mimeTypes.put("application/vnd.dvb.ait",174);
        mimeTypes.put("application/vnd.dvb.service",175);
        mimeTypes.put("application/vnd.dynageo",176);
        mimeTypes.put("application/vnd.ecowin.chart",177);
        mimeTypes.put("application/vnd.enliven",178);
        mimeTypes.put("application/vnd.epson.esf",179);
        mimeTypes.put("application/vnd.epson.msf",180);
        mimeTypes.put("application/vnd.epson.quickanime",181);
        mimeTypes.put("application/vnd.epson.salt",182);
        mimeTypes.put("application/vnd.epson.ssf",183);
        mimeTypes.put("application/vnd.eszigno3+xml",184);
        mimeTypes.put("application/vnd.ezpix-album",185);
        mimeTypes.put("application/vnd.ezpix-package",186);
        mimeTypes.put("application/vnd.fdf",187);
        mimeTypes.put("application/vnd.fdsn.mseed",188);
        mimeTypes.put("application/vnd.fdsn.seed",189);
        mimeTypes.put("application/vnd.flographit",190);
        mimeTypes.put("application/vnd.fluxtime.clip",191);
        mimeTypes.put("application/vnd.framemaker",192);
        mimeTypes.put("application/vnd.frogans.fnc",193);
        mimeTypes.put("application/vnd.frogans.ltf",194);
        mimeTypes.put("application/vnd.fsc.weblaunch",195);
        mimeTypes.put("application/vnd.fujitsu.oasys",196);
        mimeTypes.put("application/vnd.fujitsu.oasys2",197);
        mimeTypes.put("application/vnd.fujitsu.oasys3",198);
        mimeTypes.put("application/vnd.fujitsu.oasysgp",199);
        mimeTypes.put("application/vnd.fujitsu.oasysprs",200);
        mimeTypes.put("application/vnd.fujixerox.ddd",201);
        mimeTypes.put("application/vnd.fujixerox.docuworks",202);
        mimeTypes.put("application/vnd.fujixerox.docuworks.binder",203);
        mimeTypes.put("application/vnd.fuzzysheet",204);
        mimeTypes.put("application/vnd.genomatix.tuxedo",205);
        mimeTypes.put("application/vnd.geogebra.file",206);
        mimeTypes.put("application/vnd.geogebra.tool",207);
        mimeTypes.put("application/vnd.geometry-explorer",208);
        mimeTypes.put("application/vnd.geonext",209);
        mimeTypes.put("application/vnd.geoplan",210);
        mimeTypes.put("application/vnd.geospace",211);
        mimeTypes.put("application/vnd.gmx",212);
        mimeTypes.put("application/vnd.google-earth.kml+xml",213);
        mimeTypes.put("application/vnd.google-earth.kmz",214);
        mimeTypes.put("application/vnd.grafeq",215);
        mimeTypes.put("application/vnd.groove-account",216);
        mimeTypes.put("application/vnd.groove-help",217);
        mimeTypes.put("application/vnd.groove-identity-message",218);
        mimeTypes.put("application/vnd.groove-injector",219);
        mimeTypes.put("application/vnd.groove-tool-message",220);
        mimeTypes.put("application/vnd.groove-tool-template",221);
        mimeTypes.put("application/vnd.groove-vcard",222);
        mimeTypes.put("application/vnd.hal+xml",223);
        mimeTypes.put("application/vnd.handheld-entertainment+xml",224);
        mimeTypes.put("application/vnd.hbci",225);
        mimeTypes.put("application/vnd.hhe.lesson-player",226);
        mimeTypes.put("application/vnd.hp-hpgl",227);
        mimeTypes.put("application/vnd.hp-hpid",228);
        mimeTypes.put("application/vnd.hp-hps",229);
        mimeTypes.put("application/vnd.hp-jlyt",230);
        mimeTypes.put("application/vnd.hp-pcl",231);
        mimeTypes.put("application/vnd.hp-pclxl",232);
        mimeTypes.put("application/vnd.hydrostatix.sof-data",233);
        mimeTypes.put("application/vnd.ibm.minipay",234);
        mimeTypes.put("application/vnd.ibm.modcap",235);
        mimeTypes.put("application/vnd.ibm.rights-management",236);
        mimeTypes.put("application/vnd.ibm.secure-container",237);
        mimeTypes.put("application/vnd.iccprofile",238);
        mimeTypes.put("application/vnd.igloader",239);
        mimeTypes.put("application/vnd.immervision-ivp",240);
        mimeTypes.put("application/vnd.immervision-ivu",241);
        mimeTypes.put("application/vnd.insors.igm",242);
        mimeTypes.put("application/vnd.intercon.formnet",243);
        mimeTypes.put("application/vnd.intergeo",244);
        mimeTypes.put("application/vnd.intu.qbo",245);
        mimeTypes.put("application/vnd.intu.qfx",246);
        mimeTypes.put("application/vnd.ipunplugged.rcprofile",247);
        mimeTypes.put("application/vnd.irepository.package+xml",248);
        mimeTypes.put("application/vnd.is-xpr",249);
        mimeTypes.put("application/vnd.isac.fcs",250);
        mimeTypes.put("application/vnd.jam",251);
        mimeTypes.put("application/vnd.jcp.javame.midlet-rms",252);
        mimeTypes.put("application/vnd.jisp",253);
        mimeTypes.put("application/vnd.joost.joda-archive",254);
        mimeTypes.put("application/vnd.kahootz",255);
        mimeTypes.put("application/vnd.kde.karbon",256);
        mimeTypes.put("application/vnd.kde.kchart",257);
        mimeTypes.put("application/vnd.kde.kformula",258);
        mimeTypes.put("application/vnd.kde.kivio",259);
        mimeTypes.put("application/vnd.kde.kontour",260);
        mimeTypes.put("application/vnd.kde.kpresenter",261);
        mimeTypes.put("application/vnd.kde.kspread",262);
        mimeTypes.put("application/vnd.kde.kword",263);
        mimeTypes.put("application/vnd.kenameaapp",264);
        mimeTypes.put("application/vnd.kidspiration",265);
        mimeTypes.put("application/vnd.kinar",266);
        mimeTypes.put("application/vnd.koan",267);
        mimeTypes.put("application/vnd.kodak-descriptor",268);
        mimeTypes.put("application/vnd.las.las+xml",269);
        mimeTypes.put("application/vnd.llamagraphics.life-balance.desktop",270);
        mimeTypes.put("application/vnd.llamagraphics.life-balance.exchange+xml",271);
        mimeTypes.put("application/vnd.lotus-1-2-3",272);
        mimeTypes.put("application/vnd.lotus-approach",273);
        mimeTypes.put("application/vnd.lotus-freelance",274);
        mimeTypes.put("application/vnd.lotus-notes",275);
        mimeTypes.put("application/vnd.lotus-organizer",276);
        mimeTypes.put("application/vnd.lotus-screencam",277);
        mimeTypes.put("application/vnd.lotus-wordpro",278);
        mimeTypes.put("application/vnd.macports.portpkg",279);
        mimeTypes.put("application/vnd.mcd",280);
        mimeTypes.put("application/vnd.medcalcdata",281);
        mimeTypes.put("application/vnd.mediastation.cdkey",282);
        mimeTypes.put("application/vnd.mfer",283);
        mimeTypes.put("application/vnd.mfmp",284);
        mimeTypes.put("application/vnd.micrografx.flo",285);
        mimeTypes.put("application/vnd.micrografx.igx",286);
        mimeTypes.put("application/vnd.mif",287);
        mimeTypes.put("application/vnd.mobius.daf",288);
        mimeTypes.put("application/vnd.mobius.dis",289);
        mimeTypes.put("application/vnd.mobius.mbk",290);
        mimeTypes.put("application/vnd.mobius.mqy",291);
        mimeTypes.put("application/vnd.mobius.msl",292);
        mimeTypes.put("application/vnd.mobius.plc",293);
        mimeTypes.put("application/vnd.mobius.txf",294);
        mimeTypes.put("application/vnd.mophun.application",295);
        mimeTypes.put("application/vnd.mophun.certificate",296);
        mimeTypes.put("application/vnd.mozilla.xul+xml",297);
        mimeTypes.put("application/vnd.ms-artgalry",298);
        mimeTypes.put("application/vnd.ms-cab-compressed",299);
        mimeTypes.put("application/vnd.ms-excel",300);
        mimeTypes.put("application/vnd.ms-excel.addin.macroenabled.12",301);
        mimeTypes.put("application/vnd.ms-excel.sheet.binary.macroenabled.12",302);
        mimeTypes.put("application/vnd.ms-excel.sheet.macroenabled.12",303);
        mimeTypes.put("application/vnd.ms-excel.template.macroenabled.12",304);
        mimeTypes.put("application/vnd.ms-fontobject",305);
        mimeTypes.put("application/vnd.ms-htmlhelp",306);
        mimeTypes.put("application/vnd.ms-ims",307);
        mimeTypes.put("application/vnd.ms-lrm",308);
        mimeTypes.put("application/vnd.ms-officetheme",309);
        mimeTypes.put("application/vnd.ms-pki.seccat",310);
        mimeTypes.put("application/vnd.ms-pki.stl",311);
        mimeTypes.put("application/vnd.ms-powerpoint",312);
        mimeTypes.put("application/vnd.ms-powerpoint.addin.macroenabled.12",313);
        mimeTypes.put("application/vnd.ms-powerpoint.presentation.macroenabled.12",314);
        mimeTypes.put("application/vnd.ms-powerpoint.slide.macroenabled.12",315);
        mimeTypes.put("application/vnd.ms-powerpoint.slideshow.macroenabled.12",316);
        mimeTypes.put("application/vnd.ms-powerpoint.template.macroenabled.12",317);
        mimeTypes.put("application/vnd.ms-project",318);
        mimeTypes.put("application/vnd.ms-word.document.macroenabled.12",319);
        mimeTypes.put("application/vnd.ms-word.template.macroenabled.12",320);
        mimeTypes.put("application/vnd.ms-works",321);
        mimeTypes.put("application/vnd.ms-wpl",322);
        mimeTypes.put("application/vnd.ms-xpsdocument",323);
        mimeTypes.put("application/vnd.mseq",324);
        mimeTypes.put("application/vnd.musician",325);
        mimeTypes.put("application/vnd.muvee.style",326);
        mimeTypes.put("application/vnd.mynfc",327);
        mimeTypes.put("application/vnd.neurolanguage.nlu",328);
        mimeTypes.put("application/vnd.nitf",329);
        mimeTypes.put("application/vnd.noblenet-directory",330);
        mimeTypes.put("application/vnd.noblenet-sealer",331);
        mimeTypes.put("application/vnd.noblenet-web",332);
        mimeTypes.put("application/vnd.nokia.n-gage.data",333);
        mimeTypes.put("application/vnd.nokia.n-gage.symbian.install",334);
        mimeTypes.put("application/vnd.nokia.radio-preset",335);
        mimeTypes.put("application/vnd.nokia.radio-presets",336);
        mimeTypes.put("application/vnd.novadigm.edm",337);
        mimeTypes.put("application/vnd.novadigm.edx",338);
        mimeTypes.put("application/vnd.novadigm.ext",339);
        mimeTypes.put("application/vnd.oasis.opendocument.chart",340);
        mimeTypes.put("application/vnd.oasis.opendocument.chart-template",341);
        mimeTypes.put("application/vnd.oasis.opendocument.database",342);
        mimeTypes.put("application/vnd.oasis.opendocument.formula",343);
        mimeTypes.put("application/vnd.oasis.opendocument.formula-template",344);
        mimeTypes.put("application/vnd.oasis.opendocument.graphics",345);
        mimeTypes.put("application/vnd.oasis.opendocument.graphics-template",346);
        mimeTypes.put("application/vnd.oasis.opendocument.image",347);
        mimeTypes.put("application/vnd.oasis.opendocument.image-template",348);
        mimeTypes.put("application/vnd.oasis.opendocument.presentation",349);
        mimeTypes.put("application/vnd.oasis.opendocument.presentation-template",350);
        mimeTypes.put("application/vnd.oasis.opendocument.spreadsheet",351);
        mimeTypes.put("application/vnd.oasis.opendocument.spreadsheet-template",352);
        mimeTypes.put("application/vnd.oasis.opendocument.text",353);
        mimeTypes.put("application/vnd.oasis.opendocument.text-master",354);
        mimeTypes.put("application/vnd.oasis.opendocument.text-template",355);
        mimeTypes.put("application/vnd.oasis.opendocument.text-web",356);
        mimeTypes.put("application/vnd.olpc-sugar",357);
        mimeTypes.put("application/vnd.oma.dd2+xml",358);
        mimeTypes.put("application/vnd.openofficeorg.extension",359);
        mimeTypes.put("application/vnd.openxmlformats-officedocument.presentationml.presentation",360);
        mimeTypes.put("application/vnd.openxmlformats-officedocument.presentationml.slide",361);
        mimeTypes.put("application/vnd.openxmlformats-officedocument.presentationml.slideshow",362);
        mimeTypes.put("application/vnd.openxmlformats-officedocument.presentationml.template",363);
        mimeTypes.put("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",364);
        mimeTypes.put("application/vnd.openxmlformats-officedocument.spreadsheetml.template",365);
        mimeTypes.put("application/vnd.openxmlformats-officedocument.wordprocessingml.document",366);
        mimeTypes.put("application/vnd.openxmlformats-officedocument.wordprocessingml.template",367);
        mimeTypes.put("application/vnd.osgeo.mapguide.package",368);
        mimeTypes.put("application/vnd.osgi.dp",369);
        mimeTypes.put("application/vnd.osgi.subsystem",370);
        mimeTypes.put("application/vnd.palm",371);
        mimeTypes.put("application/vnd.pawaafile",372);
        mimeTypes.put("application/vnd.pg.format",373);
        mimeTypes.put("application/vnd.pg.osasli",374);
        mimeTypes.put("application/vnd.picsel",375);
        mimeTypes.put("application/vnd.pmi.widget",376);
        mimeTypes.put("application/vnd.pocketlearn",377);
        mimeTypes.put("application/vnd.powerbuilder6",378);
        mimeTypes.put("application/vnd.previewsystems.box",379);
        mimeTypes.put("application/vnd.proteus.magazine",380);
        mimeTypes.put("application/vnd.publishare-delta-tree",381);
        mimeTypes.put("application/vnd.pvi.ptid1",382);
        mimeTypes.put("application/vnd.quark.quarkxpress",383);
        mimeTypes.put("application/vnd.realvnc.bed",384);
        mimeTypes.put("application/vnd.recordare.musicxml",385);
        mimeTypes.put("application/vnd.recordare.musicxml+xml",386);
        mimeTypes.put("application/vnd.rig.cryptonote",387);
        mimeTypes.put("application/vnd.rim.cod",388);
        mimeTypes.put("application/vnd.rn-realmedia",389);
        mimeTypes.put("application/vnd.rn-realmedia-vbr",390);
        mimeTypes.put("application/vnd.route66.link66+xml",391);
        mimeTypes.put("application/vnd.sailingtracker.track",392);
        mimeTypes.put("application/vnd.seemail",393);
        mimeTypes.put("application/vnd.sema",394);
        mimeTypes.put("application/vnd.semd",395);
        mimeTypes.put("application/vnd.semf",396);
        mimeTypes.put("application/vnd.shana.informed.formdata",397);
        mimeTypes.put("application/vnd.shana.informed.formtemplate",398);
        mimeTypes.put("application/vnd.shana.informed.interchange",399);
        mimeTypes.put("application/vnd.shana.informed.package",400);
        mimeTypes.put("application/vnd.simtech-mindmapper",401);
        mimeTypes.put("application/vnd.smaf",402);
        mimeTypes.put("application/vnd.smart.teacher",403);
        mimeTypes.put("application/vnd.solent.sdkm+xml",404);
        mimeTypes.put("application/vnd.spotfire.dxp",405);
        mimeTypes.put("application/vnd.spotfire.sfs",406);
        mimeTypes.put("application/vnd.stardivision.calc",407);
        mimeTypes.put("application/vnd.stardivision.draw",408);
        mimeTypes.put("application/vnd.stardivision.impress",409);
        mimeTypes.put("application/vnd.stardivision.math",410);
        mimeTypes.put("application/vnd.stardivision.writer",411);
        mimeTypes.put("application/vnd.stardivision.writer-global",412);
        mimeTypes.put("application/vnd.stepmania.package",413);
        mimeTypes.put("application/vnd.stepmania.stepchart",414);
        mimeTypes.put("application/vnd.sun.xml.calc",415);
        mimeTypes.put("application/vnd.sun.xml.calc.template",416);
        mimeTypes.put("application/vnd.sun.xml.draw",417);
        mimeTypes.put("application/vnd.sun.xml.draw.template",418);
        mimeTypes.put("application/vnd.sun.xml.impress",419);
        mimeTypes.put("application/vnd.sun.xml.impress.template",420);
        mimeTypes.put("application/vnd.sun.xml.math",421);
        mimeTypes.put("application/vnd.sun.xml.writer",422);
        mimeTypes.put("application/vnd.sun.xml.writer.global",423);
        mimeTypes.put("application/vnd.sun.xml.writer.template",424);
        mimeTypes.put("application/vnd.sus-calendar",425);
        mimeTypes.put("application/vnd.svd",426);
        mimeTypes.put("application/vnd.symbian.install",427);
        mimeTypes.put("application/vnd.syncml+xml",428);
        mimeTypes.put("application/vnd.syncml.dm+wbxml",429);
        mimeTypes.put("application/vnd.syncml.dm+xml",430);
        mimeTypes.put("application/vnd.tao.intent-module-archive",431);
        mimeTypes.put("application/vnd.tcpdump.pcap",432);
        mimeTypes.put("application/vnd.tmobile-livetv",433);
        mimeTypes.put("application/vnd.trid.tpt",434);
        mimeTypes.put("application/vnd.triscape.mxs",435);
        mimeTypes.put("application/vnd.trueapp",436);
        mimeTypes.put("application/vnd.ufdl",437);
        mimeTypes.put("application/vnd.uiq.theme",438);
        mimeTypes.put("application/vnd.umajin",439);
        mimeTypes.put("application/vnd.unity",440);
        mimeTypes.put("application/vnd.uoml+xml",441);
        mimeTypes.put("application/vnd.vcx",442);
        mimeTypes.put("application/vnd.visio",443);
        mimeTypes.put("application/vnd.visionary",444);
        mimeTypes.put("application/vnd.vsf",445);
        mimeTypes.put("application/vnd.wap.wbxml",446);
        mimeTypes.put("application/vnd.wap.wmlc",447);
        mimeTypes.put("application/vnd.wap.wmlscriptc",448);
        mimeTypes.put("application/vnd.webturbo",449);
        mimeTypes.put("application/vnd.wolfram.player",450);
        mimeTypes.put("application/vnd.wordperfect",451);
        mimeTypes.put("application/vnd.wqd",452);
        mimeTypes.put("application/vnd.wt.stf",453);
        mimeTypes.put("application/vnd.xara",454);
        mimeTypes.put("application/vnd.xfdl",455);
        mimeTypes.put("application/vnd.yamaha.hv-dic",456);
        mimeTypes.put("application/vnd.yamaha.hv-script",457);
        mimeTypes.put("application/vnd.yamaha.hv-voice",458);
        mimeTypes.put("application/vnd.yamaha.openscoreformat",459);
        mimeTypes.put("application/vnd.yamaha.openscoreformat.osfpvg+xml",460);
        mimeTypes.put("application/vnd.yamaha.smaf-audio",461);
        mimeTypes.put("application/vnd.yamaha.smaf-phrase",462);
        mimeTypes.put("application/vnd.yellowriver-custom-menu",463);
        mimeTypes.put("application/vnd.zul",464);
        mimeTypes.put("application/vnd.zzazz.deck+xml",465);
        mimeTypes.put("application/voicexml+xml",466);
        mimeTypes.put("application/widget",467);
        mimeTypes.put("application/winhlp",468);
        mimeTypes.put("application/wsdl+xml",469);
        mimeTypes.put("application/wspolicy+xml",470);
        mimeTypes.put("application/x-7z-compressed",471);
        mimeTypes.put("application/x-abiword",472);
        mimeTypes.put("application/x-ace-compressed",473);
        mimeTypes.put("application/x-apple-diskimage",474);
        mimeTypes.put("application/x-authorware-bin",475);
        mimeTypes.put("application/x-authorware-map",476);
        mimeTypes.put("application/x-authorware-seg",477);
        mimeTypes.put("application/x-bcpio",478);
        mimeTypes.put("application/x-bittorrent",479);
        mimeTypes.put("application/x-blorb",480);
        mimeTypes.put("application/x-bzip",481);
        mimeTypes.put("application/x-bzip2",482);
        mimeTypes.put("application/x-cbr",483);
        mimeTypes.put("application/x-cdlink",484);
        mimeTypes.put("application/x-cfs-compressed",485);
        mimeTypes.put("application/x-chat",486);
        mimeTypes.put("application/x-chess-pgn",487);
        mimeTypes.put("application/x-conference",488);
        mimeTypes.put("application/x-cpio",489);
        mimeTypes.put("application/x-csh",490);
        mimeTypes.put("application/x-debian-package",491);
        mimeTypes.put("application/x-dgc-compressed",492);
        mimeTypes.put("application/x-director",493);
        mimeTypes.put("application/x-doom",494);
        mimeTypes.put("application/x-dtbncx+xml",495);
        mimeTypes.put("application/x-dtbook+xml",496);
        mimeTypes.put("application/x-dtbresource+xml",497);
        mimeTypes.put("application/x-dvi",498);
        mimeTypes.put("application/x-envoy",499);
        mimeTypes.put("application/x-eva",500);
        mimeTypes.put("application/x-font-bdf",501);
        mimeTypes.put("application/x-font-ghostscript",502);
        mimeTypes.put("application/x-font-linux-psf",503);
        mimeTypes.put("application/x-font-otf",504);
        mimeTypes.put("application/x-font-pcf",505);
        mimeTypes.put("application/x-font-snf",506);
        mimeTypes.put("application/x-font-ttf",507);
        mimeTypes.put("application/x-font-type1",508);
        mimeTypes.put("application/x-font-woff",509);
        mimeTypes.put("application/x-freearc",510);
        mimeTypes.put("application/x-futuresplash",511);
        mimeTypes.put("application/x-gca-compressed",512);
        mimeTypes.put("application/x-glulx",513);
        mimeTypes.put("application/x-gnumeric",514);
        mimeTypes.put("application/x-gramps-xml",515);
        mimeTypes.put("application/x-gtar",516);
        mimeTypes.put("application/x-hdf",517);
        mimeTypes.put("application/x-install-instructions",518);
        mimeTypes.put("application/x-iso9660-image",519);
        mimeTypes.put("application/x-java-jnlp-file",520);
        mimeTypes.put("application/x-latex",521);
        mimeTypes.put("application/x-lzh-compressed",522);
        mimeTypes.put("application/x-mie",523);
        mimeTypes.put("application/x-mobipocket-ebook",524);
        mimeTypes.put("application/x-ms-application",525);
        mimeTypes.put("application/x-ms-shortcut",526);
        mimeTypes.put("application/x-ms-wmd",527);
        mimeTypes.put("application/x-ms-wmz",528);
        mimeTypes.put("application/x-ms-xbap",529);
        mimeTypes.put("application/x-msaccess",530);
        mimeTypes.put("application/x-msbinder",531);
        mimeTypes.put("application/x-mscardfile",532);
        mimeTypes.put("application/x-msclip",533);
        mimeTypes.put("application/x-msdownload",534);
        mimeTypes.put("application/x-msmediaview",535);
        mimeTypes.put("application/x-msmetafile",536);
        mimeTypes.put("application/x-msmoney",537);
        mimeTypes.put("application/x-mspublisher",538);
        mimeTypes.put("application/x-msschedule",539);
        mimeTypes.put("application/x-msterminal",540);
        mimeTypes.put("application/x-mswrite",541);
        mimeTypes.put("application/x-netcdf",542);
        mimeTypes.put("application/x-nzb",543);
        mimeTypes.put("application/x-pkcs12",544);
        mimeTypes.put("application/x-pkcs7-certificates",545);
        mimeTypes.put("application/x-pkcs7-certreqresp",546);
        mimeTypes.put("application/x-rar-compressed",547);
        mimeTypes.put("application/x-research-info-systems",548);
        mimeTypes.put("application/x-sh",549);
        mimeTypes.put("application/x-shar",550);
        mimeTypes.put("application/x-shockwave-flash",551);
        mimeTypes.put("application/x-silverlight-app",552);
        mimeTypes.put("application/x-sql",553);
        mimeTypes.put("application/x-stuffit",554);
        mimeTypes.put("application/x-stuffitx",555);
        mimeTypes.put("application/x-subrip",556);
        mimeTypes.put("application/x-sv4cpio",557);
        mimeTypes.put("application/x-sv4crc",558);
        mimeTypes.put("application/x-t3vm-image",559);
        mimeTypes.put("application/x-tads",560);
        mimeTypes.put("application/x-tar",561);
        mimeTypes.put("application/x-tcl",562);
        mimeTypes.put("application/x-tex",563);
        mimeTypes.put("application/x-tex-tfm",564);
        mimeTypes.put("application/x-texinfo",565);
        mimeTypes.put("application/x-tgif",566);
        mimeTypes.put("application/x-ustar",567);
        mimeTypes.put("application/x-wais-source",568);
        mimeTypes.put("application/x-x509-ca-cert",569);
        mimeTypes.put("application/x-xfig",570);
        mimeTypes.put("application/x-xliff+xml",571);
        mimeTypes.put("application/x-xpinstall",572);
        mimeTypes.put("application/x-xz",573);
        mimeTypes.put("application/x-zmachine",574);
        mimeTypes.put("application/xaml+xml",575);
        mimeTypes.put("application/xcap-diff+xml",576);
        mimeTypes.put("application/xenc+xml",577);
        mimeTypes.put("application/xhtml+xml",578);
        mimeTypes.put("application/xml",579);
        mimeTypes.put("application/xml-dtd",580);
        mimeTypes.put("application/xop+xml",581);
        mimeTypes.put("application/xproc+xml",582);
        mimeTypes.put("application/xslt+xml",583);
        mimeTypes.put("application/xspf+xml",584);
        mimeTypes.put("application/xv+xml",585);
        mimeTypes.put("application/yang",586);
        mimeTypes.put("application/yin+xml",587);
        mimeTypes.put("application/zip",588);
        mimeTypes.put("audio/adpcm",589);
        mimeTypes.put("audio/basic",590);
        mimeTypes.put("audio/midi",591);
        mimeTypes.put("audio/mp4",592);
        mimeTypes.put("audio/mpeg",593);
        mimeTypes.put("audio/ogg",594);
        mimeTypes.put("audio/s3m",595);
        mimeTypes.put("audio/silk",596);
        mimeTypes.put("audio/vnd.dece.audio",597);
        mimeTypes.put("audio/vnd.digital-winds",598);
        mimeTypes.put("audio/vnd.dra",599);
        mimeTypes.put("audio/vnd.dts",600);
        mimeTypes.put("audio/vnd.dts.hd",601);
        mimeTypes.put("audio/vnd.lucent.voice",602);
        mimeTypes.put("audio/vnd.ms-playready.media.pya",603);
        mimeTypes.put("audio/vnd.nuera.ecelp4800",604);
        mimeTypes.put("audio/vnd.nuera.ecelp7470",605);
        mimeTypes.put("audio/vnd.nuera.ecelp9600",606);
        mimeTypes.put("audio/vnd.rip",607);
        mimeTypes.put("audio/webm",608);
        mimeTypes.put("audio/x-aac",609);
        mimeTypes.put("audio/x-aiff",610);
        mimeTypes.put("audio/x-caf",611);
        mimeTypes.put("audio/x-flac",612);
        mimeTypes.put("audio/x-matroska",613);
        mimeTypes.put("audio/x-mpegurl",614);
        mimeTypes.put("audio/x-ms-wax",615);
        mimeTypes.put("audio/x-ms-wma",616);
        mimeTypes.put("audio/x-pn-realaudio",617);
        mimeTypes.put("audio/x-pn-realaudio-plugin",618);
        mimeTypes.put("audio/x-wav",619);
        mimeTypes.put("audio/xm",620);
        mimeTypes.put("chemical/x-cdx",621);
        mimeTypes.put("chemical/x-cif",622);
        mimeTypes.put("chemical/x-cmdf",623);
        mimeTypes.put("chemical/x-cml",624);
        mimeTypes.put("chemical/x-csml",625);
        mimeTypes.put("chemical/x-xyz",626);
        mimeTypes.put("image/bmp",627);
        mimeTypes.put("image/cgm",628);
        mimeTypes.put("image/g3fax",629);
        mimeTypes.put("image/gif",630);
        mimeTypes.put("image/ief",631);
        mimeTypes.put("image/jpeg",632);
        mimeTypes.put("image/ktx",633);
        mimeTypes.put("image/png",634);
        mimeTypes.put("image/prs.btif",635);
        mimeTypes.put("image/sgi",636);
        mimeTypes.put("image/svg+xml",637);
        mimeTypes.put("image/tiff",638);
        mimeTypes.put("image/vnd.adobe.photoshop",639);
        mimeTypes.put("image/vnd.dece.graphic",640);
        mimeTypes.put("image/vnd.dvb.subtitle",641);
        mimeTypes.put("image/vnd.djvu",642);
        mimeTypes.put("image/vnd.dwg",643);
        mimeTypes.put("image/vnd.dxf",644);
        mimeTypes.put("image/vnd.fastbidsheet",645);
        mimeTypes.put("image/vnd.fpx",646);
        mimeTypes.put("image/vnd.fst",647);
        mimeTypes.put("image/vnd.fujixerox.edmics-mmr",648);
        mimeTypes.put("image/vnd.fujixerox.edmics-rlc",649);
        mimeTypes.put("image/vnd.ms-modi",650);
        mimeTypes.put("image/vnd.ms-photo",651);
        mimeTypes.put("image/vnd.net-fpx",652);
        mimeTypes.put("image/vnd.wap.wbmp",653);
        mimeTypes.put("image/vnd.xiff",654);
        mimeTypes.put("image/webp",655);
        mimeTypes.put("image/x-3ds",656);
        mimeTypes.put("image/x-cmu-raster",657);
        mimeTypes.put("image/x-cmx",658);
        mimeTypes.put("image/x-freehand",659);
        mimeTypes.put("image/x-icon",660);
        mimeTypes.put("image/x-mrsid-image",661);
        mimeTypes.put("image/x-pcx",662);
        mimeTypes.put("image/x-pict",663);
        mimeTypes.put("image/x-portable-anymap",664);
        mimeTypes.put("image/x-portable-bitmap",665);
        mimeTypes.put("image/x-portable-graymap",666);
        mimeTypes.put("image/x-portable-pixmap",667);
        mimeTypes.put("image/x-rgb",668);
        mimeTypes.put("image/x-tga",669);
        mimeTypes.put("image/x-xbitmap",670);
        mimeTypes.put("image/x-xpixmap",671);
        mimeTypes.put("image/x-xwindowdump",672);
        mimeTypes.put("message/rfc822",673);
        mimeTypes.put("model/iges",674);
        mimeTypes.put("model/mesh",675);
        mimeTypes.put("model/vnd.collada+xml",676);
        mimeTypes.put("model/vnd.dwf",677);
        mimeTypes.put("model/vnd.gdl",678);
        mimeTypes.put("model/vnd.gtw",679);
        mimeTypes.put("model/vnd.mts",680);
        mimeTypes.put("model/vnd.vtu",681);
        mimeTypes.put("model/vrml",682);
        mimeTypes.put("model/x3d+binary",683);
        mimeTypes.put("model/x3d+vrml",684);
        mimeTypes.put("model/x3d+xml",685);
        mimeTypes.put("text/cache-manifest",686);
        mimeTypes.put("text/calendar",687);
        mimeTypes.put("text/css",688);
        mimeTypes.put("text/csv",689);
        mimeTypes.put("text/html",690);
        mimeTypes.put("text/n3",691);
        mimeTypes.put("text/plain",692);
        mimeTypes.put("text/prs.lines.tag",693);
        mimeTypes.put("text/richtext",694);
        mimeTypes.put("text/sgml",695);
        mimeTypes.put("text/tab-separated-values",696);
        mimeTypes.put("text/troff",697);
        mimeTypes.put("text/turtle",698);
        mimeTypes.put("text/uri-list",699);
        mimeTypes.put("text/vcard",700);
        mimeTypes.put("text/vnd.curl",701);
        mimeTypes.put("text/vnd.curl.dcurl",702);
        mimeTypes.put("text/vnd.curl.scurl",703);
        mimeTypes.put("text/vnd.curl.mcurl",704);
        mimeTypes.put("text/vnd.dvb.subtitle",705);
        mimeTypes.put("text/vnd.fly",706);
        mimeTypes.put("text/vnd.fmi.flexstor",707);
        mimeTypes.put("text/vnd.graphviz",708);
        mimeTypes.put("text/vnd.in3d.3dml",709);
        mimeTypes.put("text/vnd.in3d.spot",710);
        mimeTypes.put("text/vnd.sun.j2me.app-descriptor",711);
        mimeTypes.put("text/vnd.wap.wml",712);
        mimeTypes.put("text/vnd.wap.wmlscript",713);
        mimeTypes.put("text/x-asm",714);
        mimeTypes.put("text/x-c",715);
        mimeTypes.put("text/x-fortran",716);
        mimeTypes.put("text/x-java-source",717);
        mimeTypes.put("text/x-opml",718);
        mimeTypes.put("text/x-pascal",719);
        mimeTypes.put("text/x-nfo",720);
        mimeTypes.put("text/x-setext",721);
        mimeTypes.put("text/x-sfv",722);
        mimeTypes.put("text/x-uuencode",723);
        mimeTypes.put("text/x-vcalendar",724);
        mimeTypes.put("text/x-vcard",725);
        mimeTypes.put("video/3gpp",726);
        mimeTypes.put("video/3gpp2",727);
        mimeTypes.put("video/h261",728);
        mimeTypes.put("video/h263",729);
        mimeTypes.put("video/h264",730);
        mimeTypes.put("video/jpeg",731);
        mimeTypes.put("video/jpm",732);
        mimeTypes.put("video/mj2",733);
        mimeTypes.put("video/mp4",734);
        mimeTypes.put("video/mpeg",735);
        mimeTypes.put("video/ogg",736);
        mimeTypes.put("video/quicktime",737);
        mimeTypes.put("video/vnd.dece.hd",738);
        mimeTypes.put("video/vnd.dece.mobile",739);
        mimeTypes.put("video/vnd.dece.pd",740);
        mimeTypes.put("video/vnd.dece.sd",741);
        mimeTypes.put("video/vnd.dece.video",742);
        mimeTypes.put("video/vnd.dvb.file",743);
        mimeTypes.put("video/vnd.fvt",744);
        mimeTypes.put("video/vnd.mpegurl",745);
        mimeTypes.put("video/vnd.ms-playready.media.pyv",746);
        mimeTypes.put("video/vnd.uvvu.mp4",747);
        mimeTypes.put("video/vnd.vivo",748);
        mimeTypes.put("video/webm",749);
        mimeTypes.put("video/x-f4v",750);
        mimeTypes.put("video/x-fli",751);
        mimeTypes.put("video/x-flv",752);
        mimeTypes.put("video/x-m4v",753);
        mimeTypes.put("video/x-matroska",754);
        mimeTypes.put("video/x-mng",755);
        mimeTypes.put("video/x-ms-asf",756);
        mimeTypes.put("video/x-ms-vob",757);
        mimeTypes.put("video/x-ms-wm",758);
        mimeTypes.put("video/x-ms-wmv",759);
        mimeTypes.put("video/x-ms-wmx",760);
        mimeTypes.put("video/x-ms-wvx",761);
        mimeTypes.put("video/x-msvideo",762);
        mimeTypes.put("video/x-sgi-movie",763);
        mimeTypes.put("video/x-smv",764);
        mimeTypes.put("x-conference/x-cooltalk",765);
        mimeTypes.put("application/x-download",766);
        mimeTypes.put("image/pjpeg",767);
        mimeTypes.put("audio/flac",768);
        mimeTypes.put("pdf",769);
        mimeTypes.put("image",770);
        mimeTypes.put("image/pjpeg",771);
        mimeTypes.put("image/jpg",772);
        mimeTypes.put("audio/x-mpeg",773);
        mimeTypes.put("audio/mpeg3",774);
        mimeTypes.put("image/x-ms-bmp",775);
    }

    /**
     * Codifies the given mimetype
     * @param type the mimetype of the resource
     * @return the integer represantation of the mimetype
     */
    public static Integer getMimeTypeCode(final String type) {
        Integer retval = StringUtils.isNotBlank(type) ? mimeTypes.get(type.toLowerCase()) : 0;
        return retval == null ? 0 : retval;
    }

    public static String getMimeType(Integer tag) {
        return CommonTagExtractor.mimeTypes.inverse().get(TagEncoding.MIME_TYPE.extractValue(tag));
    }

    public static boolean isImageMimeType(String type) {
        return (StringUtils.startsWithIgnoreCase(type, "image"));
    }

    public static boolean isSoundMimeType(String type) {
        return (StringUtils.startsWithIgnoreCase(type, "sound") || StringUtils.startsWithIgnoreCase(type, "audio"));
    }

    public static boolean isVideoMimeType(String type) {
        return (StringUtils.startsWithIgnoreCase(type, "video"));
    }

    public static MediaTypeEncoding getType(Integer tag) {
        return MediaTypeEncoding.valueOf(TagEncoding.MEDIA_TYPE.extractValue(tag));
    }
}
