/*
 * Decompiled with CFR 0.152.
 */
package eu.europeana.entitymanagement.web.service;

import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import eu.europeana.entitymanagement.definitions.exceptions.UnsupportedEntityTypeException;
import eu.europeana.entitymanagement.definitions.model.Entity;
import eu.europeana.entitymanagement.exception.FunctionalRuntimeException;
import eu.europeana.entitymanagement.exception.ScoringComputationException;
import eu.europeana.entitymanagement.vocabulary.EntityTypes;
import eu.europeana.entitymanagement.web.model.scoring.EntityMetrics;
import eu.europeana.entitymanagement.web.model.scoring.MaxEntityMetrics;
import eu.europeana.entitymanagement.web.model.scoring.PageRank;
import eu.europeana.entitymanagement.web.service.EnrichmentCountQueryService;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.params.SolrParams;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;

@Service(value="emScoringService")
public class ScoringService {
    private static final Logger logger = LogManager.getLogger(ScoringService.class);
    private final EnrichmentCountQueryService enrichmentCountQueryService;
    private final SolrClient prSolrClient;
    private MaxEntityMetrics maxEntityMetrics;
    private EntityMetrics maxOverallMetrics;
    private static final int RANGE_EXTENSION_FACTOR = 100;
    public static final String WIKIDATA_PREFFIX = "http://www.wikidata.org/entity/";
    public static final String WIKIDATA_DBPEDIA_PREFIX = "http://wikidata.dbpedia.org/resource/";

    public ScoringService(EnrichmentCountQueryService enrichmentCountQueryService, @Qualifier(value="prSolrClient") SolrClient prSolrClient) {
        this.enrichmentCountQueryService = enrichmentCountQueryService;
        this.prSolrClient = prSolrClient;
    }

    public EntityMetrics computeMetrics(Entity entity) throws FunctionalRuntimeException, UnsupportedEntityTypeException {
        EntityMetrics metrics = new EntityMetrics(entity.getEntityId());
        if (entity.getType() != null) {
            metrics.setEntityType(entity.getType());
        } else {
            metrics.setEntityType(EntityTypes.getByEntityId((String)entity.getEntityId()).name());
        }
        PageRank pr = this.getPageRank(entity);
        if (pr != null && pr.getPageRank() != null) {
            metrics.setPageRank(Integer.valueOf(pr.getPageRank().intValue()));
        }
        metrics.setEnrichmentCount(this.getEnrichmentCount(entity));
        this.computeScore(metrics);
        return metrics;
    }

    private void computeScore(EntityMetrics metrics) throws FunctionalRuntimeException {
        int normalizedPR = 0;
        int normalizedEC = 0;
        try {
            normalizedPR = this.computeNormalizedMetricValue(metrics, "pageRank");
            normalizedEC = this.computeNormalizedMetricValue(metrics, "enrichmentCount");
        }
        catch (IOException e) {
            throw new FunctionalRuntimeException("Cannot compute entity score. There is probably a sistem configuration issue", (Throwable)e);
        }
        int score = normalizedPR * normalizedEC;
        metrics.setScore(Integer.valueOf(score));
    }

    private int computeNormalizedMetricValue(EntityMetrics metrics, String metric) throws IOException {
        int maxValueOverall;
        int maxValueForType;
        int metricValue;
        float trust = 1.0f;
        switch (metric) {
            case "pageRank": {
                metricValue = metrics.getPageRank();
                if (metricValue <= 1) {
                    return 1;
                }
                maxValueForType = this.getMaxEntityMetrics().maxValues(metrics.getEntityType()).getPageRank();
                maxValueOverall = this.getMaxOverallMetrics().getPageRank();
                break;
            }
            case "enrichmentCount": {
                metricValue = metrics.getEnrichmentCount();
                if (metricValue <= 1) {
                    return 1;
                }
                maxValueForType = this.getMaxEntityMetrics().maxValues(metrics.getEntityType()).getEnrichmentCount();
                maxValueOverall = this.getMaxOverallMetrics().getEnrichmentCount();
                break;
            }
            default: {
                throw new FunctionalRuntimeException("Unknown/unsuported metric: " + metric);
            }
        }
        long coordinationFactor = maxValueOverall / maxValueForType;
        double normalizedValue = 1.0 + (double)(trust * 100.0f) * Math.log((double)coordinationFactor * (double)metricValue);
        return (int)normalizedValue;
    }

    public EntityMetrics getMaxOverallMetrics() throws IOException {
        if (this.maxOverallMetrics == null) {
            this.maxOverallMetrics = new EntityMetrics();
            this.maxOverallMetrics.setEntityType("all");
            this.maxOverallMetrics.setEntityId("*");
            int maxPR = 1;
            int maxEC = 1;
            int maxHC = 1;
            for (EntityMetrics metrics : this.getMaxEntityMetrics().getMaxValues()) {
                maxPR = Math.max(metrics.getPageRank(), maxPR);
                maxEC = Math.max(metrics.getEnrichmentCount(), maxEC);
                maxHC = Math.max(metrics.getHitCount(), maxHC);
            }
            this.maxOverallMetrics.setPageRank(Integer.valueOf(maxPR));
            this.maxOverallMetrics.setEnrichmentCount(Integer.valueOf(maxEC));
            this.maxOverallMetrics.setHitCount(Integer.valueOf(maxHC));
        }
        return this.maxOverallMetrics;
    }

    private Integer getEnrichmentCount(Entity entity) {
        return this.enrichmentCountQueryService.getEnrichmentCount(entity.getEntityId(), entity.getType());
    }

    private PageRank getPageRank(Entity entity) {
        SolrQuery query = new SolrQuery();
        String wikidataUrl = this.getWikidataUrl(entity);
        if (wikidataUrl == null) {
            return null;
        }
        query.setQuery("page_url:\"" + wikidataUrl + "\"");
        try {
            Instant start = Instant.now();
            QueryResponse rsp = this.prSolrClient.query((SolrParams)query);
            List beans = rsp.getBeans(PageRank.class);
            if (logger.isDebugEnabled()) {
                logger.debug("Retrieved pageRank for entityId={} in {}ms", (Object)entity.getEntityId(), (Object)Duration.between(start, Instant.now()).toMillis());
            }
            if (beans.isEmpty()) {
                return null;
            }
            return (PageRank)beans.get(0);
        }
        catch (Exception e) {
            throw new ScoringComputationException("Unexpected exception occured when retrieving pagerank: " + wikidataUrl, (Throwable)e);
        }
    }

    private String getWikidataUrl(Entity entity) {
        if (entity.getSameReferenceLinks() == null) {
            return null;
        }
        List values = entity.getSameReferenceLinks();
        return values.stream().filter(value -> value.startsWith(WIKIDATA_PREFFIX)).findFirst().orElse(null);
    }

    public MaxEntityMetrics getMaxEntityMetrics() throws IOException {
        if (this.maxEntityMetrics == null) {
            XmlMapper xmlMapper = new XmlMapper();
            try (InputStream inputStream = this.getClass().getResourceAsStream("/max-entity-metrics.xml");
                 BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));){
                String contents = reader.lines().collect(Collectors.joining(System.lineSeparator()));
                this.maxEntityMetrics = (MaxEntityMetrics)xmlMapper.readValue(contents, MaxEntityMetrics.class);
            }
        }
        return this.maxEntityMetrics;
    }
}

