/*
 * Decompiled with CFR 0.152.
 */
package eu.erasmuswithoutpaper.registryclient;

import eu.erasmuswithoutpaper.registryclient.ApiSearchConditions;
import eu.erasmuswithoutpaper.registryclient.CatalogueDocument;
import eu.erasmuswithoutpaper.registryclient.CatalogueFetcher;
import eu.erasmuswithoutpaper.registryclient.ClientImplOptions;
import eu.erasmuswithoutpaper.registryclient.HeiEntry;
import eu.erasmuswithoutpaper.registryclient.RegistryClient;
import eu.erasmuswithoutpaper.registryclient.SelfSchedulableTask;
import eu.erasmuswithoutpaper.registryclient.Utils;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.cert.Certificate;
import java.security.interfaces.RSAPublicKey;
import java.util.Collection;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

public class ClientImpl
implements RegistryClient {
    private static final String CATALOGUE_CACHE_KEY = "latest-catalogue";
    private static final Logger logger = LoggerFactory.getLogger(ClientImpl.class);
    private final ClientImplOptions options;
    private volatile CatalogueDocument doc;
    private final ScheduledExecutorService executor;

    public ClientImpl() {
        this(new ClientImplOptions());
    }

    public ClientImpl(ClientImplOptions options) {
        this.options = options;
        logger.info("Constructing new ClientImpl with options: " + options);
        Map<String, byte[]> cache = this.options.getPersistentCacheMap();
        if (cache != null) {
            logger.debug("Attempting to load a catalogue from cache");
            byte[] data = cache.get(CATALOGUE_CACHE_KEY);
            if (data != null) {
                try {
                    CatalogueFetcher.Http200RegistryResponse cachedResponse = CatalogueFetcher.Http200RegistryResponse.deserialize(cache.get(CATALOGUE_CACHE_KEY));
                    this.doc = new CatalogueDocument(cachedResponse);
                    logger.info("Loaded a catalogue from cache: " + this.doc);
                }
                catch (CatalogueDocument.CatalogueParserException | CatalogueFetcher.Http200RegistryResponse.CouldNotDeserialize e) {
                    logger.debug("Could not load the catalogue from cache: " + e);
                }
            }
        }
        if (this.doc == null) {
            StringBuilder sb = new StringBuilder();
            sb.append("<catalogue xmlns='");
            sb.append("https://github.com/erasmus-without-paper/ewp-specs-api-registry/tree/stable-v1");
            sb.append("'></catalogue>");
            byte[] content = sb.toString().getBytes(StandardCharsets.UTF_8);
            String newETag = "empty-placeholder";
            Date expires = new Date(0L);
            CatalogueFetcher.Http200RegistryResponse emptyResponse = new CatalogueFetcher.Http200RegistryResponse(content, newETag, expires);
            try {
                this.doc = new CatalogueDocument(emptyResponse);
            }
            catch (CatalogueDocument.CatalogueParserException e) {
                throw new RuntimeException(e);
            }
        }
        if (options.isAutoRefreshing()) {
            if (this.getExpiryDate().after(new Date())) {
                logger.debug("The cached copy seems to be acceptable. We won't be refreshing it (this will speed up the construction).");
            } else {
                logger.debug("Our cached copy of the catalogue has expired. We will refresh it now (this might make construction a bit slower).");
                try {
                    this.refresh();
                }
                catch (RegistryClient.RefreshFailureException e2) {
                    this.logRefreshFailure("Failed to fetch a fresh copy of the catalogue during construction (we will keep trying in a background thread).", e2);
                }
            }
            this.executor = Executors.newSingleThreadScheduledExecutor();
            this.executor.schedule(new SelfSchedulableTask(this.executor, this.options.getMinTimeBetweenQueries()){

                @Override
                protected Date runAndScheduleNext() {
                    try {
                        logger.trace("runAndScheduleNext was called");
                        Date now = new Date();
                        Date expiryDate = ClientImpl.this.getExpiryDate();
                        if (expiryDate.after(now)) {
                            logger.trace("No refresh was necessary. Will retry at " + expiryDate);
                            return ClientImpl.this.getExpiryDate();
                        }
                        ClientImpl.this.refresh();
                        return ClientImpl.this.getExpiryDate();
                    }
                    catch (RegistryClient.RefreshFailureException e) {
                        ClientImpl.this.logRefreshFailure("Scheduled catalogue refresh failed. Will retry in " + ClientImpl.this.options.getTimeBetweenRetries() + "ms.", e);
                        return new Date(new Date().getTime() + ClientImpl.this.options.getTimeBetweenRetries());
                    }
                }
            }, 0L, TimeUnit.MILLISECONDS);
        } else {
            this.executor = null;
        }
    }

    @Override
    public boolean areHeisCoveredByCertificate(Collection<String> heiIds, Certificate clientCert) throws RegistryClient.UnacceptableStalenessException {
        return this.getHeisCoveredByCertificate(clientCert).containsAll(heiIds);
    }

    @Override
    public boolean areHeisCoveredByCertificate(String[] heiIds, Certificate clientCert) throws RegistryClient.UnacceptableStalenessException {
        Collection<String> heis = this.getHeisCoveredByCertificate(clientCert);
        for (String heiId : heiIds) {
            if (heis.contains(heiId)) continue;
            return false;
        }
        return true;
    }

    @Override
    public boolean areHeisCoveredByClientKey(Collection<String> heiIds, RSAPublicKey clientKey) throws RegistryClient.UnacceptableStalenessException {
        return this.getHeisCoveredByClientKey(clientKey).containsAll(heiIds);
    }

    @Override
    public boolean areHeisCoveredByClientKey(String[] heiIds, RSAPublicKey clientKey) throws RegistryClient.UnacceptableStalenessException {
        Collection<String> heis = this.getHeisCoveredByClientKey(clientKey);
        for (String heiId : heiIds) {
            if (heis.contains(heiId)) continue;
            return false;
        }
        return true;
    }

    @Override
    public void assertApiIsCoveredByServerKey(Element apiElement, RSAPublicKey serverKey) throws RegistryClient.AssertionFailedException, RegistryClient.InvalidApiEntryElement, RegistryClient.UnacceptableStalenessException {
        if (!this.isApiCoveredByServerKey(apiElement, serverKey)) {
            throw new RegistryClient.AssertionFailedException("API entry " + apiElement.toString() + " doesn't seem to be covered by this server key: " + Utils.extractFingerprint(serverKey));
        }
    }

    @Override
    public void assertCertificateIsKnown(Certificate clientCert) throws RegistryClient.AssertionFailedException {
        if (!this.isCertificateKnown(clientCert)) {
            throw new RegistryClient.AssertionFailedException("Certificate was not recognized as a known EWP Client: " + Utils.extractFingerprint(clientCert));
        }
    }

    @Override
    public void assertClientKeyIsKnown(RSAPublicKey clientKey) throws RegistryClient.AssertionFailedException {
        if (!this.isClientKeyKnown(clientKey)) {
            throw new RegistryClient.AssertionFailedException("Key was not recognized as a known EWP Client: " + Utils.extractFingerprint(clientKey));
        }
    }

    @Override
    public void assertHeiIsCoveredByCertificate(String heiId, Certificate clientCert) throws RegistryClient.AssertionFailedException {
        if (!this.isHeiCoveredByCertificate(heiId, clientCert)) {
            throw new RegistryClient.AssertionFailedException("HEI " + heiId + " is not covered by this certificate.");
        }
    }

    @Override
    public void assertHeiIsCoveredByClientKey(String heiId, RSAPublicKey clientKey) throws RegistryClient.AssertionFailedException {
        if (!this.isHeiCoveredByClientKey(heiId, clientKey)) {
            throw new RegistryClient.AssertionFailedException("HEI " + heiId + " is not covered by this client key.");
        }
    }

    @Override
    public void assertHeisAreCoveredByCertificate(Collection<String> heiIds, Certificate clientCert) throws RegistryClient.AssertionFailedException {
        if (!this.areHeisCoveredByCertificate(heiIds, clientCert)) {
            throw new RegistryClient.AssertionFailedException("Some of the HEIs are not covered by this certificate.");
        }
    }

    @Override
    public void assertHeisAreCoveredByCertificate(String[] heiIds, Certificate clientCert) throws RegistryClient.AssertionFailedException {
        if (!this.areHeisCoveredByCertificate(heiIds, clientCert)) {
            throw new RegistryClient.AssertionFailedException("Some of the HEIs are not covered by this certificate.");
        }
    }

    @Override
    public void assertHeisAreCoveredByClientKey(Collection<String> heiIds, RSAPublicKey clientKey) throws RegistryClient.AssertionFailedException {
        if (!this.areHeisCoveredByClientKey(heiIds, clientKey)) {
            throw new RegistryClient.AssertionFailedException("Some of the HEIs are not covered by this client key.");
        }
    }

    @Override
    public void assertHeisAreCoveredByClientKey(String[] heiIds, RSAPublicKey clientKey) throws RegistryClient.AssertionFailedException, RegistryClient.UnacceptableStalenessException {
        if (!this.areHeisCoveredByClientKey(heiIds, clientKey)) {
            throw new RegistryClient.AssertionFailedException("Some of the HEIs are not covered by this client key.");
        }
    }

    @Override
    public void close() {
        logger.info("ClientImpl is closing");
        if (this.executor != null) {
            this.executor.shutdownNow();
            try {
                if (this.executor.awaitTermination(30L, TimeUnit.SECONDS)) {
                    logger.info("All threads exited successfully.");
                } else {
                    logger.warn("Some threads are still running, but we won't wait anymore.");
                }
            }
            catch (InterruptedException e) {
                logger.warn("Interrupted while waiting for threads to finish.");
            }
        }
        logger.info("ClientImpl finished closing");
    }

    @Override
    public Element findApi(ApiSearchConditions conditions) {
        this.assertAcceptableStaleness();
        return this.doc.findApi(conditions);
    }

    @Override
    public Collection<Element> findApis(ApiSearchConditions conditions) {
        this.assertAcceptableStaleness();
        return this.doc.findApis(conditions);
    }

    @Override
    public HeiEntry findHei(String id) throws RegistryClient.UnacceptableStalenessException {
        this.assertAcceptableStaleness();
        return this.doc.findHei(id);
    }

    @Override
    public HeiEntry findHei(String type, String value) throws RegistryClient.UnacceptableStalenessException {
        this.assertAcceptableStaleness();
        return this.doc.findHei(type, value);
    }

    @Override
    public String findHeiId(String type, String value) {
        this.assertAcceptableStaleness();
        return this.doc.findHeiId(type, value);
    }

    @Override
    public Collection<HeiEntry> findHeis(ApiSearchConditions conditions) throws RegistryClient.UnacceptableStalenessException {
        this.assertAcceptableStaleness();
        return this.doc.findHeis(conditions);
    }

    @Override
    public RSAPublicKey findRsaPublicKey(String fingerprint) throws RegistryClient.UnacceptableStalenessException {
        this.assertAcceptableStaleness();
        return this.doc.findRsaPublicKey(fingerprint);
    }

    @Override
    public Collection<HeiEntry> getAllHeis() throws RegistryClient.UnacceptableStalenessException {
        this.assertAcceptableStaleness();
        return this.doc.getAllHeis();
    }

    @Override
    public Date getExpiryDate() {
        return this.doc.getExpiryDate();
    }

    @Override
    public Collection<String> getHeisCoveredByCertificate(Certificate clientCert) {
        this.assertAcceptableStaleness();
        return this.doc.getHeisCoveredByCertificate(clientCert);
    }

    @Override
    public Collection<String> getHeisCoveredByClientKey(RSAPublicKey clientKey) throws RegistryClient.UnacceptableStalenessException {
        this.assertAcceptableStaleness();
        return this.doc.getHeisCoveredByClientKey(clientKey);
    }

    @Override
    public RSAPublicKey getServerKeyCoveringApi(Element apiElement) throws RegistryClient.UnacceptableStalenessException, RegistryClient.InvalidApiEntryElement {
        this.assertAcceptableStaleness();
        return this.doc.getServerKeyCoveringApi(apiElement);
    }

    @Override
    public Collection<RSAPublicKey> getServerKeysCoveringApi(Element apiElement) throws RegistryClient.UnacceptableStalenessException, RegistryClient.InvalidApiEntryElement {
        this.assertAcceptableStaleness();
        return this.doc.getServerKeysCoveringApi(apiElement);
    }

    @Override
    public boolean isApiCoveredByServerKey(Element apiElement, RSAPublicKey serverKey) throws RegistryClient.UnacceptableStalenessException, RegistryClient.InvalidApiEntryElement {
        this.assertAcceptableStaleness();
        return this.doc.isApiCoveredByServerKey(apiElement, serverKey);
    }

    @Override
    public boolean isCertificateKnown(Certificate clientCert) {
        this.assertAcceptableStaleness();
        return this.doc.isCertificateKnown(clientCert);
    }

    @Override
    public boolean isClientKeyKnown(RSAPublicKey clientKey) throws RegistryClient.UnacceptableStalenessException {
        this.assertAcceptableStaleness();
        return this.doc.isClientKeyKnown(clientKey);
    }

    @Override
    public boolean isHeiCoveredByCertificate(String heiId, Certificate clientCert) {
        this.assertAcceptableStaleness();
        return this.areHeisCoveredByCertificate(new String[]{heiId}, clientCert);
    }

    @Override
    public boolean isHeiCoveredByClientKey(String heiId, RSAPublicKey clientKey) throws RegistryClient.UnacceptableStalenessException {
        this.assertAcceptableStaleness();
        return this.areHeisCoveredByClientKey(new String[]{heiId}, clientKey);
    }

    @Override
    public void refresh() throws RegistryClient.RefreshFailureException {
        CatalogueFetcher.RegistryResponse someResponse;
        logger.trace("Starting a new refresh call");
        CatalogueFetcher catalogueFetcher = this.options.getCatalogueFetcher();
        try {
            logger.trace("Fetching response from the catalogueFetcher");
            someResponse = catalogueFetcher.fetchCatalogue(this.doc.getETag());
            logger.trace("Response fetched successfully: " + someResponse.getClass());
        }
        catch (IOException e) {
            logger.debug("CatalogueFetcher has thrown an IOException", (Throwable)e);
            throw new RegistryClient.RefreshFailureException("Problem fetching the catalogue from server", e);
        }
        if (someResponse instanceof CatalogueFetcher.Http304RegistryResponse) {
            logger.info("Extending the expiry date of our catalogue copy: " + someResponse.getExpires());
            this.doc.extendExpiryDate(someResponse.getExpires());
            Map<String, byte[]> cache = this.options.getPersistentCacheMap();
            if (cache != null) {
                logger.trace("Trying to extend the expiry date of the cached copy too...");
                byte[] data = cache.get(CATALOGUE_CACHE_KEY);
                if (data != null) {
                    try {
                        CatalogueFetcher.Http200RegistryResponse oldCachedResponse = CatalogueFetcher.Http200RegistryResponse.deserialize(data);
                        CatalogueFetcher.Http200RegistryResponse newCachedResponse = new CatalogueFetcher.Http200RegistryResponse(oldCachedResponse.getContent(), oldCachedResponse.getETag(), this.doc.getExpiryDate());
                        cache.put(CATALOGUE_CACHE_KEY, newCachedResponse.serialize());
                        logger.trace("Successfully updated");
                    }
                    catch (CatalogueFetcher.Http200RegistryResponse.CouldNotDeserialize e) {
                        logger.info("Could not extend the expiry date of the cached copy", (Throwable)e);
                    }
                } else {
                    logger.debug("Cached copy not found");
                }
            }
            return;
        }
        if (someResponse instanceof CatalogueFetcher.Http200RegistryResponse) {
            logger.trace("Preparing a new catalogue copy");
            CatalogueFetcher.Http200RegistryResponse response = (CatalogueFetcher.Http200RegistryResponse)someResponse;
            try {
                this.doc = new CatalogueDocument(response);
                logger.info("Catalogue copy successfully updated: " + this.doc);
            }
            catch (CatalogueDocument.CatalogueParserException e) {
                logger.debug("Could not parse the new catalogue", (Throwable)e);
                throw new RegistryClient.RefreshFailureException(e);
            }
            Map<String, byte[]> cache = this.options.getPersistentCacheMap();
            if (cache != null) {
                logger.trace("Storing the new copy to cache...");
                cache.put(CATALOGUE_CACHE_KEY, response.serialize());
            }
        } else {
            throw new RuntimeException("CatalogueFetcher returned an unsupported RegistryResponse subclass: " + someResponse.getClass());
        }
    }

    private void assertAcceptableStaleness() {
        Date acceptableUntil = new Date(this.getExpiryDate().getTime() + this.options.getMaxAcceptableStaleness());
        if (new Date().after(acceptableUntil)) {
            throw new RegistryClient.UnacceptableStalenessException();
        }
    }

    private void logRefreshFailure(String message, RegistryClient.RefreshFailureException ex) {
        long age = new Date().getTime() - this.getExpiryDate().getTime();
        if (age > this.options.getStalenessWarningThreshold()) {
            logger.warn(message + ": " + ex);
        } else {
            logger.info(message + ": " + ex);
        }
    }
}

