package pushlibrary.utils;

import android.annotation.TargetApi;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.DialogInterface.OnClickListener;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.SharedPreferences.Editor;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.preference.PreferenceManager;
import android.provider.Settings.Secure;
import android.text.TextUtils;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.TextView;

import java.util.Arrays;
import java.util.Calendar;
import java.util.HashSet;
import java.util.Set;

import eu.easypush.pushlibrary.R;
import pushlibrary.constants.EasyPushConstants;
import pushlibrary.preferences.EasyPushMultiSelectListPreference;
import pushlibrary.preferences.EasypushPreferences;
import pushlibrary.tasks.AddDeviceDebugTask;
import pushlibrary.tasks.AddDeviceProjectTask;
import pushlibrary.tasks.RemoveDeviceDebugTask;
import pushlibrary.tasks.SetMessageReadTask;
import pushlibrary.tasks.SetMessageReceivedTask;

@TargetApi(Build.VERSION_CODES.HONEYCOMB)
public class PushUtils implements PushConstants {

	public static final String PROPERTY_REG_ID = "registration_id";
	public static final String PROPERTY_APP_VERSION = "appVersion";
	public static final String PROPERTY_GROUPS = "registrationGroups";
	public static final String PROPERTY_MODE = "easypush_mode";
	public static final String PROPERTY_APPLICATION_NAME = "easypushAppName";
	public static final String PROPERTY_PROJECT_TOKEN = "easypushProjectToken";
	public static final String PROPERTY_API_SERVER = "easypushApiServer";

	public static boolean isGlobalPushEnabled(Context context) {
		return true;
	}

	public static boolean isApplicationPushEnabled(Context context) {
		if (context == null) {
			Log.e(EasyPushConstants.TAG, "Context can not be null");
			return true;
		}

		SharedPreferences sp = PreferenceManager
				.getDefaultSharedPreferences(context);
		boolean isPushEnabledInPreference = sp.getBoolean(
				context.getString(R.string.easypush_alert_preference), false);
		if (!isPushEnabledInPreference) {
			return false;
		}

		Set<String> hoursNotificationDisabled = null;

		// Android > 11
		if (Build.VERSION.SDK_INT >= 11) {
			hoursNotificationDisabled = sp.getStringSet(
					context.getString(R.string.easypush_hours_preference),
					new HashSet<String>());
		} else {
			hoursNotificationDisabled = new HashSet<String>(
					Arrays.asList(sp
							.getString(
									context.getString(R.string.easypush_hours_preference),
									"")
							.split(EasyPushMultiSelectListPreference.DEFAULT_SEPARATOR)));
		}
		Calendar c = Calendar.getInstance();
		String hour = "" + c.get(Calendar.HOUR_OF_DAY);
		if (hoursNotificationDisabled.contains(hour)) {
			return false;
		}
		return true;
	}

	public static void setPushEnabled(Context context, boolean enabled) {
		if (context == null) {
			Log.e(EasyPushConstants.TAG, "Context can not be null");
			return;
		}

		SharedPreferences sp = PreferenceManager
				.getDefaultSharedPreferences(context);
		Editor editor = sp.edit();
		editor.putBoolean(
				context.getString(R.string.easypush_alert_preference), enabled);
		editor.commit();
	}

	public static Uri getPushSoundUri(Context context) {
		if (context == null) {
			Log.e(EasyPushConstants.TAG, "Context can not be null");
			return null;
		}

		SharedPreferences sp = PreferenceManager
				.getDefaultSharedPreferences(context);
		String defaultSoundUri = sp.getString(
				context.getString(R.string.easypush_sound_preference), null);
		if (defaultSoundUri == null) {
			return RingtoneManager.getActualDefaultRingtoneUri(context,
					RingtoneManager.TYPE_NOTIFICATION);

		}
		return Uri.parse(defaultSoundUri);
	}

	public static boolean isUserNotificationChoiceDone(Context context) {
		if (context == null) {
			Log.e(EasyPushConstants.TAG, "Context can not be null");
			return false;
		}
		SharedPreferences preferences = PreferenceManager
				.getDefaultSharedPreferences(context);
		return preferences.getBoolean(
				context.getString(R.string.isusernotificationchoicedone_key),
				false);
	}

	public static String getProjectToken(Context context) {
		if (context == null) {
			Log.e(EasyPushConstants.TAG, "Context can not be null");
			return "";
		}

		final SharedPreferences prefs = getGCMPreferences(context);
		if (prefs.contains(PROPERTY_PROJECT_TOKEN)) {
			return prefs.getString(PROPERTY_PROJECT_TOKEN, "");
		}
		try {
			ApplicationInfo ai = context.getPackageManager()
					.getApplicationInfo(context.getPackageName(),
							PackageManager.GET_META_DATA);
			Bundle bundle = ai.metaData;
			String easyPushProject = bundle.getString("EASYPUSH_PROJECT");
			if (!TextUtils.isEmpty(easyPushProject)) {
				return easyPushProject;
			}

		} catch (NameNotFoundException e) {
		} catch (NullPointerException e) {
		}
		return "";
	}

	public static void setEPApplicationName(Context context,
			String applicationName) {
		if (context == null || applicationName == null) {
			Log.e(EasyPushConstants.TAG,
					"Context or application Name can not be null");
			return;
		}

		final SharedPreferences prefs = getGCMPreferences(context);
		Editor editor = prefs.edit();
		editor.putString(PROPERTY_APPLICATION_NAME, applicationName);
		editor.commit();
	}

	public static String getApplicationName(Context context) {
		if (context == null) {
			Log.e(EasyPushConstants.TAG, "Context can not be null");
			return "";
		}

		final SharedPreferences prefs = getGCMPreferences(context);
		if (prefs.contains(PROPERTY_APPLICATION_NAME)) {
			return prefs.getString(PROPERTY_APPLICATION_NAME, "");
		}

		try {
			ApplicationInfo ai = context.getPackageManager()
					.getApplicationInfo(context.getPackageName(),
							PackageManager.GET_META_DATA);
			Bundle bundle = ai.metaData;
			String easyPushApplication = bundle
					.getString("EASYPUSH_APPLICATION");
			if (!TextUtils.isEmpty(easyPushApplication)) {
				return easyPushApplication;
			}

		} catch (NameNotFoundException e) {
		} catch (NullPointerException e) {
		}
		return "";
	}

	public static void setEPMode(Context context, boolean isProductionMode) {
		final SharedPreferences prefs = getGCMPreferences(context);
		Editor editor = prefs.edit();
		editor.putBoolean(PROPERTY_MODE, isProductionMode);
		editor.commit();
	}

	public static void setEPProjectToken(Context context, String projectToken) {
		if (context == null || projectToken == null) {
			Log.e(EasyPushConstants.TAG,
					"Context or projectToken can not be null");
			return;
		}

		final SharedPreferences prefs = getGCMPreferences(context);
		Editor editor = prefs.edit();
		editor.putString(PROPERTY_PROJECT_TOKEN, projectToken);
		editor.commit();
	}

	public static boolean isProductionMode(Context context) {
		final SharedPreferences prefs = getGCMPreferences(context);
		if (prefs.contains(PROPERTY_MODE)) {
			return prefs.getBoolean(PROPERTY_MODE, true);
		}

		try {
			ApplicationInfo ai = context.getPackageManager()
					.getApplicationInfo(context.getPackageName(),
							PackageManager.GET_META_DATA);
			Bundle bundle = ai.metaData;
			String easypushMode = bundle.getString("EASYPUSH_MODE");
			if (!TextUtils.isEmpty(easypushMode)) {
				return easypushMode == "PRODUCTION";
			}

		} catch (NameNotFoundException e) {
		} catch (NullPointerException e) {
		}
		return true;
	}

	public static String getDeviceId(Context context) {
		return Secure
				.getString(context.getContentResolver(), Secure.ANDROID_ID);
	}

	/**
	 * Gets the current registration ID for application on GCM service.
	 * <p>
	 * If result is empty, the app needs to register.
	 * 
	 * @return registration ID, or empty string if there is no existing
	 *         registration ID.
	 */
	public static String getRegistrationId(Context context) {
		final SharedPreferences prefs = getGCMPreferences(context);
		String registrationId = prefs.getString(PROPERTY_REG_ID, "");
		if (TextUtils.isEmpty(registrationId)) {
			Log.i(TAG, "Registration not found.");
			return "";
		}
		// Check if app was updated; if so, it must clear the registration ID
		// since the existing regID is not guaranteed to work with the new
		// app version.
		int registeredVersion = prefs.getInt(PROPERTY_APP_VERSION,
				Integer.MIN_VALUE);
		int currentVersion = getAppVersion(context);
		if (registeredVersion != currentVersion) {
			Log.i(TAG, "App version changed.");
			return "";
		}
		return registrationId;
	}

	/**
	 * @return Application's {@code SharedPreferences}.
	 */
	public static SharedPreferences getGCMPreferences(Context context) {
		// This sample app persists the registration ID in shared preferences,
		// but
		// how you store the regID in your app is up to you.
		return context.getSharedPreferences(context.getPackageName(),
				Context.MODE_PRIVATE);
	}

	/**
	 * @return Application's version code from the {@code PackageManager}.
	 */
	public static int getAppVersion(Context context) {
		try {
			PackageInfo packageInfo = context.getPackageManager()
					.getPackageInfo(context.getPackageName(), 0);
			return packageInfo.versionCode;
		} catch (NameNotFoundException e) {
			// should never happen
			throw new RuntimeException("Could not get package name: " + e);
		}
	}

	public static void setGroupsForPush(Context c, String... groups) {
		if (groups != null) {
			final SharedPreferences prefs = PushUtils.getGCMPreferences(c);
			Editor editor = prefs.edit();
			editor.putString(PushUtils.PROPERTY_GROUPS,
					TextUtils.join(",", groups));
			editor.commit();
		} else {
			Log.w(EasyPushConstants.TAG,
					"Warning : groups are null, all groups will be deleted for this user");
			final SharedPreferences prefs = PushUtils.getGCMPreferences(c);
			Editor editor = prefs.edit();
			editor.remove(PushUtils.PROPERTY_GROUPS);
			editor.commit();

		}
	}

	public static void modifyGroupsForPush(Context c, String... groups) {
		if (groups != null) {
			setGroupsForPush(c, groups);
			new AddDeviceProjectTask(c).execute();
		}
	}

	public static String[] getGroupsForPush(Context c) {
		final SharedPreferences prefs = PushUtils.getGCMPreferences(c);
		return prefs.getString(PushUtils.PROPERTY_GROUPS, "").split(",");
	}

	public static void setEPServerAdress(Context context, String serverAdress) {

		final SharedPreferences prefs = getGCMPreferences(context);
		Editor editor = prefs.edit();
		editor.putString(PROPERTY_API_SERVER, serverAdress);
		editor.commit();
	}

	public static String getServerAdress(Context context) {
		final SharedPreferences prefs = getGCMPreferences(context);
		if (prefs.contains(PROPERTY_API_SERVER)) {
			return prefs.getString(PROPERTY_API_SERVER, "api.easypush.eu");
		}
		try {
			ApplicationInfo ai = context.getPackageManager()
					.getApplicationInfo(context.getPackageName(),
							PackageManager.GET_META_DATA);
			Bundle bundle = ai.metaData;
			String apiServer = bundle.getString("EASYPUSH_API_SERVER");
			if (!TextUtils.isEmpty(apiServer)) {
				return apiServer;
			}
		} catch (NameNotFoundException e) {
		} catch (NullPointerException e) {
		}
		return "api.easypush.eu";
	}

	public static void launchPreferenceActivity(Context context) {
		context.startActivity(new Intent(context, EasypushPreferences.class));
	}

	public static void addDeviceDebug(Context _context) {
		new AddDeviceDebugTask(_context).execute();
	}

	public static void removeDeviceDebug(Context _context) {
		new RemoveDeviceDebugTask(_context).execute();
	}

	public static void setMessageRead(Context _context, String _messageId) {
		new SetMessageReadTask(_context, _messageId).execute();
	}

	public static void setMessageReceived(Context _context, String _messageId) {
		new SetMessageReceivedTask(_context, _messageId).execute();
	}

	public static void askUserForPush(final Context c) {
		final SharedPreferences sp = PreferenceManager
				.getDefaultSharedPreferences(c);
		AlertDialog.Builder b = new AlertDialog.Builder(c);
		b.setTitle(R.string.easypush_account_name);

		View dialogLayout = LayoutInflater.from(c).inflate(
				R.layout.easypush_alert_dialog, null);
		b.setView(dialogLayout);

		PackageInfo pinfo;
		try {
			pinfo = c.getPackageManager().getPackageInfo(c.getPackageName(), 0);
			((TextView) dialogLayout.findViewById(R.id.activatePushQuestionId))
					.setText(c.getResources().getString(
							R.string.activatePushQuestion,
							pinfo.applicationInfo.loadLabel(c
									.getPackageManager())));
		} catch (NameNotFoundException e) {
		}

		if (TextUtils.isEmpty(c.getString(R.string.activatePushExplanation))) {
			dialogLayout.findViewById(R.id.explanation_block).setVisibility(
					View.GONE);
		} else {
			dialogLayout.findViewById(R.id.explanation_block).setVisibility(
					View.VISIBLE);
		}
		b.setPositiveButton(android.R.string.yes, new OnClickListener() {

			@Override
			public void onClick(DialogInterface dialog, int which) {
				Editor editor = sp.edit();
				editor.putBoolean(
						c.getString(R.string.easypush_alert_preference), true);

				editor.putBoolean("Easypush_Notification_Preference_set", true);
				editor.commit();
				new AddDeviceProjectTask(c).execute();
			}
		});

		b.setNegativeButton(android.R.string.no, new OnClickListener() {

			@Override
			public void onClick(DialogInterface dialog, int which) {
				Editor editor = sp.edit();
				editor.putBoolean(
						c.getString(R.string.easypush_alert_preference), false);
				editor.putBoolean("Easypush_Notification_Preference_set", true);
				editor.commit();
				new AddDeviceProjectTask(c).execute();
			}
		});
		b.setCancelable(false);
		b.show();

	}
}
