/**
 * Copyright (C) 2010, 2011 Neofonie GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package eu.dicodeproject.analysis.lucene;

import eu.dicodeproject.analysis.util.Language;

import org.apache.lucene.analysis.PorterStemFilter;
import org.apache.lucene.analysis.TokenFilter;
import org.apache.lucene.analysis.TokenStream;
import org.apache.lucene.analysis.de.GermanStemFilter;
import org.apache.lucene.analysis.id.IndonesianStemFilter;
import org.apache.lucene.analysis.snowball.SnowballFilter;
import org.apache.lucene.analysis.standard.StandardFilter;
import org.apache.lucene.util.Version;
import org.tartarus.snowball.ext.DutchStemmer;
import org.tartarus.snowball.ext.FrenchStemmer;
import org.tartarus.snowball.ext.German2Stemmer;
import org.tartarus.snowball.ext.ItalianStemmer;
import org.tartarus.snowball.ext.PortugueseStemmer;
import org.tartarus.snowball.ext.RussianStemmer;
import org.tartarus.snowball.ext.SpanishStemmer;
import org.tartarus.snowball.ext.TurkishStemmer;

/**
 * @author jakob
 */
public class StemmerFactory {

  public static TokenFilter getStemmer(Language language, Version matchVersion, TokenStream result) {
    if (language == Language.ENGLISH) {
      return new PorterStemFilter(result);
    }
    else if (language == Language.SPANISH) {
      return new SnowballFilter(result, new SpanishStemmer());
    }
    else if (language == Language.PORTUGUESE) {
      return new SnowballFilter(result, new PortugueseStemmer());
    }
    else if (language == Language.FRENCH) {
      return new SnowballFilter(result, new FrenchStemmer());
    }
    else if (language == Language.DUTCH) {
      return new SnowballFilter(result, new DutchStemmer());
    }
    else if (language == Language.GERMAN) {
      if (matchVersion.onOrAfter(Version.LUCENE_31)) {
        return new SnowballFilter(result, new German2Stemmer());
      }
      else {
        return new GermanStemFilter(result);
      }
    }
    else if (language == Language.INDONESIAN) {
      return new IndonesianStemFilter(result);
    }
    else if (language == Language.ITALIAN) {
      return new SnowballFilter(result, new ItalianStemmer());
    }
    else if (language == Language.RUSSIAN) {
      return new SnowballFilter(result, new RussianStemmer());
    }
    else if (language == Language.TURKISH) {
      return new SnowballFilter(result, new TurkishStemmer());
    }

    return new StandardFilter(matchVersion, result);
  }

}
