/**
 * Copyright (C) 2010, 2011 Neofonie GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.dicodeproject.analysis.histogram;

import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.mapreduce.TableReducer;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.IntWritable;
import org.apache.hadoop.io.Text;

import java.io.IOException;

/**
 * Sums all the values coming from the mapper. Splits the key and writes
 * the first part as column qualifier and the second part as row key. The
 * sum of values is the cell value.
 */
public class DateHistogramReducer extends TableReducer<Text, IntWritable, Text> {

  private String splitExpression = DateHistogramDriver.splitExpression;

  private Text text = new Text();
  private String family = "d";

  /**
   * Add up data and write to table.
   */
  @Override
  protected void reduce(Text keyin, Iterable<IntWritable> values, Context context) throws IOException, InterruptedException {
    String[] elements = keyin.toString().split(splitExpression);
    String dateType = elements[0];  // year / yearMonth / yearMonthDate ...
    String dateValue = elements[1];

    int sum = 0;
    for (IntWritable i : values) {
      sum += i.get();
    }

    writeRow(context, dateValue, dateType, String.valueOf(sum));
  }

  /**
   * Write row to table.
   * @param context MapReduce context
   * @param rowKey to write in table
   * @param qualifier to write in table
   * @param value to write in table
   * @throws IOException inherited from Context.write
   * @throws InterruptedException inherited from Context.write
   */
  private void writeRow(Context context, String rowKey, String qualifier, String value) throws IOException, InterruptedException {
    Put put = new Put(Bytes.toBytes(rowKey));
    put.add(Bytes.toBytes(family), Bytes.toBytes(qualifier), Bytes.toBytes(value));
    text.set(rowKey);
    context.write(text, put);
  }



}
