/**
 * Copyright (C) 2010, 2011 Neofonie GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.dicodeproject.analysis.histogram;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.mapreduce.TableMapReduceUtil;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.IntWritable;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.util.ToolRunner;
import org.apache.mahout.common.AbstractJob;

import java.io.IOException;
import java.util.Map;

/**
 * Generates an HBase table that contains creation date information for all tweets.
 */
public class DateHistogramDriver extends AbstractJob {

  public static final String splitExpression = "_";

  public static void main(String args[]) throws Exception {
    ToolRunner.run(new DateHistogramDriver(), args);
  }

  @Override
  public int run(String[] args) throws ClassNotFoundException, InterruptedException, IOException {
    addOption("inputTable",   "i",  "The hbase table holding our data.",                            "thtweets");
    addOption("family",       "f",  "The column family holding our data.",                          "d");
    addOption("column",       "c",  "The column qualifier holding our creationDate.",               "creationDate");
    addOption("outputTable",  "t",  "The resulting hbase table in which the histogram is written.", "thtweets-creationDates");

    Map<String, String> argMap = parseArguments(args);
    if (argMap == null) {
      return -1;
    }

    String inputTable = argMap.get("--inputTable");
    String dataFamily = argMap.get("--family");
    String timeColumn = argMap.get("--column");
    String outputTable = argMap.get("--outputTable");

    boolean success = generateDateHistogramData(inputTable, dataFamily, timeColumn, outputTable);

    return (success ? 0 : 1);
  }

  private boolean generateDateHistogramData(String inputTable, String dataFamily, String timeCol, String outputTable)
      throws IOException, InterruptedException, ClassNotFoundException {

    Configuration conf = HBaseConfiguration.create();
    Job job = new Job(conf, "CreationDateDriver::GenerateHistogramData");
    job.setJarByClass(DateHistogramDriver.class);

    Scan scan = new Scan();
    scan.addColumn(Bytes.toBytes(dataFamily), Bytes.toBytes(timeCol));
    scan.setMaxVersions(1);

    TableMapReduceUtil.initTableMapperJob(inputTable, scan, DateHistogramMapper.class, Text.class, IntWritable.class, job);
    TableMapReduceUtil.initTableReducerJob(outputTable, DateHistogramReducer.class, job);

    job.setNumReduceTasks(10);

    return job.waitForCompletion(true);
  }

}
