/**
 * Copyright (C) 2010, 2011 Neofonie GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.dicodeproject.analysis.generic;

import java.io.IOException;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.Map;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.filter.CompareFilter;
import org.apache.hadoop.hbase.filter.RegexStringComparator;
import org.apache.hadoop.hbase.filter.ValueFilter;
import org.apache.hadoop.hbase.mapreduce.TableMapReduceUtil;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.IntWritable;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.lib.output.FileOutputFormat;
import org.apache.hadoop.mapreduce.lib.output.SequenceFileOutputFormat;
import org.apache.hadoop.util.ToolRunner;
import org.apache.mahout.common.AbstractJob;
import org.apache.mahout.common.commandline.DefaultOptionCreator;

/**
 * Simple "word count" for HBase columns: Aggregates values from a configurable
 * HBase table and column TODO: add Filters, e.g. for language etc.
 * 
 */
public final class GenericDriver extends AbstractJob {

  private static final String NONE = "none";

  private GenericDriver() {
    // don't instantiate drivers
  }

  public static void main(String args[]) throws Exception {
    ToolRunner.run(new GenericDriver(), args);
  }

  @Override
  public int run(String[] args) throws ClassNotFoundException, InterruptedException, IOException {
    addOutputOption();
    addOption(DefaultOptionCreator.numReducersOption().create());
    addOption("table", "t", "The hbase table holding our data.", "thtweets");
    addOption("family", "f", "The column family holding our data.", "d");
    addOption("column", "c", "The column holding our data.", "lang");
    addOption("separator", "s", "Separator used for splitting in column", NONE);
    addOption("query", "q", "The query, a simple string representing the topic.", "");
    addOption("daysBack", "d", "Number of Days - starting from last midnight - we want to scan", "1");

    Map<String, String> argMap = parseArguments(args);
    if (argMap == null) {
      return -1;
    }
    String table = argMap.get("--table");
    String family = argMap.get("--family");
    String column = argMap.get("--column");
    String separator = argMap.get("--separator");
    String query = argMap.get("--query");
    String daysBack = argMap.get("--daysBack");

    Path output = getOutputPath();

    Configuration conf = HBaseConfiguration.create();

    if (separator != NONE) {
      conf.set("separator", argMap.get("--separator")); // for the reducer
    }

    Job job = new Job(conf);
    job.setJarByClass(GenericDriver.class);

    Scan scan = new Scan();
    scan.addColumn(Bytes.toBytes(family), Bytes.toBytes(column));

    if (query != null && !query.equals("")) {
      scan.setFilter(new ValueFilter(CompareFilter.CompareOp.EQUAL, new RegexStringComparator(query)));
    }
    
    if (daysBack != null && !daysBack.equals("")) {
      try {
        Calendar cal = this.getEndOfTimeRange();
        long maxStamp = cal.getTimeInMillis();
        int days = Integer.parseInt(daysBack);
        cal.add(Calendar.DAY_OF_MONTH, days * -1);
        long minStamp = cal.getTimeInMillis();
        scan.setTimeRange(minStamp, maxStamp);
      } catch (NumberFormatException e) {
        System.out.println("Error: "+e.getMessage());
      }
    }
    TableMapReduceUtil.initTableMapperJob(table, scan, GenericMapper.class, Text.class, IntWritable.class,
        job);

    job.setJobName("GenericCounter::" + column); 

    job.setMapOutputKeyClass(Text.class);
    job.setMapOutputValueClass(IntWritable.class);

    job.setReducerClass(GenericReducer.class);
    job.setOutputKeyClass(IntWritable.class);
    job.setOutputValueClass(Text.class);

    job.setOutputFormatClass(SequenceFileOutputFormat.class);
    FileOutputFormat.setOutputPath(job, output);
    job.setNumReduceTasks(1);
    job.waitForCompletion(true);
    return 0;
  }

  /**
   * Returns a calender set to the start of today (midnight)
   * 
   * @return calendar set to end of time range
   */
  private Calendar getEndOfTimeRange() {
    Date d = new Date();
    Calendar cal = Calendar.getInstance(Locale.GERMANY);
    cal.setTime(d);
    cal.set(Calendar.HOUR_OF_DAY, 0);
    cal.set(Calendar.MINUTE, 0);
    cal.set(Calendar.SECOND, 0);
    cal.set(Calendar.MILLISECOND, 0);
    return cal;
  }

}