/**
 * Copyright (C) 2010, 2011 Neofonie GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.dicodeproject.analysis.export;

import java.io.IOException;
import java.util.Map;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.filter.CompareFilter;
import org.apache.hadoop.hbase.filter.RegexStringComparator;
import org.apache.hadoop.hbase.filter.ValueFilter;
import org.apache.hadoop.hbase.mapreduce.TableMapReduceUtil;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Job;
import org.apache.hadoop.mapreduce.Reducer;
import org.apache.hadoop.mapreduce.lib.output.FileOutputFormat;
import org.apache.hadoop.mapreduce.lib.output.SequenceFileOutputFormat;
import org.apache.hadoop.util.ToolRunner;
import org.apache.mahout.common.AbstractJob;
import org.apache.mahout.common.commandline.DefaultOptionCreator;

/**
 * Reads text from a configurable HBase table and column and writes row key and
 * column content to HDFS.
 */
public final class TwitterExportDriver extends AbstractJob {

  private TwitterExportDriver() {
    // don't instantiate drivers
  }

  public static void main(String args[]) throws Exception {
    ToolRunner.run(new TwitterExportDriver(), args);
  }

  @Override
  public int run(String[] args) throws IOException, ClassNotFoundException, InterruptedException {
    addOutputOption();
    addOption(DefaultOptionCreator.numReducersOption().create());
    addOption("table", "t", "The hbase table holding our data.", "twittertracker");
    addOption("family", "f", "The column family holding our data.", "textFamily");
    addOption("column", "c", "The column holding our data.", "text");
    addOption("query", "q", "The query, a simple string representing the topic.", "Dicode");

    Map<String, String> argMap = parseArguments(args);
    if (argMap == null) {
      return -1;
    }
    String table = argMap.get("--table");
    String family = argMap.get("--family");
    String column = argMap.get("--column");
    String query = argMap.get("--query");

    Path output = getOutputPath();

    Configuration conf = HBaseConfiguration.create();
    Job job = new Job(conf);
    job.setJarByClass(TwitterExportDriver.class);

    Scan scan = new Scan();
    scan.addColumn(Bytes.toBytes(family), Bytes.toBytes(column));

    // TODO: check URL encoding
    scan.setFilter(new ValueFilter(CompareFilter.CompareOp.EQUAL, new RegexStringComparator(query)));
    // scan.setFilter(new ValueFilter(CompareFilter.CompareOp.EQUAL, new
    // RegexStringComparator("("+query+"[\\s\\.,;?!]+) | (" +query + "^)")));

    TableMapReduceUtil.initTableMapperJob(table, scan, TwitterExportMapper.class, Text.class, Text.class, job);

    job.setJobName("TwitterExport");
    job.setReducerClass(Reducer.class);
    job.setOutputKeyClass(Text.class);
    job.setOutputValueClass(Text.class);

    job.setOutputFormatClass(SequenceFileOutputFormat.class);
    FileOutputFormat.setOutputPath(job, output);
    job.setNumReduceTasks(1);
    job.waitForCompletion(true);
    return 0;
  }
}