/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.math.BigInteger;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.TreeMap;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HRegionLocation;
import org.apache.hadoop.hbase.HServerAddress;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.NoServerForRegionException;
import org.apache.hadoop.hbase.regionserver.Store;
import org.apache.hadoop.hbase.regionserver.StoreFile;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.Pair;

public class RegionSplitter {
    static final Log LOG = LogFactory.getLog(RegionSplitter.class);

    public static void main(String[] args) throws IOException, InterruptedException, ParseException {
        Configuration conf = HBaseConfiguration.create();
        Options opt = new Options();
        OptionBuilder.withArgName((String)"property=value");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Override HBase Configuration Settings");
        opt.addOption(OptionBuilder.create((String)"D"));
        OptionBuilder.withArgName((String)"region count");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Create a new table with a pre-split number of regions");
        opt.addOption(OptionBuilder.create((String)"c"));
        OptionBuilder.withArgName((String)"family:family:...");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Column Families to create with new table.  Required with -c");
        opt.addOption(OptionBuilder.create((String)"f"));
        opt.addOption("h", false, "Print this usage help");
        opt.addOption("r", false, "Perform a rolling split of an existing region");
        OptionBuilder.withArgName((String)"count");
        OptionBuilder.hasArg();
        OptionBuilder.withDescription((String)"Max outstanding splits that have unfinished major compactions");
        opt.addOption(OptionBuilder.create((String)"o"));
        opt.addOption(null, "risky", false, "Skip verification steps to complete quickly.STRONGLY DISCOURAGED for production systems.  ");
        CommandLine cmd = new GnuParser().parse(opt, args);
        if (cmd.hasOption("D")) {
            for (String confOpt : cmd.getOptionValues("D")) {
                String[] kv = confOpt.split("=", 2);
                if (kv.length != 2) {
                    throw new ParseException("-D option format invalid: " + confOpt);
                }
                conf.set(kv[0], kv[1]);
                LOG.debug((Object)("-D configuration override: " + kv[0] + "=" + kv[1]));
            }
        }
        if (cmd.hasOption("risky")) {
            conf.setBoolean("split.verify", false);
        }
        boolean createTable2 = cmd.hasOption("c") && cmd.hasOption("f");
        boolean rollingSplit = cmd.hasOption("r");
        boolean oneOperOnly = createTable2 ^ rollingSplit;
        if (1 != cmd.getArgList().size() || !oneOperOnly || cmd.hasOption("h")) {
            new HelpFormatter().printHelp("RegionSplitter <TABLE>", opt);
            return;
        }
        String tableName = cmd.getArgs()[0];
        if (createTable2) {
            conf.set("split.count", cmd.getOptionValue("c"));
            RegionSplitter.createPresplitTable(tableName, cmd.getOptionValue("f").split(":"), conf);
        }
        if (rollingSplit) {
            if (cmd.hasOption("o")) {
                conf.set("split.outstanding", cmd.getOptionValue("o"));
            }
            RegionSplitter.rollingSplit(tableName, conf);
        }
    }

    static void createPresplitTable(String tableName, String[] columnFamilies, Configuration conf) throws IOException, InterruptedException {
        SplitAlgorithm splitAlgo;
        Class splitClass = conf.getClass("split.algorithm", MD5StringSplit.class, SplitAlgorithm.class);
        try {
            splitAlgo = (SplitAlgorithm)splitClass.newInstance();
        }
        catch (Exception e) {
            throw new IOException("Problem loading split algorithm: ", e);
        }
        int splitCount = conf.getInt("split.count", 0);
        Preconditions.checkArgument((splitCount > 1 ? 1 : 0) != 0, (Object)"Split count must be > 1");
        Preconditions.checkArgument((columnFamilies.length > 0 ? 1 : 0) != 0, (Object)"Must specify at least one column family. ");
        LOG.debug((Object)("Creating table " + tableName + " with " + columnFamilies.length + " column families.  Presplitting to " + splitCount + " regions"));
        HTableDescriptor desc = new HTableDescriptor(tableName);
        for (String cf : columnFamilies) {
            desc.addFamily(new HColumnDescriptor(Bytes.toBytes(cf)));
        }
        HBaseAdmin admin = new HBaseAdmin(conf);
        Preconditions.checkArgument((!admin.tableExists(tableName) ? 1 : 0) != 0, (Object)("Table already exists: " + tableName));
        admin.createTable(desc, splitAlgo.split(splitCount));
        LOG.debug((Object)"Table created!  Waiting for regions to show online in META...");
        if (!conf.getBoolean("split.verify", true)) {
            HTable table = new HTable(tableName);
            int onlineRegions = 0;
            while (onlineRegions < splitCount) {
                onlineRegions = table.getRegionsInfo().size();
                LOG.debug((Object)(onlineRegions + " of " + splitCount + " regions online..."));
                if (onlineRegions >= splitCount) continue;
                Thread.sleep(10000L);
            }
        }
        LOG.debug((Object)("Finished creating table with " + splitCount + " regions"));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void rollingSplit(String tableName, Configuration conf) throws IOException, InterruptedException {
        SplitAlgorithm splitAlgo;
        Class splitClass = conf.getClass("split.algorithm", MD5StringSplit.class, SplitAlgorithm.class);
        try {
            splitAlgo = (SplitAlgorithm)splitClass.newInstance();
        }
        catch (Exception e) {
            throw new IOException("Problem loading split algorithm: ", e);
        }
        int minOS = conf.getInt("split.outstanding", 2);
        HTable table = new HTable(conf, tableName);
        int MAX_OUTSTANDING = Math.max(table.getCurrentNrHRS() / 2, minOS);
        Path hbDir = new Path(conf.get("hbase.rootdir"));
        Path tableDir = HTableDescriptor.getTableDir(hbDir, table.getTableName());
        Path splitFile = new Path(tableDir, "_balancedSplit");
        FileSystem fs = FileSystem.get((Configuration)conf);
        LinkedList<Pair<byte[], byte[]>> tmpRegionSet = RegionSplitter.getSplits(table, splitAlgo);
        LinkedList outstanding = Lists.newLinkedList();
        int splitCount = 0;
        int origCount = tmpRegionSet.size();
        LOG.debug((Object)"Bucketing regions by regionserver...");
        TreeMap daughterRegions = Maps.newTreeMap();
        for (Pair pair : tmpRegionSet) {
            HServerAddress rsLocation = table.getRegionLocation((byte[])pair.getSecond()).getServerAddress();
            if (!daughterRegions.containsKey(rsLocation)) {
                LinkedList entry = Lists.newLinkedList();
                daughterRegions.put(rsLocation, entry);
            }
            ((LinkedList)daughterRegions.get(rsLocation)).add(pair);
        }
        LOG.debug((Object)"Done with bucketing.  Split time!");
        long startTime = System.currentTimeMillis();
        FSDataInputStream tmpIn = fs.open(splitFile);
        byte[] rawData = new byte[tmpIn.available()];
        tmpIn.readFully(rawData);
        tmpIn.close();
        FSDataOutputStream splitOut = fs.create(splitFile);
        splitOut.write(rawData);
        try {
            while (!daughterRegions.isEmpty()) {
                LOG.debug((Object)(daughterRegions.size() + " RS have regions to splt."));
                HServerAddress rsLoc = (HServerAddress)daughterRegions.firstKey();
                while (rsLoc != null) {
                    byte[] split;
                    Pair dr = null;
                    LOG.debug((Object)("Finding a region on " + rsLoc));
                    LinkedList linkedList = (LinkedList)daughterRegions.get(rsLoc);
                    while (!linkedList.isEmpty()) {
                        dr = (Pair)linkedList.pop();
                        split = (byte[])dr.getSecond();
                        HRegionLocation regionLoc = table.getRegionLocation(split);
                        HServerAddress newRs = regionLoc.getServerAddress();
                        if (newRs.compareTo(rsLoc) != 0) {
                            LOG.debug((Object)("Region with " + splitAlgo.rowToStr(split) + " moved to " + newRs + ". Relocating..."));
                            if (!daughterRegions.containsKey(newRs)) {
                                LinkedList entry = Lists.newLinkedList();
                                daughterRegions.put(newRs, entry);
                            }
                            ((LinkedList)daughterRegions.get(newRs)).add(dr);
                            dr = null;
                            continue;
                        }
                        byte[] sk = regionLoc.getRegionInfo().getStartKey();
                        if (sk.length == 0) break;
                        if (Bytes.equals(split, sk)) {
                            LOG.debug((Object)("Region already split on " + splitAlgo.rowToStr(split) + ".  Skipping this region..."));
                            ++splitCount;
                            dr = null;
                            continue;
                        }
                        byte[] byArray = (byte[])dr.getFirst();
                        Preconditions.checkArgument((boolean)Bytes.equals(byArray, sk), (Object)(splitAlgo.rowToStr(byArray) + " != " + splitAlgo.rowToStr(sk)));
                        break;
                    }
                    if (linkedList.isEmpty()) {
                        daughterRegions.remove(rsLoc);
                    }
                    if (dr != null) {
                        split = (byte[])dr.getSecond();
                        LOG.debug((Object)("Splitting at " + splitAlgo.rowToStr(split)));
                        HBaseAdmin admin = new HBaseAdmin(table.getConfiguration());
                        admin.split(table.getTableName(), split);
                        LinkedList<Pair<byte[], byte[]>> finished = Lists.newLinkedList();
                        if (conf.getBoolean("split.verify", true)) {
                            outstanding.addLast(dr);
                            while (outstanding.size() >= MAX_OUTSTANDING) {
                                finished = RegionSplitter.splitScan(outstanding, table, splitAlgo);
                                if (finished.isEmpty()) {
                                    Thread.sleep(30000L);
                                    continue;
                                }
                                outstanding.removeAll(finished);
                            }
                        } else {
                            finished.add(dr);
                        }
                        for (Pair pair : finished) {
                            splitOut.writeChars("- " + splitAlgo.rowToStr((byte[])pair.getFirst()) + " " + splitAlgo.rowToStr((byte[])pair.getSecond()) + "\n");
                            if (++splitCount % 10 != 0) continue;
                            long tDiff = (System.currentTimeMillis() - startTime) / (long)splitCount;
                            LOG.debug((Object)("STATUS UPDATE: " + splitCount + " / " + origCount + ". Avg Time / Split = " + org.apache.hadoop.util.StringUtils.formatTime((long)tDiff)));
                        }
                    }
                    rsLoc = daughterRegions.higherKey(rsLoc);
                }
            }
            if (conf.getBoolean("split.verify", true)) {
                while (!outstanding.isEmpty()) {
                    LinkedList<Pair<byte[], byte[]>> finished = RegionSplitter.splitScan(outstanding, table, splitAlgo);
                    if (finished.isEmpty()) {
                        Thread.sleep(30000L);
                        continue;
                    }
                    outstanding.removeAll(finished);
                    for (Pair pair : finished) {
                        splitOut.writeChars("- " + splitAlgo.rowToStr((byte[])pair.getFirst()) + " " + splitAlgo.rowToStr((byte[])pair.getSecond()) + "\n");
                    }
                }
            }
            LOG.debug((Object)"All regions have been sucesfully split!");
        }
        finally {
            long tDiff = System.currentTimeMillis() - startTime;
            LOG.debug((Object)("TOTAL TIME = " + org.apache.hadoop.util.StringUtils.formatTime((long)tDiff)));
            LOG.debug((Object)("Splits = " + splitCount));
            LOG.debug((Object)("Avg Time / Split = " + org.apache.hadoop.util.StringUtils.formatTime((long)(tDiff / (long)splitCount))));
            splitOut.close();
        }
        fs.delete(splitFile, false);
    }

    static LinkedList<Pair<byte[], byte[]>> splitScan(LinkedList<Pair<byte[], byte[]>> regionList, HTable table, SplitAlgorithm splitAlgo) throws IOException, InterruptedException {
        LinkedList finished = Lists.newLinkedList();
        LinkedList logicalSplitting = Lists.newLinkedList();
        LinkedList physicalSplitting = Lists.newLinkedList();
        Path hbDir = new Path(table.getConfiguration().get("hbase.rootdir"));
        Path tableDir = HTableDescriptor.getTableDir(hbDir, table.getTableName());
        Path splitFile = new Path(tableDir, "_balancedSplit");
        FileSystem fs = FileSystem.get((Configuration)table.getConfiguration());
        table.clearRegionCache();
        for (Pair pair : regionList) {
            byte[] start = (byte[])pair.getFirst();
            byte[] split = (byte[])pair.getSecond();
            HRegionInfo dri = table.getRegionLocation(split).getRegionInfo();
            if (dri.isOffline() || !Bytes.equals(dri.getStartKey(), split)) {
                logicalSplitting.add(pair);
                continue;
            }
            try {
                LinkedList check = Lists.newLinkedList();
                check.add(table.getRegionLocation(start).getRegionInfo());
                check.add(table.getRegionLocation(split).getRegionInfo());
                for (HRegionInfo hri : check.toArray(new HRegionInfo[0])) {
                    boolean refFound = false;
                    byte[] sk = hri.getStartKey();
                    if (sk.length == 0) {
                        sk = splitAlgo.firstRow();
                    }
                    String startKey = splitAlgo.rowToStr(sk);
                    for (HColumnDescriptor c : hri.getTableDesc().getFamilies()) {
                        Path cfDir = Store.getStoreHomedir(tableDir, hri.getEncodedName(), c.getName());
                        if (fs.exists(cfDir)) {
                            for (FileStatus file : fs.listStatus(cfDir)) {
                                if (refFound |= StoreFile.isReference(file.getPath())) break;
                            }
                        }
                        if (!refFound) continue;
                        break;
                    }
                    if (refFound) continue;
                    check.remove((Object)hri);
                }
                if (check.isEmpty()) {
                    finished.add(pair);
                    continue;
                }
                physicalSplitting.add(pair);
            }
            catch (NoServerForRegionException nsfre) {
                LOG.debug((Object)("No Server Exception thrown for: " + splitAlgo.rowToStr(start)));
                physicalSplitting.add(pair);
                table.clearRegionCache();
            }
        }
        LOG.debug((Object)("Split Scan: " + finished.size() + " finished / " + logicalSplitting.size() + " split wait / " + physicalSplitting.size() + " reference wait"));
        return finished;
    }

    static LinkedList<Pair<byte[], byte[]>> getSplits(HTable table, SplitAlgorithm splitAlgo) throws IOException {
        Path hbDir = new Path(table.getConfiguration().get("hbase.rootdir"));
        Path tableDir = HTableDescriptor.getTableDir(hbDir, table.getTableName());
        Path splitFile = new Path(tableDir, "_balancedSplit");
        FileSystem fs = FileSystem.get((Configuration)table.getConfiguration());
        HashSet daughterRegions = Sets.newHashSet();
        if (!fs.exists(splitFile)) {
            LOG.debug((Object)"No _balancedSplit file.  Calculating splits...");
            HashSet rows = Sets.newHashSet();
            Pair<byte[][], byte[][]> tmp = table.getStartEndKeys();
            Preconditions.checkArgument((tmp.getFirst().length == tmp.getSecond().length ? 1 : 0) != 0, (Object)"Start and End rows should be equivalent");
            for (int i = 0; i < tmp.getFirst().length; ++i) {
                byte[] start = tmp.getFirst()[i];
                byte[] end = tmp.getSecond()[i];
                if (start.length == 0) {
                    start = splitAlgo.firstRow();
                }
                if (end.length == 0) {
                    end = splitAlgo.lastRow();
                }
                rows.add(Pair.newPair(start, end));
            }
            LOG.debug((Object)("Table " + Bytes.toString(table.getTableName()) + " has " + rows.size() + " regions that will be split."));
            Path tmpFile = new Path(tableDir, "_balancedSplit_prepare");
            FSDataOutputStream tmpOut = fs.create(tmpFile);
            for (Pair r : rows) {
                byte[] splitPoint = splitAlgo.split((byte[])r.getFirst(), (byte[])r.getSecond());
                String startStr = splitAlgo.rowToStr((byte[])r.getFirst());
                String splitStr = splitAlgo.rowToStr(splitPoint);
                daughterRegions.add(Pair.newPair(startStr, splitStr));
                LOG.debug((Object)("Will Split [" + startStr + " , " + splitAlgo.rowToStr((byte[])r.getSecond()) + ") at " + splitStr));
                tmpOut.writeChars("+ " + startStr + splitAlgo.separator() + splitStr + "\n");
            }
            tmpOut.close();
            fs.rename(tmpFile, splitFile);
        } else {
            LOG.debug((Object)"_balancedSplit file found. Replay log to restore state...");
            FSUtils.recoverFileLease(fs, splitFile, table.getConfiguration());
            FSDataInputStream tmpIn = fs.open(splitFile);
            StringBuilder sb = new StringBuilder(tmpIn.available());
            while (tmpIn.available() > 0) {
                sb.append(tmpIn.readChar());
            }
            tmpIn.close();
            for (String line : sb.toString().split("\n")) {
                String[] cmd = line.split(splitAlgo.separator());
                Preconditions.checkArgument((3 == cmd.length ? 1 : 0) != 0);
                byte[] start = splitAlgo.strToRow(cmd[1]);
                String startStr = splitAlgo.rowToStr(start);
                byte[] splitPoint = splitAlgo.strToRow(cmd[2]);
                String splitStr = splitAlgo.rowToStr(splitPoint);
                Pair<String, String> r = Pair.newPair(startStr, splitStr);
                if (cmd[0].equals("+")) {
                    LOG.debug((Object)("Adding: " + r));
                    daughterRegions.add(r);
                    continue;
                }
                LOG.debug((Object)("Removing: " + r));
                Preconditions.checkArgument((boolean)cmd[0].equals("-"), (Object)("Unknown option: " + cmd[0]));
                Preconditions.checkState((boolean)daughterRegions.contains(r), (Object)("Missing row: " + r));
                daughterRegions.remove(r);
            }
            LOG.debug((Object)("Done reading. " + daughterRegions.size() + " regions left."));
        }
        LinkedList ret = Lists.newLinkedList();
        for (Pair r : daughterRegions) {
            ret.add(Pair.newPair(splitAlgo.strToRow((String)r.getFirst()), splitAlgo.strToRow((String)r.getSecond())));
        }
        return ret;
    }

    public static class MD5StringSplit
    implements SplitAlgorithm {
        static final String MAXMD5 = "7FFFFFFF";
        static final BigInteger MAXMD5_INT = new BigInteger("7FFFFFFF", 16);
        static final int rowComparisonLength = "7FFFFFFF".length();

        @Override
        public byte[] split(byte[] start, byte[] end) {
            BigInteger s = MD5StringSplit.convertToBigInteger(start);
            BigInteger e = MD5StringSplit.convertToBigInteger(end);
            Preconditions.checkArgument((!e.equals(BigInteger.ZERO) ? 1 : 0) != 0);
            return MD5StringSplit.convertToByte(MD5StringSplit.split2(s, e));
        }

        @Override
        public byte[][] split(int n) {
            BigInteger[] splits = new BigInteger[n - 1];
            BigInteger sizeOfEachSplit = MAXMD5_INT.divide(BigInteger.valueOf(n));
            for (int i = 1; i < n; ++i) {
                splits[i - 1] = sizeOfEachSplit.multiply(BigInteger.valueOf(i));
            }
            return MD5StringSplit.convertToBytes(splits);
        }

        @Override
        public byte[] firstRow() {
            return MD5StringSplit.convertToByte(BigInteger.ZERO);
        }

        @Override
        public byte[] lastRow() {
            return MD5StringSplit.convertToByte(MAXMD5_INT);
        }

        @Override
        public byte[] strToRow(String in) {
            return MD5StringSplit.convertToByte(new BigInteger(in, 16));
        }

        @Override
        public String rowToStr(byte[] row) {
            return Bytes.toStringBinary(row);
        }

        @Override
        public String separator() {
            return " ";
        }

        static BigInteger split2(BigInteger minValue, BigInteger maxValue) {
            return maxValue.add(minValue).divide(BigInteger.valueOf(2L));
        }

        static byte[][] convertToBytes(BigInteger[] bigIntegers) {
            byte[][] returnBytes = new byte[bigIntegers.length][];
            for (int i = 0; i < bigIntegers.length; ++i) {
                returnBytes[i] = MD5StringSplit.convertToByte(bigIntegers[i]);
            }
            return returnBytes;
        }

        static byte[] convertToByte(BigInteger bigInteger) {
            String bigIntegerString = bigInteger.toString(16);
            bigIntegerString = StringUtils.leftPad((String)bigIntegerString, (int)rowComparisonLength, (char)'0');
            return Bytes.toBytes(bigIntegerString);
        }

        static BigInteger convertToBigInteger(byte[] row) {
            return row.length > 0 ? new BigInteger(Bytes.toString(row), 16) : BigInteger.ZERO;
        }
    }

    public static interface SplitAlgorithm {
        public byte[] split(byte[] var1, byte[] var2);

        public byte[][] split(int var1);

        public byte[] firstRow();

        public byte[] lastRow();

        public byte[] strToRow(String var1);

        public String rowToStr(byte[] var1);

        public String separator();
    }
}

