/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import java.io.DataInput;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.lang.reflect.InvocationTargetException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.RemoteExceptionHandler;
import org.apache.hadoop.hbase.master.HMaster;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.util.FileSystemVersionException;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.AlreadyBeingCreatedException;
import org.apache.hadoop.hdfs.protocol.FSConstants;
import org.apache.hadoop.hdfs.server.namenode.LeaseExpiredException;
import org.apache.hadoop.io.SequenceFile;

public class FSUtils {
    private static final Log LOG = LogFactory.getLog(FSUtils.class);

    private FSUtils() {
    }

    public static boolean deleteDirectory(FileSystem fs, Path dir) throws IOException {
        return fs.exists(dir) && fs.delete(dir, true);
    }

    public Path checkdir(FileSystem fs, Path dir) throws IOException {
        if (!fs.exists(dir)) {
            fs.mkdirs(dir);
        }
        return dir;
    }

    public static Path create(FileSystem fs, Path p) throws IOException {
        if (fs.exists(p)) {
            throw new IOException("File already exists " + p.toString());
        }
        if (!fs.createNewFile(p)) {
            throw new IOException("Failed create of " + p);
        }
        return p;
    }

    public static void checkFileSystemAvailable(FileSystem fs) throws IOException {
        if (!(fs instanceof DistributedFileSystem)) {
            return;
        }
        IOException exception = null;
        DistributedFileSystem dfs = (DistributedFileSystem)fs;
        try {
            if (dfs.exists(new Path("/"))) {
                return;
            }
        }
        catch (IOException e) {
            exception = RemoteExceptionHandler.checkIOException(e);
        }
        try {
            fs.close();
        }
        catch (Exception e) {
            LOG.error((Object)"file system close failed: ", (Throwable)e);
        }
        IOException io = new IOException("File system is not available");
        io.initCause(exception);
        throw io;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String getVersion(FileSystem fs, Path rootdir) throws IOException {
        Path versionFile = new Path(rootdir, "hbase.version");
        String version = null;
        if (fs.exists(versionFile)) {
            FSDataInputStream s = fs.open(versionFile);
            try {
                version = DataInputStream.readUTF((DataInput)s);
            }
            catch (EOFException eof) {
                LOG.warn((Object)"Version file was empty, odd, will try to set it.");
            }
            finally {
                s.close();
            }
        }
        return version;
    }

    public static void checkVersion(FileSystem fs, Path rootdir, boolean message) throws IOException {
        FSUtils.checkVersion(fs, rootdir, message, 0);
    }

    public static void checkVersion(FileSystem fs, Path rootdir, boolean message, int wait) throws IOException {
        String version = FSUtils.getVersion(fs, rootdir);
        if (version == null) {
            if (!FSUtils.rootRegionExists(fs, rootdir)) {
                FSUtils.setVersion(fs, rootdir, wait);
                return;
            }
        } else if (version.compareTo("7") == 0) {
            return;
        }
        String msg = "File system needs to be upgraded.  You have version " + version + " and I want version " + "7" + ".  Run the '${HBASE_HOME}/bin/hbase migrate' script.";
        if (message) {
            System.out.println("WARNING! " + msg);
        }
        throw new FileSystemVersionException(msg);
    }

    public static void setVersion(FileSystem fs, Path rootdir) throws IOException {
        FSUtils.setVersion(fs, rootdir, "7", 0);
    }

    public static void setVersion(FileSystem fs, Path rootdir, int wait) throws IOException {
        FSUtils.setVersion(fs, rootdir, "7", wait);
    }

    public static void setVersion(FileSystem fs, Path rootdir, String version, int wait) throws IOException {
        Path versionFile = new Path(rootdir, "hbase.version");
        while (true) {
            try {
                FSDataOutputStream s = fs.create(versionFile);
                s.writeUTF(version);
                LOG.debug((Object)("Created version file at " + rootdir.toString() + " set its version at:" + version));
                s.close();
                return;
            }
            catch (IOException e) {
                if (wait <= 0) continue;
                LOG.warn((Object)("Unable to create version file at " + rootdir.toString() + ", retrying: " + e.getMessage()));
                fs.delete(versionFile, false);
                try {
                    Thread.sleep(wait);
                }
                catch (InterruptedException ex) {
                }
                continue;
            }
            break;
        }
    }

    public static Path validateRootPath(Path root) throws IOException {
        try {
            URI rootURI = new URI(root.toString());
            String scheme = rootURI.getScheme();
            if (scheme == null) {
                throw new IOException("Root directory does not have a scheme");
            }
            return root;
        }
        catch (URISyntaxException e) {
            IOException io = new IOException("Root directory path is not a valid URI -- check your hbase.rootdir configuration");
            io.initCause(e);
            throw io;
        }
    }

    public static void waitOnSafeMode(Configuration conf, long wait) throws IOException {
        FileSystem fs = FileSystem.get((Configuration)conf);
        if (!(fs instanceof DistributedFileSystem)) {
            return;
        }
        DistributedFileSystem dfs = (DistributedFileSystem)fs;
        while (dfs.setSafeMode(FSConstants.SafeModeAction.SAFEMODE_GET)) {
            LOG.info((Object)"Waiting for dfs to exit safe mode...");
            try {
                Thread.sleep(wait);
            }
            catch (InterruptedException e) {}
        }
    }

    public static String getPath(Path p) {
        return p.toUri().getPath();
    }

    public static Path getRootDir(Configuration c) throws IOException {
        Path p = new Path(c.get("hbase.rootdir"));
        FileSystem fs = p.getFileSystem(c);
        return p.makeQualified(fs);
    }

    public static boolean rootRegionExists(FileSystem fs, Path rootdir) throws IOException {
        Path rootRegionDir = HRegion.getRegionDir(rootdir, HRegionInfo.ROOT_REGIONINFO);
        return fs.exists(rootRegionDir);
    }

    public static boolean isMajorCompacted(FileSystem fs, Path hbaseRootDir) throws IOException {
        FileStatus[] tableDirs;
        for (FileStatus tableDir : tableDirs = fs.listStatus(hbaseRootDir, (PathFilter)new DirFilter(fs))) {
            FileStatus[] regionDirs;
            Path d = tableDir.getPath();
            if (d.getName().equals(".logs")) continue;
            for (FileStatus regionDir : regionDirs = fs.listStatus(d, (PathFilter)new DirFilter(fs))) {
                FileStatus[] familyDirs;
                Path dd = regionDir.getPath();
                if (dd.getName().equals("compaction.dir")) continue;
                for (FileStatus familyDir : familyDirs = fs.listStatus(dd, (PathFilter)new DirFilter(fs))) {
                    Path family = familyDir.getPath();
                    FileStatus[] familyStatus = fs.listStatus(family);
                    if (familyStatus.length <= 1) continue;
                    LOG.debug((Object)(family.toString() + " has " + familyStatus.length + " files."));
                    return false;
                }
            }
        }
        return true;
    }

    public static int getTotalTableFragmentation(HMaster master) throws IOException {
        Map<String, Integer> map = FSUtils.getTableFragmentation(master);
        return map != null && map.size() > 0 ? map.get("-TOTAL-") : -1;
    }

    public static Map<String, Integer> getTableFragmentation(HMaster master) throws IOException {
        Path path = FSUtils.getRootDir(master.getConfiguration());
        FileSystem fs = path.getFileSystem(master.getConfiguration());
        return FSUtils.getTableFragmentation(fs, path);
    }

    public static Map<String, Integer> getTableFragmentation(FileSystem fs, Path hbaseRootDir) throws IOException {
        FileStatus[] tableDirs;
        HashMap<String, Integer> frags = new HashMap<String, Integer>();
        int cfCountTotal = 0;
        int cfFragTotal = 0;
        DirFilter df = new DirFilter(fs);
        for (FileStatus tableDir : tableDirs = fs.listStatus(hbaseRootDir, (PathFilter)df)) {
            FileStatus[] regionDirs;
            Path d = tableDir.getPath();
            if (d.getName().equals(".logs")) continue;
            int cfCount = 0;
            int cfFrag = 0;
            for (FileStatus regionDir : regionDirs = fs.listStatus(d, (PathFilter)df)) {
                FileStatus[] familyDirs;
                Path dd = regionDir.getPath();
                if (dd.getName().equals("compaction.dir")) continue;
                for (FileStatus familyDir : familyDirs = fs.listStatus(dd, (PathFilter)df)) {
                    ++cfCount;
                    ++cfCountTotal;
                    Path family = familyDir.getPath();
                    FileStatus[] familyStatus = fs.listStatus(family);
                    if (familyStatus.length <= 1) continue;
                    ++cfFrag;
                    ++cfFragTotal;
                }
            }
            frags.put(d.getName(), Math.round((float)cfFrag / (float)cfCount * 100.0f));
        }
        frags.put("-TOTAL-", Math.round((float)cfFragTotal / (float)cfCountTotal * 100.0f));
        return frags;
    }

    public static boolean isPre020FileLayout(FileSystem fs, Path hbaseRootDir) throws IOException {
        Path mapfiles = new Path(new Path(new Path(new Path(hbaseRootDir, "-ROOT-"), "70236052"), "info"), "mapfiles");
        return fs.exists(mapfiles);
    }

    public static boolean isMajorCompactedPre020(FileSystem fs, Path hbaseRootDir) throws IOException {
        FileStatus[] tableDirs;
        for (FileStatus tableDir : tableDirs = fs.listStatus(hbaseRootDir, (PathFilter)new DirFilter(fs))) {
            FileStatus[] regionDirs;
            Path d = tableDir.getPath();
            if (d.getName().equals(".logs")) continue;
            for (FileStatus regionDir : regionDirs = fs.listStatus(d, (PathFilter)new DirFilter(fs))) {
                FileStatus[] familyDirs;
                Path dd = regionDir.getPath();
                if (dd.getName().equals("compaction.dir")) continue;
                for (FileStatus familyDir : familyDirs = fs.listStatus(dd, (PathFilter)new DirFilter(fs))) {
                    Path family = familyDir.getPath();
                    FileStatus[] infoAndMapfile = fs.listStatus(family);
                    if (infoAndMapfile.length != 0 && infoAndMapfile.length != 2) {
                        LOG.debug((Object)(family.toString() + " has more than just info and mapfile: " + infoAndMapfile.length));
                        return false;
                    }
                    for (int ll = 0; ll < 2; ++ll) {
                        if (infoAndMapfile[ll].getPath().getName().equals("info") || infoAndMapfile[ll].getPath().getName().equals("mapfiles")) continue;
                        LOG.debug((Object)("Unexpected directory name: " + infoAndMapfile[ll].getPath()));
                        return false;
                    }
                    FileStatus[] familyStatus = fs.listStatus(new Path(family, "mapfiles"));
                    if (familyStatus.length <= 1) continue;
                    LOG.debug((Object)(family.toString() + " has " + familyStatus.length + " files."));
                    return false;
                }
            }
        }
        return true;
    }

    public static boolean isAppendSupported(Configuration conf) {
        boolean append = conf.getBoolean("dfs.support.append", false);
        if (append) {
            try {
                SequenceFile.Writer.class.getMethod("syncFs", new Class[0]);
                append = true;
            }
            catch (SecurityException e) {
            }
            catch (NoSuchMethodException e) {
                append = false;
            }
        } else {
            try {
                FSDataOutputStream.class.getMethod("hflush", new Class[0]);
            }
            catch (NoSuchMethodException e) {
                append = false;
            }
        }
        return append;
    }

    public static boolean isHDFS(Configuration conf) throws IOException {
        FileSystem fs = FileSystem.get((Configuration)conf);
        String scheme = fs.getUri().getScheme();
        return scheme.equalsIgnoreCase("hdfs");
    }

    public static void recoverFileLease(FileSystem fs, Path p, Configuration conf) throws IOException {
        if (!FSUtils.isAppendSupported(conf)) {
            LOG.warn((Object)"Running on HDFS without append enabled may result in data loss");
            return;
        }
        if (!(fs instanceof DistributedFileSystem)) {
            return;
        }
        LOG.info((Object)("Recovering file " + p));
        long startWaiting = System.currentTimeMillis();
        boolean recovered = false;
        while (!recovered) {
            try {
                try {
                    if (!(fs instanceof DistributedFileSystem)) {
                        throw new Exception("Not a DistributedFileSystem");
                    }
                    DistributedFileSystem dfs = (DistributedFileSystem)fs;
                    DistributedFileSystem.class.getMethod("recoverLease", Path.class).invoke((Object)dfs, p);
                }
                catch (InvocationTargetException ite) {
                    throw (IOException)ite.getCause();
                }
                catch (Exception e) {
                    LOG.debug((Object)("Failed fs.recoverLease invocation, " + e.toString() + ", trying fs.append instead"));
                    FSDataOutputStream out = fs.append(p);
                    out.close();
                }
                recovered = true;
            }
            catch (IOException e) {
                e = RemoteExceptionHandler.checkIOException(e);
                if (e instanceof AlreadyBeingCreatedException) {
                    long waitedFor = System.currentTimeMillis() - startWaiting;
                    if (waitedFor > 60000L) {
                        LOG.warn((Object)("Waited " + waitedFor + "ms for lease recovery on " + p + ":" + e.getMessage()));
                    }
                }
                if (e instanceof LeaseExpiredException && e.getMessage().contains("File does not exist")) {
                    throw new FileNotFoundException("The given HLog wasn't found at " + p.toString());
                }
                throw new IOException("Failed to open " + p + " for append", e);
            }
            try {
                Thread.sleep(1000L);
            }
            catch (InterruptedException ex) {
                new InterruptedIOException().initCause(ex);
            }
        }
        LOG.info((Object)("Finished lease recover attempt for " + p));
    }

    public static class DirFilter
    implements PathFilter {
        private final FileSystem fs;

        public DirFilter(FileSystem fs) {
            this.fs = fs;
        }

        public boolean accept(Path p) {
            boolean isdir = false;
            try {
                isdir = this.fs.getFileStatus(p).isDir();
            }
            catch (IOException e) {
                e.printStackTrace();
            }
            return isdir;
        }
    }
}

