/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.Server;
import org.apache.hadoop.hbase.catalog.MetaEditor;
import org.apache.hadoop.hbase.io.Reference;
import org.apache.hadoop.hbase.regionserver.FlushRequester;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.regionserver.OnlineRegions;
import org.apache.hadoop.hbase.regionserver.RegionServerServices;
import org.apache.hadoop.hbase.regionserver.Store;
import org.apache.hadoop.hbase.regionserver.StoreFile;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.CancelableProgressable;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.PairOfSameType;
import org.apache.zookeeper.KeeperException;

class SplitTransaction {
    private static final Log LOG = LogFactory.getLog(SplitTransaction.class);
    private static final String SPLITDIR = "splits";
    private final HRegion parent;
    private HRegionInfo hri_a;
    private HRegionInfo hri_b;
    private Path splitdir;
    private long fileSplitTimeout = 30000L;
    private final byte[] splitrow;
    private final List<JournalEntry> journal = new ArrayList<JournalEntry>();

    SplitTransaction(HRegion r, byte[] splitrow) {
        this.parent = r;
        this.splitrow = splitrow;
        this.splitdir = SplitTransaction.getSplitDir(this.parent);
    }

    public boolean prepare() {
        if (this.parent.isClosed() || this.parent.isClosing()) {
            return false;
        }
        HRegionInfo hri = this.parent.getRegionInfo();
        this.parent.prepareToSplit();
        byte[] startKey = hri.getStartKey();
        byte[] endKey = hri.getEndKey();
        if (Bytes.equals(startKey, this.splitrow) || !this.parent.getRegionInfo().containsRow(this.splitrow)) {
            LOG.info((Object)("Split row is not inside region key range or is equal to startkey: " + Bytes.toString(this.splitrow)));
            return false;
        }
        long rid = SplitTransaction.getDaughterRegionIdTimestamp(hri);
        this.hri_a = new HRegionInfo(hri.getTableDesc(), startKey, this.splitrow, false, rid);
        this.hri_b = new HRegionInfo(hri.getTableDesc(), this.splitrow, endKey, false, rid);
        return true;
    }

    private static long getDaughterRegionIdTimestamp(HRegionInfo hri) {
        long rid = EnvironmentEdgeManager.currentTimeMillis();
        if (rid < hri.getRegionId()) {
            LOG.warn((Object)("Clock skew; parent regions id is " + hri.getRegionId() + " but current time here is " + rid));
            rid = hri.getRegionId() + 1L;
        }
        return rid;
    }

    PairOfSameType<HRegion> execute(Server server, RegionServerServices services) throws IOException {
        LOG.info((Object)("Starting split of region " + this.parent));
        if (server != null && server.isStopped() || services != null && services.isStopping()) {
            throw new IOException("Server is stopped or stopping");
        }
        assert (!this.parent.lock.writeLock().isHeldByCurrentThread()) : "Unsafe to hold write lock while performing RPCs";
        boolean testing = server == null ? true : server.getConfiguration().getBoolean("hbase.testing.nocluster", false);
        this.fileSplitTimeout = testing ? this.fileSplitTimeout : server.getConfiguration().getLong("hbase.regionserver.fileSplitTimeout", this.fileSplitTimeout);
        SplitTransaction.createSplitDir(this.parent.getFilesystem(), this.splitdir);
        this.journal.add(JournalEntry.CREATE_SPLIT_DIR);
        List<StoreFile> hstoreFilesToSplit = this.parent.close(false);
        if (hstoreFilesToSplit == null) {
            throw new IOException("Failed to close region: already closed by another thread");
        }
        this.journal.add(JournalEntry.CLOSED_PARENT_REGION);
        if (!testing) {
            services.removeFromOnlineRegions(this.parent.getRegionInfo().getEncodedName());
        }
        this.journal.add(JournalEntry.OFFLINED_PARENT);
        this.splitStoreFiles(this.splitdir, hstoreFilesToSplit);
        this.journal.add(JournalEntry.STARTED_REGION_A_CREATION);
        HRegion a = this.createDaughterRegion(this.hri_a, this.parent.flushRequester);
        this.journal.add(JournalEntry.STARTED_REGION_B_CREATION);
        HRegion b = this.createDaughterRegion(this.hri_b, this.parent.flushRequester);
        if (!testing) {
            MetaEditor.offlineParentInMeta(server.getCatalogTracker(), this.parent.getRegionInfo(), a.getRegionInfo(), b.getRegionInfo());
        }
        if (!testing) {
            DaughterOpener aOpener = new DaughterOpener(server, services, a);
            DaughterOpener bOpener = new DaughterOpener(server, services, b);
            aOpener.start();
            bOpener.start();
            try {
                aOpener.join();
                bOpener.join();
            }
            catch (InterruptedException e) {
                server.abort("Exception running daughter opens", e);
            }
        }
        return new PairOfSameType<HRegion>(a, b);
    }

    void openDaughterRegion(Server server, RegionServerServices services, HRegion daughter) throws IOException, KeeperException {
        if (server.isStopped() || services.isStopping()) {
            MetaEditor.addDaughter(server.getCatalogTracker(), daughter.getRegionInfo(), null);
            LOG.info((Object)("Not opening daughter " + daughter.getRegionInfo().getRegionNameAsString() + " because stopping=" + services.isStopping() + ", stopped=" + server.isStopped()));
            return;
        }
        HRegionInfo hri = daughter.getRegionInfo();
        LoggingProgressable reporter = new LoggingProgressable(hri, server.getConfiguration());
        HRegion r = daughter.openHRegion(reporter);
        services.postOpenDeployTasks(r, server.getCatalogTracker(), true);
    }

    private static Path getSplitDir(HRegion r) {
        return new Path(r.getRegionDir(), SPLITDIR);
    }

    private static void createSplitDir(FileSystem fs, Path splitdir) throws IOException {
        if (fs.exists(splitdir)) {
            throw new IOException("Splitdir already exits? " + splitdir);
        }
        if (!fs.mkdirs(splitdir)) {
            throw new IOException("Failed create of " + splitdir);
        }
    }

    private static void cleanupSplitDir(FileSystem fs, Path splitdir) throws IOException {
        SplitTransaction.deleteDir(fs, splitdir, false);
    }

    private static void deleteDir(FileSystem fs, Path dir, boolean mustPreExist) throws IOException {
        if (!fs.exists(dir)) {
            if (mustPreExist) {
                throw new IOException(dir.toString() + " does not exist!");
            }
        } else if (!fs.delete(dir, true)) {
            throw new IOException("Failed delete of " + dir);
        }
    }

    private void splitStoreFiles(Path splitdir, List<StoreFile> hstoreFilesToSplit) throws IOException {
        if (hstoreFilesToSplit == null) {
            throw new IOException("Close returned empty list of StoreFiles");
        }
        int nbFiles = hstoreFilesToSplit.size();
        ThreadFactoryBuilder builder = new ThreadFactoryBuilder();
        builder.setNameFormat("StoreFileSplitter-%1$d");
        ThreadFactory factory = builder.build();
        ThreadPoolExecutor threadPool = (ThreadPoolExecutor)Executors.newFixedThreadPool(nbFiles, factory);
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>(nbFiles);
        for (StoreFile storeFile : hstoreFilesToSplit) {
            StoreFileSplitter sfs = new StoreFileSplitter(storeFile, splitdir);
            futures.add(threadPool.submit(sfs));
        }
        threadPool.shutdown();
        try {
            boolean stillRunning;
            boolean bl = stillRunning = !threadPool.awaitTermination(this.fileSplitTimeout, TimeUnit.MILLISECONDS);
            if (stillRunning) {
                threadPool.shutdownNow();
                throw new IOException("Took too long to split the files and create the references, aborting split");
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Interrupted while waiting for file splitters", e);
        }
        for (Future future : futures) {
            try {
                future.get();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new IOException("Interrupted while trying to get the results of file splitters", e);
            }
            catch (ExecutionException e) {
                throw new IOException(e);
            }
        }
    }

    private void splitStoreFile(StoreFile sf, Path splitdir) throws IOException {
        FileSystem fs = this.parent.getFilesystem();
        byte[] family = sf.getFamily();
        String encoded = this.hri_a.getEncodedName();
        Path storedir = Store.getStoreHomedir(splitdir, encoded, family);
        StoreFile.split(fs, storedir, sf, this.splitrow, Reference.Range.bottom);
        encoded = this.hri_b.getEncodedName();
        storedir = Store.getStoreHomedir(splitdir, encoded, family);
        StoreFile.split(fs, storedir, sf, this.splitrow, Reference.Range.top);
    }

    HRegion createDaughterRegion(HRegionInfo hri, FlushRequester flusher) throws IOException {
        FileSystem fs = this.parent.getFilesystem();
        Path regionDir = SplitTransaction.getSplitDirForDaughter(this.parent.getFilesystem(), this.splitdir, hri);
        HRegion r = HRegion.newHRegion(this.parent.getTableDir(), this.parent.getLog(), fs, this.parent.getConf(), hri, flusher);
        HRegion.moveInitialFilesIntoPlace(fs, regionDir, r.getRegionDir());
        return r;
    }

    private static void cleanupDaughterRegion(FileSystem fs, Path tabledir, String encodedName) throws IOException {
        Path regiondir = HRegion.getRegionDir(tabledir, encodedName);
        SplitTransaction.deleteDir(fs, regiondir, false);
    }

    private static Path getSplitDirForDaughter(FileSystem fs, Path splitdir, HRegionInfo hri) throws IOException {
        return new Path(splitdir, hri.getEncodedName());
    }

    public void rollback(OnlineRegions or) throws IOException {
        FileSystem fs = this.parent.getFilesystem();
        ListIterator<JournalEntry> iterator = this.journal.listIterator(this.journal.size());
        block7: while (iterator.hasPrevious()) {
            JournalEntry je = iterator.previous();
            switch (je) {
                case CREATE_SPLIT_DIR: {
                    this.parent.writestate.writesEnabled = true;
                    SplitTransaction.cleanupSplitDir(fs, this.splitdir);
                    continue block7;
                }
                case CLOSED_PARENT_REGION: {
                    this.parent.initialize();
                    continue block7;
                }
                case STARTED_REGION_A_CREATION: {
                    SplitTransaction.cleanupDaughterRegion(fs, this.parent.getTableDir(), this.hri_a.getEncodedName());
                    continue block7;
                }
                case STARTED_REGION_B_CREATION: {
                    SplitTransaction.cleanupDaughterRegion(fs, this.parent.getTableDir(), this.hri_b.getEncodedName());
                    continue block7;
                }
                case OFFLINED_PARENT: {
                    if (or == null) continue block7;
                    or.addToOnlineRegions(this.parent);
                    continue block7;
                }
            }
            throw new RuntimeException("Unhandled journal entry: " + (Object)((Object)je));
        }
    }

    HRegionInfo getFirstDaughter() {
        return this.hri_a;
    }

    HRegionInfo getSecondDaughter() {
        return this.hri_b;
    }

    Path getSplitDir() {
        return this.splitdir;
    }

    static void cleanupAnySplitDetritus(HRegion r) throws IOException {
        Path splitdir = SplitTransaction.getSplitDir(r);
        FileSystem fs = r.getFilesystem();
        if (!fs.exists(splitdir)) {
            return;
        }
        FileStatus[] daughters = fs.listStatus(splitdir, (PathFilter)new FSUtils.DirFilter(fs));
        for (int i = 0; i < daughters.length; ++i) {
            SplitTransaction.cleanupDaughterRegion(fs, r.getTableDir(), daughters[i].getPath().getName());
        }
        SplitTransaction.cleanupSplitDir(r.getFilesystem(), splitdir);
        LOG.info((Object)("Cleaned up old failed split transaction detritus: " + splitdir));
    }

    class StoreFileSplitter
    implements Callable<Void> {
        private final StoreFile sf;
        private final Path splitdir;

        public StoreFileSplitter(StoreFile sf, Path splitdir) {
            this.sf = sf;
            this.splitdir = splitdir;
        }

        @Override
        public Void call() throws IOException {
            SplitTransaction.this.splitStoreFile(this.sf, this.splitdir);
            return null;
        }
    }

    static class LoggingProgressable
    implements CancelableProgressable {
        private final HRegionInfo hri;
        private long lastLog = -1L;
        private final long interval;

        LoggingProgressable(HRegionInfo hri, Configuration c) {
            this.hri = hri;
            this.interval = c.getLong("hbase.regionserver.split.daughter.open.log.interval", 10000L);
        }

        @Override
        public boolean progress() {
            long now = System.currentTimeMillis();
            if (now - this.lastLog > this.interval) {
                LOG.info((Object)("Opening " + this.hri.getRegionNameAsString()));
                this.lastLog = now;
            }
            return true;
        }
    }

    class DaughterOpener
    extends Thread {
        private final RegionServerServices services;
        private final Server server;
        private final HRegion r;

        DaughterOpener(Server s, RegionServerServices services, HRegion r) {
            super(s.getServerName() + "-daughterOpener=" + r.getRegionInfo().getEncodedName());
            this.setDaemon(true);
            this.services = services;
            this.server = s;
            this.r = r;
        }

        @Override
        public void run() {
            try {
                SplitTransaction.this.openDaughterRegion(this.server, this.services, this.r);
            }
            catch (Throwable t) {
                this.server.abort("Failed open of daughter " + this.r.getRegionInfo().getRegionNameAsString(), t);
            }
        }
    }

    static enum JournalEntry {
        CREATE_SPLIT_DIR,
        CLOSED_PARENT_REGION,
        OFFLINED_PARENT,
        STARTED_REGION_A_CREATION,
        STARTED_REGION_B_CREATION;

    }
}

