package eu.codlab.compose.widgets

import androidx.compose.desktop.ui.tooling.preview.Preview
import androidx.compose.foundation.layout.Column
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.MoreVert
import androidx.compose.material3.Icon
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.vector.ImageVector
import androidx.compose.ui.text.font.FontFamily
import androidx.compose.ui.text.font.FontStyle
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.TextUnit
import androidx.compose.ui.unit.sp
import eu.codlab.compose.preview.PreviewAppMaterialTheme
import eu.codlab.compose.theme.LocalDarkTheme

private val title = 32.sp
private val subtitle = 22.sp
private val normal = 14.sp

private data class PairColor(
    val dark: Color,
    val light: Color
)

private val titleColor = PairColor(
    dark = Color.White,
    light = Color.Black
)

private val subtitleColor = PairColor(
    dark = Color(0xff8f959f),
    light = Color(0xff343434)
)

@Composable
fun VectorIcon(
    modifier: Modifier = Modifier,
    imageVector: ImageVector,
    isDark: Boolean = LocalDarkTheme.current,
    color: Color? = null,
    contentDescription: String = "Icon"
) {
    val actual = color ?: if (isDark) titleColor.dark else titleColor.light

    return Icon(
        modifier = modifier,
        imageVector = imageVector,
        contentDescription = contentDescription,
        tint = actual
    )
}

@Composable
fun TextTitle(
    text: String,
    modifier: Modifier = Modifier,
    textAlign: TextAlign = TextAlign.Start,
    fontSize: TextUnit = title,
    color: Color? = null,
    fontFamily: FontFamily? = null,
    fontStyle: FontStyle? = null,
    overflow: TextOverflow = TextOverflow.Clip,
    softWrap: Boolean = true,
    maxLines: Int = Int.MAX_VALUE
) = CreateText(
    modifier,
    text,
    textAlign,
    fontSize,
    FontWeight.Bold,
    LocalDarkTheme.current,
    titleColor,
    color,
    fontFamily,
    fontStyle,
    overflow,
    softWrap,
    maxLines
)

@Composable
fun TextSubtitle(
    text: String,
    modifier: Modifier = Modifier,
    textAlign: TextAlign = TextAlign.Start,
    fontSize: TextUnit = subtitle,
    color: Color? = null,
    fontFamily: FontFamily? = null,
    fontStyle: FontStyle? = null,
    overflow: TextOverflow = TextOverflow.Clip,
    softWrap: Boolean = true,
    maxLines: Int = Int.MAX_VALUE
) = CreateText(
    modifier,
    text,
    textAlign,
    fontSize,
    FontWeight.Bold,
    LocalDarkTheme.current,
    subtitleColor,
    color,
    fontFamily,
    fontStyle,
    overflow,
    softWrap,
    maxLines
)

@Composable
fun TextNormal(
    text: String,
    modifier: Modifier = Modifier,
    textAlign: TextAlign = TextAlign.Start,
    fontSize: TextUnit = normal,
    color: Color? = null,
    fontWeight: FontWeight = FontWeight.Normal,
    fontFamily: FontFamily? = null,
    fontStyle: FontStyle? = null,
    overflow: TextOverflow = TextOverflow.Clip,
    softWrap: Boolean = true,
    maxLines: Int = Int.MAX_VALUE
) = CreateText(
    modifier = modifier,
    text,
    textAlign,
    fontSize,
    fontWeight,
    LocalDarkTheme.current,
    titleColor,
    color,
    fontFamily,
    fontStyle,
    overflow,
    softWrap,
    maxLines
)

@Composable
private fun CreateText(
    modifier: Modifier,
    text: String,
    textAlign: TextAlign = TextAlign.Start,
    fontSize: TextUnit,
    fontWeight: FontWeight = FontWeight.Bold,
    isDark: Boolean = LocalDarkTheme.current,
    pair: PairColor,
    color: Color?,
    fontFamily: FontFamily?,
    fontStyle: FontStyle?,
    overflow: TextOverflow,
    softWrap: Boolean,
    maxLines: Int
) {
    val actual = color ?: if (isDark) pair.dark else pair.light

    return Text(
        modifier = modifier,
        text = text,
        textAlign = textAlign,
        fontSize = fontSize,
        fontWeight = fontWeight,
        color = actual,
        fontFamily = fontFamily,
        fontStyle = fontStyle,
        overflow = overflow,
        softWrap = softWrap,
        maxLines = maxLines
    )
}

@Preview
@Composable
private fun RenderTextDark() {
    PreviewAppMaterialTheme(darkTheme = true) {
        Column(modifier = Modifier.systemBackground()) {
            TextTitle(text = "title")
            TextSubtitle(text = "subtitle")
            TextNormal(text = "text")
            VectorIcon(imageVector = Icons.Default.MoreVert)
        }
    }
}

@Preview
@Composable
private fun RenderTextLight() {
    PreviewAppMaterialTheme(darkTheme = false) {
        Column(modifier = Modifier.systemBackground()) {
            TextTitle(text = "title")
            TextSubtitle(text = "subtitle")
            TextNormal(text = "text")

            VectorIcon(imageVector = Icons.Default.MoreVert)
        }
    }
}
