package eu.codlab.viewpager.pagination

import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.width
import androidx.compose.foundation.layout.wrapContentSize
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.unit.Dp
import androidx.compose.ui.unit.dp
import kotlin.math.absoluteValue

/**
 * Simple pager indicator
 *
 * Source : https://medium.com/@domen.lanisnik/exploring-the-official-pager-in-compose-8c2698c49a98
 */
@Composable
internal fun HorizontalDotPagerIndicatorWrapper(
    pageCount: Int,
    currentPage: Int,
    targetPage: Int,
    currentPageOffsetFraction: Float,
    modifier: Modifier = Modifier,
    indicatorColor: Color = Color.DarkGray,
    unselectedIndicatorSize: Dp = 8.dp,
    selectedIndicatorSize: Dp = 10.dp,
    indicatorPadding: Dp = 2.dp,
    sizeMorphing: (Dp) -> MorphedSize = { size ->
        MorphedSize(
            padding = DpRectangle(
                width = ((selectedIndicatorSize + indicatorPadding * 2) - size) / 2,
                height = size / 4
            ),
            cornerSize = size / 2,
            indicator = DpRectangle(size)
        )
    }
) {
    Row(
        horizontalArrangement = Arrangement.Center,
        verticalAlignment = Alignment.CenterVertically,
        modifier = modifier
            .wrapContentSize()
            .height(selectedIndicatorSize + indicatorPadding * 2)
    ) {

        // draw an indicator for each page
        repeat(pageCount) { page ->
            // calculate color and size of the indicator
            val (color, size) =
                if (currentPage == page || targetPage == page) {
                    // calculate page offset
                    val pageOffset =
                        ((currentPage - page) + currentPageOffsetFraction).absoluteValue
                    // calculate offset percentage between 0.0 and 1.0
                    val offsetPercentage = 1f - pageOffset.coerceIn(0f, 1f)

                    val size =
                        unselectedIndicatorSize + ((selectedIndicatorSize - unselectedIndicatorSize) * offsetPercentage)

                    indicatorColor.copy(
                        alpha = offsetPercentage
                    ) to size
                } else {
                    indicatorColor.copy(alpha = 0.1f) to unselectedIndicatorSize
                }

            val morphed = sizeMorphing(size)

            // draw indicator
            Box(
                modifier = Modifier
                    .padding(
                        // apply horizontal padding, so that each indicator is same width
                        horizontal = morphed.padding.width,
                        vertical = morphed.padding.height
                    )
                    .clip(RoundedCornerShape(morphed.cornerSize))
                    .background(color)
                    .width(morphed.indicator.width)
                    .height(morphed.indicator.height)
            )
        }
    }
}

data class MorphedSize(
    val padding: DpRectangle,
    val cornerSize: Dp,
    val indicator: DpRectangle
)

data class DpRectangle(
    val width: Dp,
    val height: Dp = width
)