package eu.codlab.openweathermap.calls

import eu.codlab.openweathermap.rest.RestClientNetworkException
import eu.codlab.openweathermap.rest.RestOptions
import eu.codlab.openweathermap.utils.ClientProvider
import io.ktor.client.statement.HttpResponse
import io.ktor.client.statement.bodyAsText
import io.ktor.http.HttpStatusCode
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.KSerializer
import kotlinx.serialization.json.Json

abstract class InternalRestClient(options: RestOptions) {
    @OptIn(ExperimentalSerializationApi::class)
    protected val json = Json {
        encodeDefaults = true
        ignoreUnknownKeys = true
        isLenient = true
        explicitNulls = false
    }

    protected val options: RestOptions = options.copy()
    protected val client = ClientProvider.createClient(options.timeout)

    protected suspend fun <O> map(
        response: HttpResponse,
        deserializer: KSerializer<O>
    ): O {
        if (response.status != HttpStatusCode.OK) {
            val text = response.bodyAsText()
            throw RestClientNetworkException(response.status, text)
        }

        // currently not using an inline type inference for now
        val result = json.parseToJsonElement(response.bodyAsText())
        return json.decodeFromJsonElement(deserializer, result)
    }
}
