package eu.codlab.moko.maps

import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import com.google.android.gms.maps.CameraUpdateFactory
import com.google.android.gms.maps.model.LatLng
import com.google.android.gms.maps.model.LatLngBounds
import com.google.maps.android.compose.GoogleMap
import com.google.maps.android.compose.MapUiSettings
import com.google.maps.android.compose.Marker
import com.google.maps.android.compose.MarkerComposable
import com.google.maps.android.compose.Polyline
import com.google.maps.android.compose.rememberCameraPositionState
import com.google.maps.android.compose.rememberMarkerState

fun MapType.toMapType() = com.google.maps.android.compose.MapType.valueOf(this.name)

fun MapSettings.toUiSettings() = MapUiSettings(
    compassEnabled = this.compassEnabled,
    indoorLevelPickerEnabled = this.indoorLevelPickerEnabled,
    mapToolbarEnabled = this.mapToolbarEnabled,
    myLocationButtonEnabled = this.myLocationButtonEnabled,
    rotationGesturesEnabled = this.rotationGesturesEnabled,
    scrollGesturesEnabled = this.scrollGesturesEnabled,
    scrollGesturesEnabledDuringRotateOrZoom = this.scrollGesturesEnabledDuringRotateOrZoom,
    tiltGesturesEnabled = this.tiltGesturesEnabled,
    zoomControlsEnabled = this.zoomControlsEnabled,
    zoomGesturesEnabled = this.zoomGesturesEnabled,
)

fun MapProperties.toProperties() = com.google.maps.android.compose.MapProperties(
    isBuildingEnabled = this.isBuildingEnabled,
    isIndoorEnabled = this.isIndoorEnabled,
    isMyLocationEnabled = this.isMyLocationEnabled,
    isTrafficEnabled = this.isTrafficEnabled,
    latLngBoundsForCameraTarget = this.latLngBoundsForCameraTarget?.let {
        LatLngBounds(
            LatLng(
                it.southwest.latitude,
                it.southwest.longitude
            ),
            LatLng(
                it.northeast.latitude,
                it.northeast.longitude
            )
        )
    },
    mapType = mapType.toMapType(),
    maxZoomPreference = maxZoomPreference,
    minZoomPreference = minZoomPreference
)

@Composable
actual fun MapsView(
    modifier: Modifier,
    properties: MapProperties,
    settings: MapSettings,
    markers: List<MapMarker>?,
    markerComposables: List<MapMarkerComposable>?,
    cameraPosition: CameraPosition?,
    cameraPositionLatLongBounds: CameraPositionLatLongBounds?,
    polyLines: List<PolyLine>?
) {
    var onMapLoaded by remember { mutableStateOf(false) }
    val cameraPositionState = rememberCameraPositionState()

    LaunchedEffect(onMapLoaded, cameraPosition) {
        if (!onMapLoaded) {
            // avoid camera crash if the map is not yet loaded
            return@LaunchedEffect
        }

        cameraPosition?.let {
            cameraPositionState.animate(
                CameraUpdateFactory.newLatLngZoom(
                    LatLng(
                        it.target.latitude,
                        it.target.longitude
                    ), it.zoom
                )
            )
        }
    }

    LaunchedEffect(onMapLoaded, cameraPositionLatLongBounds) {
        if (!onMapLoaded) {
            // avoid camera crash if the map is not yet loaded
            return@LaunchedEffect
        }

        cameraPositionLatLongBounds?.let {
            val latLngBounds = LatLngBounds.builder().apply {
                it.coordinates.forEach { latLong ->
                    include(LatLng(latLong.latitude, latLong.longitude))
                }
            }.build()

            cameraPositionState.move(
                CameraUpdateFactory.newLatLngBounds(latLngBounds, it.padding)
            )
        }
    }

    GoogleMap(
        cameraPositionState = cameraPositionState,
        modifier = modifier,
        properties = properties.toProperties(),
        uiSettings = settings.toUiSettings(),
        onMapLoaded = {
            onMapLoaded = true
        }
    ) {
        markers?.forEach { marker ->
            Marker(
                state = rememberMarkerState(
                    key = marker.key,
                    position = LatLng(marker.position.latitude, marker.position.longitude)
                ),
                alpha = marker.alpha,
                title = marker.title,
                snippet = marker.title,
                zIndex = marker.zIndex
            )
        }

        markerComposables?.forEach { marker ->
            MarkerComposable(
                state = rememberMarkerState(
                    key = marker.key,
                    position = LatLng(marker.position.latitude, marker.position.longitude)
                ),
                alpha = marker.alpha,
                title = marker.title,
                content = marker.content,
                zIndex = marker.zIndex
            )
        }

        polyLines?.forEach { polyLine ->
            Polyline(
                points = List(polyLine.points.size) {
                    val latLong = polyLine.points[it]
                    LatLng(latLong.latitude, latLong.longitude)
                },
                color = polyLine.color,
                width = polyLine.width,
                zIndex = polyLine.zIndex
            )
        }

    }
}
