package eu.codlab.http

import io.ktor.client.HttpClient
import io.ktor.client.engine.darwin.Darwin
import io.ktor.client.plugins.HttpTimeout
import io.ktor.client.plugins.auth.Auth
import io.ktor.client.plugins.auth.AuthConfig
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.plugins.cookies.AcceptAllCookiesStorage
import io.ktor.client.plugins.cookies.HttpCookies
import io.ktor.client.plugins.logging.LogLevel
import io.ktor.client.plugins.logging.Logging
import io.ktor.client.plugins.websocket.WebSockets
import io.ktor.client.request.HttpRequestBuilder
import io.ktor.serialization.kotlinx.json.json
import kotlinx.serialization.json.Json

@Suppress("MagicNumber")
actual fun createClient(
    configuration: Configuration,
    onAuth: AuthConfig.() -> Unit,
    onRequest: ((request: HttpRequestBuilder) -> Unit)?
): HttpClient {
    return HttpClient(Darwin) {
        followRedirects = configuration.followRedirects

        install(Auth) {
            onAuth()
        }

        if (configuration.enableLogs) {
            install(Logging) {
                level = LogLevel.ALL
            }
        }

        if (configuration.enableSocket) {
            install(WebSockets)
        }

        install(HttpTimeout) {
            connectTimeoutMillis = configuration.connectTimeoutMillis
            requestTimeoutMillis = configuration.requestTimeoutMillis
            socketTimeoutMillis = configuration.socketTimeoutMillis
        }

        if (null != onRequest) {
            install(InjectHeaderPlugin) {
                this.onRequest = onRequest
            }
        }

        install(ContentNegotiation) {
            json(configuration.json)
        }

        if (configuration.installCookies) {
            install(HttpCookies) {
                storage = configuration.cookieStorage ?: AcceptAllCookiesStorage()
            }
        }

        engine {
            proxy = configuration.proxyConfiguration?.proxyConfig
        }
    }
}
