package eu.codlab.http

import io.ktor.client.HttpClient
import io.ktor.client.engine.okhttp.OkHttp
import io.ktor.client.plugins.Charsets
import io.ktor.client.plugins.HttpTimeout
import io.ktor.client.plugins.auth.Auth
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.plugins.logging.LogLevel
import io.ktor.client.plugins.logging.Logger
import io.ktor.client.plugins.logging.Logging
import io.ktor.client.request.HttpRequestBuilder
import io.ktor.serialization.kotlinx.json.json
import kotlinx.serialization.json.Json

actual fun createClient(
    configuration: Configuration,
    onAuth: Auth.() -> Unit,
    onRequest: ((request: HttpRequestBuilder) -> Unit)?
): HttpClient {
    return HttpClient(OkHttp) {
        Charsets {
            register(Charsets.UTF_8)
        }

        if (configuration.enableLogs) {
            install(Logging) {
                level = LogLevel.ALL
            }
        }

        install(HttpTimeout) {
            connectTimeoutMillis = configuration.connectTimeoutMillis
            requestTimeoutMillis = configuration.requestTimeoutMillis
            socketTimeoutMillis = configuration.socketTimeoutMillis
        }

        install(Auth) {
            onAuth()
        }

        install(Logging) {
            logger = object : Logger {
                override fun log(message: String) {
                    println(message)
                }
            }
            level = LogLevel.ALL
        }
        @Suppress("MagicNumber")
        install(HttpTimeout) {
            requestTimeoutMillis = 12000
        }

        if (null != onRequest) {
            install(InjectHeaderPlugin) {
                this.onRequest = onRequest
            }
        }

        install(ContentNegotiation) {
            json(
                Json {
                    ignoreUnknownKeys = true
                    prettyPrint = true
                }
            )
        }
    }
}
