package eu.codlab.google.maps.utils

import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.CoroutineStart
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.cancel
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.launch
import kotlin.coroutines.CoroutineContext
import kotlin.coroutines.EmptyCoroutineContext
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException
import kotlin.coroutines.suspendCoroutine

class Queue {
    private val scope = MainScope()
    private val internalQueue = Channel<Job>(Channel.UNLIMITED)

    init {
        scope.launch(Dispatchers.Default) {
            for (job in internalQueue) {
                try {
                    job.join()
                } catch (ignored: Throwable) {
                    // ignored
                }
            }
        }
    }

    fun post(
        context: CoroutineContext = EmptyCoroutineContext,
        block: suspend CoroutineScope.() -> Unit
    ) = internalQueue.trySend(
        scope.launch(
            context,
            CoroutineStart.LAZY,
            block
        )
    )

    suspend fun <T> execute(
        context: CoroutineContext = EmptyCoroutineContext,
        block: suspend CoroutineScope.() -> T
    ) = suspendCoroutine { continuation ->
        post(context) {
            try {
                continuation.resume(block())
            } catch (err: Throwable) {
                err.printStackTrace()
                continuation.resumeWithException(err)
            }
        }
    }

    fun cancel() {
        internalQueue.cancel()
        scope.cancel()
    }
}
