package eu.codlab.google.maps

import eu.codlab.google.maps.models.GoogleMapsError
import eu.codlab.google.maps.models.LayerType
import eu.codlab.google.maps.models.MapType
import eu.codlab.google.maps.models.SessionParam
import eu.codlab.google.maps.models.SessionResult
import eu.codlab.google.maps.utils.Queue

class Session(
    private val apiKey: String
) {
    internal var enableQueue: Boolean = true
    internal var invalidSessionByDefault: Boolean = false

    private val queue = Queue()
    private var session: SessionResult? = if (invalidSessionByDefault) {
        SessionResult("", 0, 0, "jpeg", 0)
    } else {
        null
    }
    val internalSession = InternalSession()

    suspend fun getTile(
        col: Int,
        row: Int,
        zoomLevel: Int
    ): ByteArray {
        return internalGetTile(col, row, zoomLevel, true)
    }

    private suspend fun internalGetTile(
        col: Int,
        row: Int,
        zoomLevel: Int,
        retry: Boolean
    ): ByteArray {
        if (session == null) {
            createSession() ?: throw IllegalStateException("Can't' create session")
        }

        return try {
            internalSession.getTile(
                apiKey,
                session!!.session,
                row = row,
                col = col,
                zoomLevel = zoomLevel
            )
        } catch (err: GoogleMapsError) {
            if (!retry) {
                throw err
            }

            if (err.googleError.code == 401) {
                createSession() ?: throw IllegalStateException("Can't' create session")

                return internalGetTile(col, row, zoomLevel, false)
            } else {
                throw err
            }
        }
    }

    private suspend fun createSession(): SessionResult? {
        try {
            val param = SessionParam(
                mapType = MapType.Satellite,
                layerTypes = listOf(LayerType.Roadmap),
                language = "en-US",
                region = "US"
            )

            session = if (!enableQueue) {
                internalSession.createSession(apiKey, param)
            } else {
                queue.execute { internalSession.createSession(apiKey, param) }
            }

            return session
        } catch (err: Throwable) {
            err.printStackTrace()
            return null
        }
    }

}