/*
 * Copyright 2011 Chris de Vreeze
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.cdevreeze.springtx

import org.springframework.transaction.{ PlatformTransactionManager, TransactionStatus }
import org.springframework.transaction.support.{ TransactionCallback, TransactionCallbackWithoutResult }

/**
 * Scala wrapper for a Spring `TransactionTemplate`.
 * Trivial to use and understand for users of the Spring `TransactionTemplate` API, but more idiomatic Scala.
 *
 * @author Chris de Vreeze
 */
final class TransactionTemplate(
  val wrappedObject: org.springframework.transaction.support.TransactionTemplate) extends TransactionOperations {

  require(wrappedObject ne null)

  /** Handy constructor taking a PlatformTransactionManager directly */
  def this(transactionManager: PlatformTransactionManager) = this(new org.springframework.transaction.support.TransactionTemplate(transactionManager))

  def execute[T](action: TransactionStatus => T): T = {
    wrappedObject.execute(new TransactionCallback[T] {
      override def doInTransaction(status: TransactionStatus): T = action(status)
    })
  }

  def executeWithoutResult(action: TransactionStatus => Unit): Unit = {
    wrappedObject.execute(new TransactionCallbackWithoutResult {
      override def doInTransactionWithoutResult(status: TransactionStatus): Unit = action(status)
    })
  }
}
