/*
 * Copyright 2011 Chris de Vreeze
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.cdevreeze.springjdbc
package namedparam

import java.{ util => jutil }
import scala.collection.JavaConverters._
import org.springframework.jdbc.core.namedparam.{ SqlParameterSource, MapSqlParameterSource }

/**
 * Named typed query argument lists, used in the Scala `NamedParamJdbcTemplate`. The types are integer constants from `java.sql.Types`.
 *
 * @author Chris de Vreeze
 */
final class TypedArgMap(val typedArgs: Map[String, TypedArg]) extends Immutable {

  require(typedArgs ne null)

  def toSqlParameterSource: SqlParameterSource = {
    val result = new MapSqlParameterSource

    for ((argName, typedArgValue) <- typedArgs) {
      val argValue: AnyRef = typedArgValue.argValue
      val argType: Int = typedArgValue.argType
      result.addValue(argName, argValue, argType)
    }

    result
  }
}

object TypedArgMap {

  def apply(typedArgs: (String, TypedArg)*): TypedArgMap = new TypedArgMap(Map(typedArgs: _*))
}
