/*
 * Copyright 2011 Chris de Vreeze
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.cdevreeze.springjdbc
package namedparam

import java.{ util => jutil }
import java.sql.{ Connection, Statement, PreparedStatement }
import javax.sql.DataSource
import scala.collection.{ immutable, mutable }
import scala.collection.JavaConverters._
import scala.reflect.ClassManifest
import org.springframework.jdbc.core.{ JdbcOperations => _, JdbcTemplate => _, _ }
import org.springframework.jdbc.core.namedparam._

/**
 * Scala `NamedParamJdbcTemplate`, wrapping a Spring `NamedParameterJdbcTemplate`. Unlike the wrapped Spring counterpart, immutable Scala collections
 * are used instead of Java collections (and Options instead of null).
 *
 * Other than that, the API closely resembles the wrapped API. As a consequence, this wrapper API also heavily uses overloaded
 * methods. This implies that the Spring callback interfaces are used as parameters instead of Scala functions. Fortunately,
 * using factory methods in `JdbcTemplateUtils`, Scala functions can easily be converted to Spring JDBC callback interfaces.
 *
 * For the methods named `execute` in the wrapped Spring counterpart, if needed, use that wrapped Spring counterpart.
 *
 * @author Chris de Vreeze
 */
final class NamedParamJdbcTemplate(val wrappedObject: org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate) extends NamedParamJdbcOperations {

  /** Handy constructor taking a DataSource directly */
  def this(dataSource: DataSource) = this(new org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate(dataSource))

  // TODO Batch updates

  // Using typed args

  def query[A](sql: String, typedArgs: TypedArgMap, rse: ResultSetExtractor[A]): A = {
    wrappedObject.query(sql, typedArgs.toSqlParameterSource, rse)
  }

  def query[A](sql: String, typedArgs: TypedArgMap, rch: RowCallbackHandler): Unit = {
    wrappedObject.query(sql, typedArgs.toSqlParameterSource, rch)
  }

  def query[A](sql: String, typedArgs: TypedArgMap, rowMapper: RowMapper[A]): immutable.IndexedSeq[A] = {
    wrappedObject.query(sql, typedArgs.toSqlParameterSource, rowMapper).asScala.toIndexedSeq
  }

  // Using args without SQL types

  def query[A](sql: String, args: Map[String, AnyRef], rse: ResultSetExtractor[A]): A = {
    wrappedObject.query(sql, toJavaMap(args), rse)
  }

  def query[A](sql: String, args: Map[String, AnyRef], rch: RowCallbackHandler): Unit = {
    wrappedObject.query(sql, toJavaMap(args), rch)
  }

  def query[A](sql: String, args: Map[String, AnyRef], rowMapper: RowMapper[A]): immutable.IndexedSeq[A] = {
    wrappedObject.query(sql, toJavaMap(args), rowMapper).asScala.toIndexedSeq
  }

  // Query for Int

  def queryForInt(sql: String, typedArgs: TypedArgMap): Int = {
    wrappedObject.queryForInt(sql, typedArgs.toSqlParameterSource)
  }

  def queryForInt(sql: String, args: Map[String, AnyRef]): Int = {
    wrappedObject.queryForInt(sql, toJavaMap(args))
  }

  // Query for Long

  def queryForLong(sql: String, typedArgs: TypedArgMap): Long = {
    wrappedObject.queryForLong(sql, typedArgs.toSqlParameterSource)
  }

  def queryForLong(sql: String, args: Map[String, AnyRef]): Long = {
    wrappedObject.queryForLong(sql, toJavaMap(args))
  }

  // Query for Map (record)

  def queryForMap(sql: String, typedArgs: TypedArgMap): Map[String, AnyRef] = {
    val javaResult = wrappedObject.queryForMap(sql, typedArgs.toSqlParameterSource)
    javaResult.asScala.toMap
  }

  def queryForMap(sql: String, args: Map[String, AnyRef]): Map[String, AnyRef] = {
    val javaResult = wrappedObject.queryForMap(sql, toJavaMap(args))
    javaResult.asScala.toMap
  }

  // Query for Seq (sequence of records)

  def queryForSeq[A](sql: String, typedArgs: TypedArgMap, cls: Class[A]): immutable.IndexedSeq[A] = {
    wrappedObject.queryForList(sql, typedArgs.toSqlParameterSource, cls).asScala.toIndexedSeq
  }

  def queryForSeq[A](sql: String, typedArgs: TypedArgMap): immutable.IndexedSeq[Map[String, AnyRef]] = {
    val javaResult = wrappedObject.queryForList(sql, typedArgs.toSqlParameterSource)
    javaResult.asScala.toIndexedSeq map { (row: jutil.Map[String, AnyRef]) => row.asScala.toMap }
  }

  def queryForSeq[A](sql: String, args: Map[String, AnyRef], cls: Class[A]): immutable.IndexedSeq[A] = {
    wrappedObject.queryForList(sql, toJavaMap(args), cls).asScala.toIndexedSeq
  }

  def queryForSeq[A](sql: String, args: Map[String, AnyRef]): immutable.IndexedSeq[Map[String, AnyRef]] = {
    val javaResult = wrappedObject.queryForList(sql, toJavaMap(args))
    javaResult.asScala.toIndexedSeq map { (row: jutil.Map[String, AnyRef]) => row.asScala.toMap }
  }

  // Query for Object

  def queryForObject[A](sql: String, typedArgs: TypedArgMap, cls: Class[A]): A = {
    wrappedObject.queryForObject(sql, typedArgs.toSqlParameterSource, cls)
  }

  def queryForObject[A](sql: String, args: Map[String, AnyRef], cls: Class[A]): A = {
    wrappedObject.queryForObject(sql, toJavaMap(args), cls)
  }

  def queryForObject[A](sql: String, typedArgs: TypedArgMap, rowMapper: RowMapper[A]): A = {
    wrappedObject.queryForObject(sql, typedArgs.toSqlParameterSource, rowMapper)
  }

  def queryForObject[A](sql: String, args: Map[String, AnyRef], rowMapper: RowMapper[A]): A = {
    wrappedObject.queryForObject(sql, toJavaMap(args), rowMapper)
  }

  // Updates

  def update(sql: String, args: Map[String, AnyRef]): Int = {
    wrappedObject.update(sql, toJavaMap(args))
  }

  def update(sql: String, typedArgs: TypedArgMap): Int = {
    wrappedObject.update(sql, typedArgs.toSqlParameterSource)
  }

  // Private methods

  private def toJavaMap(m: Map[String, AnyRef]): jutil.Map[String, AnyRef] = new jutil.HashMap[String, AnyRef](m.asJava)
}
