/*
 * Copyright 2011 Chris de Vreeze
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.cdevreeze.springjdbc
package namedparam

import java.sql.{ Connection, Statement, PreparedStatement }
import scala.collection.immutable
import scala.reflect.ClassManifest
import org.springframework.jdbc.core.{ JdbcOperations => _, JdbcTemplate => _, _ }
import org.springframework.jdbc.core.namedparam._

/**
 * Scala `NamedParamJdbcOperations`, wrapping a Spring `NamedParameterJdbcOperations`. Unlike the wrapped Spring counterpart, immutable Scala collections
 * are used instead of Java collections (and Options instead of null).
 *
 * Other than that, the API closely resembles the wrapped API. As a consequence, this wrapper API also heavily uses overloaded
 * methods. This implies that the Spring callback interfaces are used as parameters instead of Scala functions. Fortunately,
 * using factory methods in `JdbcTemplateUtils`, Scala functions can easily be converted to Spring JDBC callback interfaces.
 *
 * For the methods named `execute` in the wrapped Spring counterpart, if needed, use that wrapped Spring counterpart.
 *
 * @author Chris de Vreeze
 */
trait NamedParamJdbcOperations {

  def wrappedObject: org.springframework.jdbc.core.namedparam.NamedParameterJdbcOperations

  // TODO Batch updates

  // Using typed args

  def query[A](sql: String, typedArgs: TypedArgMap, rse: ResultSetExtractor[A]): A

  def query[A](sql: String, typedArgs: TypedArgMap, rch: RowCallbackHandler): Unit

  def query[A](sql: String, typedArgs: TypedArgMap, rowMapper: RowMapper[A]): immutable.IndexedSeq[A]

  // Using args without SQL types

  def query[A](sql: String, args: Map[String, AnyRef], rse: ResultSetExtractor[A]): A

  def query[A](sql: String, args: Map[String, AnyRef], rch: RowCallbackHandler): Unit

  def query[A](sql: String, args: Map[String, AnyRef], rowMapper: RowMapper[A]): immutable.IndexedSeq[A]

  // Query for Int

  def queryForInt(sql: String, typedArgs: TypedArgMap): Int

  def queryForInt(sql: String, args: Map[String, AnyRef]): Int

  // Query for Long

  def queryForLong(sql: String, typedArgs: TypedArgMap): Long

  def queryForLong(sql: String, args: Map[String, AnyRef]): Long

  // Query for Map (record)

  def queryForMap(sql: String, typedArgs: TypedArgMap): Map[String, AnyRef]

  def queryForMap(sql: String, args: Map[String, AnyRef]): Map[String, AnyRef]

  // Query for Seq (sequence of records)

  def queryForSeq[A](sql: String, typedArgs: TypedArgMap, cls: Class[A]): immutable.IndexedSeq[A]

  def queryForSeq[A](sql: String, typedArgs: TypedArgMap): immutable.IndexedSeq[Map[String, AnyRef]]

  def queryForSeq[A](sql: String, args: Map[String, AnyRef], cls: Class[A]): immutable.IndexedSeq[A]

  def queryForSeq[A](sql: String, args: Map[String, AnyRef]): immutable.IndexedSeq[Map[String, AnyRef]]

  // Query for Object

  def queryForObject[A](sql: String, typedArgs: TypedArgMap, cls: Class[A]): A

  def queryForObject[A](sql: String, args: Map[String, AnyRef], cls: Class[A]): A

  def queryForObject[A](sql: String, typedArgs: TypedArgMap, rowMapper: RowMapper[A]): A

  def queryForObject[A](sql: String, args: Map[String, AnyRef], rowMapper: RowMapper[A]): A

  // Updates

  def update(sql: String, args: Map[String, AnyRef]): Int

  def update(sql: String, typedArgs: TypedArgMap): Int
}
