/*
 * Copyright 2011 Chris de Vreeze
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.cdevreeze.springjdbc

import java.sql.{ Connection, Statement, PreparedStatement }
import scala.collection.immutable
import scala.reflect.ClassManifest
import org.springframework.jdbc.core.{ JdbcOperations => _, JdbcTemplate => _, _ }

/**
 * Scala `JdbcOperations`, wrapping a Spring `JdbcOperations`. Unlike the wrapped Spring counterpart, immutable Scala collections
 * are used instead of Java collections (and Options instead of null).
 *
 * Other than that, the API closely resembles the wrapped API. As a consequence, this wrapper API also heavily uses overloaded
 * methods. This implies that the Spring callback interfaces are used as parameters instead of Scala functions. Fortunately,
 * using factory methods in `JdbcTemplateUtils`, Scala functions can easily be converted to Spring JDBC callback interfaces.
 *
 * For the methods named `execute` in the wrapped Spring counterpart, if needed, use that wrapped Spring counterpart.
 * The same holds for some of the "callback-based" `query` methods.
 *
 * This API is more regular in the passed query arguments than the wrapped Spring API.
 * For example, in `query` methods, typically the first parameter is the SQL string, the second parameter concerns query parameters,
 * and the third parameter, if any, concerns result set handling. Unlike the wrapped Spring counterpart, a parameter concerning
 * result set handling never comes before parameters concerning query parameters.
 *
 * @author Chris de Vreeze
 */
trait JdbcOperations {

  def wrappedObject: org.springframework.jdbc.core.JdbcOperations

  // Batch updates

  def batchUpdate(sqlSeq: immutable.Seq[String]): immutable.IndexedSeq[Int]

  def batchUpdate(sql: String, pss: BatchPreparedStatementSetter): immutable.IndexedSeq[Int]

  // Using typed args

  def query[A](sql: String, typedArgs: TypedArgSeq, rse: ResultSetExtractor[A]): A

  def query[A](sql: String, typedArgs: TypedArgSeq, rch: RowCallbackHandler): Unit

  def query[A](sql: String, typedArgs: TypedArgSeq, rowMapper: RowMapper[A]): immutable.IndexedSeq[A]

  // Using args without SQL types

  def query[A](sql: String, args: immutable.Seq[AnyRef], rse: ResultSetExtractor[A]): A

  def query[A](sql: String, args: immutable.Seq[AnyRef], rch: RowCallbackHandler): Unit

  def query[A](sql: String, args: immutable.Seq[AnyRef], rowMapper: RowMapper[A]): immutable.IndexedSeq[A]

  // Using PreparedStatementSetters as "arguments"

  def query[A](sql: String, pss: PreparedStatementSetter, rse: ResultSetExtractor[A]): A

  def query[A](sql: String, pss: PreparedStatementSetter, rch: RowCallbackHandler): Unit

  def query[A](sql: String, pss: PreparedStatementSetter, rowMapper: RowMapper[A]): immutable.IndexedSeq[A]

  // Query for Int

  def queryForInt(sql: String, typedArgs: TypedArgSeq): Int

  def queryForInt(sql: String, args: immutable.Seq[AnyRef]): Int

  // Query for Long

  def queryForLong(sql: String, typedArgs: TypedArgSeq): Long

  def queryForLong(sql: String, args: immutable.Seq[AnyRef]): Long

  // Query for Map (record)

  def queryForMap(sql: String, typedArgs: TypedArgSeq): Map[String, AnyRef]

  def queryForMap(sql: String, args: immutable.Seq[AnyRef]): Map[String, AnyRef]

  // Query for Seq (sequence of records)

  def queryForSeq[A](sql: String, typedArgs: TypedArgSeq, cls: Class[A]): immutable.IndexedSeq[A]

  def queryForSeq[A](sql: String, typedArgs: TypedArgSeq): immutable.IndexedSeq[Map[String, AnyRef]]

  def queryForSeq[A](sql: String, args: immutable.Seq[AnyRef], cls: Class[A]): immutable.IndexedSeq[A]

  def queryForSeq[A](sql: String, args: immutable.Seq[AnyRef]): immutable.IndexedSeq[Map[String, AnyRef]]

  // Query for Object

  def queryForObject[A](sql: String, typedArgs: TypedArgSeq, cls: Class[A]): A

  def queryForObject[A](sql: String, args: immutable.Seq[AnyRef], cls: Class[A]): A

  def queryForObject[A](sql: String, typedArgs: TypedArgSeq, rowMapper: RowMapper[A]): A

  def queryForObject[A](sql: String, args: immutable.Seq[AnyRef], rowMapper: RowMapper[A]): A

  // Updates

  def update(sql: String, args: immutable.Seq[AnyRef]): Int

  def update(sql: String, typedArgs: TypedArgSeq): Int

  def update(sql: String, pss: PreparedStatementSetter): Int
}
