/*
 * Copyright 2011 Chris de Vreeze
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package eu.cdevreeze.springjdbc

import java.sql.{ ResultSet, Connection, Statement, PreparedStatement, CallableStatement }
import org.springframework.jdbc.core.{ JdbcOperations => _, JdbcTemplate => _, _ }

/**
 * Conversions that are useful in the context of a Spring
 * `JdbcTemplate`, to make the application code slightly more Scala-friendly.
 *
 * @author Chris de Vreeze
 */
object JdbcTemplateUtils {

  // Callbacks, given a Connection or Statement or PreparedStatement, etc.

  def newConnectionCallback[A](action: Connection => A): ConnectionCallback[A] =
    new ConnectionCallback[A] {
      def doInConnection(con: Connection): A = action(con)
    }

  def newStatementCallback[A](action: Statement => A): StatementCallback[A] =
    new StatementCallback[A] {
      def doInStatement(stmt: Statement): A = action(stmt)
    }

  def newPreparedStatementCallback[A](action: PreparedStatement => A): PreparedStatementCallback[A] =
    new PreparedStatementCallback[A] {
      def doInPreparedStatement(ps: PreparedStatement): A = action(ps)
    }

  def newCallableStatementCallback[A](action: CallableStatement => A): CallableStatementCallback[A] =
    new CallableStatementCallback[A] {
      def doInCallableStatement(cs: CallableStatement): A = action(cs)
    }

  // Handling result sets

  def newResultSetExtractor[A](extractor: ResultSet => A): ResultSetExtractor[A] = {
    new ResultSetExtractor[A] {
      def extractData(rs: ResultSet): A = extractor(rs)
    }
  }

  def newRowCallbackHandler(handler: ResultSet => Unit): RowCallbackHandler = {
    new RowCallbackHandler {
      def processRow(rs: ResultSet): Unit = handler(rs)
    }
  }

  def newRowMapper[A](mapper: (ResultSet, Int) => A): RowMapper[A] = {
    new RowMapper[A] {
      def mapRow(rs: ResultSet, rowNum: Int): A = mapper(rs, rowNum)
    }
  }

  // Setting parameters on PreparedStatements etc.

  def newPreparedStatementSetter(psSetter: PreparedStatement => Unit): PreparedStatementSetter = {
    new PreparedStatementSetter {
      def setValues(ps: PreparedStatement): Unit = psSetter(ps)
    }
  }

  def newBatchPreparedStatementSetter(batchSize: Int)(psSetter: (PreparedStatement, Int) => Unit): BatchPreparedStatementSetter = {
    new BatchPreparedStatementSetter {
      def getBatchSize: Int = batchSize
      def setValues(ps: PreparedStatement, i: Int): Unit = psSetter(ps, i)
    }
  }

  // Creating PreparedStatements etc.

  def newPreparedStatementCreator(psCreator: Connection => PreparedStatement): PreparedStatementCreator = {
    new PreparedStatementCreator {
      def createPreparedStatement(con: Connection): PreparedStatement = psCreator(con)
    }
  }

  def newCallableStatementCreator(csCreator: Connection => CallableStatement): CallableStatementCreator = {
    new CallableStatementCreator {
      def createCallableStatement(con: Connection): CallableStatement = csCreator(con)
    }
  }
}
