/*
 * Copyright 2017-2019 John A. De Goes and the ZIO Contributors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package zio.stream

import zio._
import zio.clock.Clock
import zio.Cause

object Stream extends ZStreamPlatformSpecific {

  /**
   * See [[ZStream.empty]]
   */
  final val empty: Stream[Nothing, Nothing] =
    ZStream.empty

  /**
   * See [[ZStream.never]]
   */
  final val never: Stream[Nothing, Nothing] =
    ZStream.never

  /**
   * See [[ZStream.apply]]
   */
  final def apply[A](as: A*): Stream[Nothing, A] = ZStream(as: _*)

  /**
   * See [[ZStream.bracket]]
   */
  final def bracket[E, A](acquire: IO[E, A])(release: A => UIO[_]): Stream[E, A] =
    ZStream.bracket(acquire)(release)

  /**
   * See [[ZStream.die]]
   */
  final def die(ex: Throwable): Stream[Nothing, Nothing] =
    ZStream.die(ex)

  /**
   * See [[ZStream.dieMessage]]
   */
  final def dieMessage(msg: String): Stream[Nothing, Nothing] =
    ZStream.dieMessage(msg)

  /**
   * See [[ZStream.effectAsync]]
   */
  final def effectAsync[E, A](
    register: (IO[Option[E], A] => Unit) => Unit,
    outputBuffer: Int = 16
  ): Stream[E, A] =
    ZStream.effectAsync(register, outputBuffer)

  /**
   * See [[ZStream.effectAsyncMaybe]]
   */
  final def effectAsyncMaybe[E, A](
    register: (IO[Option[E], A] => Unit) => Option[Stream[E, A]],
    outputBuffer: Int = 16
  ): Stream[E, A] =
    ZStream.effectAsyncMaybe(register, outputBuffer)

  /**
   * See [[ZStream.effectAsyncM]]
   */
  final def effectAsyncM[E, A](
    register: (IO[Option[E], A] => Unit) => IO[E, _],
    outputBuffer: Int = 16
  ): Stream[E, A] =
    ZStream.effectAsyncM(register, outputBuffer)

  /**
   * See [[ZStream.effectAsyncInterrupt]]
   */
  final def effectAsyncInterrupt[E, A](
    register: (IO[Option[E], A] => Unit) => Either[Canceler, Stream[E, A]],
    outputBuffer: Int = 16
  ): Stream[E, A] =
    ZStream.effectAsyncInterrupt(register, outputBuffer)

  /**
   * See [[ZStream.fail]]
   */
  final def fail[E](error: E): Stream[E, Nothing] =
    ZStream.fail(error)

  /**
   * See [[ZStream.finalizer]]
   */
  final def finalizer(finalizer: UIO[_]): Stream[Nothing, Nothing] =
    ZStream.finalizer(finalizer)

  /**
   * See [[ZStream.flatten]]
   */
  final def flatten[E, A](fa: Stream[E, Stream[E, A]]): Stream[E, A] =
    ZStream.flatten(fa)

  /**
   * See [[ZStream.flattenPar]]
   */
  final def flattenPar[E, A](n: Int, outputBuffer: Int = 16)(
    fa: Stream[E, Stream[E, A]]
  ): Stream[E, A] =
    ZStream.flattenPar(n, outputBuffer)(fa)

  /**
   * See [[ZStream.fromChunk]]
   */
  final def fromChunk[@specialized A](c: Chunk[A]): Stream[Nothing, A] =
    ZStream.fromChunk(c)

  /**
   * See [[ZStream.fromEffect]]
   */
  final def fromEffect[E, A](fa: IO[E, A]): Stream[E, A] =
    ZStream.fromEffect(fa)

  /**
   * See [[ZStream.repeatEffect]]
   */
  final def repeatEffect[E, A](fa: IO[E, A]): Stream[E, A] =
    ZStream.repeatEffect(fa)

  /**
   * See [[ZStream.repeatEffectWith]]
   */
  final def repeatEffectWith[E, A](
    fa: IO[E, A],
    schedule: Schedule[Unit, _]
  ): ZStream[Clock, E, A] = ZStream.repeatEffectWith(fa, schedule)

  /**
   * See [[ZStream.fromIterable]]
   */
  final def fromIterable[A](as: Iterable[A]): Stream[Nothing, A] =
    ZStream.fromIterable(as)

  /**
   * See [[ZStream.fromQueue]]
   */
  final def fromQueue[E, A](queue: ZQueue[_, _, Any, E, _, A]): Stream[E, A] =
    ZStream.fromQueue(queue)

  /**
   * See [[ZStream.halt]]
   */
  final def halt[E](cause: Cause[E]): Stream[E, Nothing] = fromEffect(ZIO.halt(cause))

  /**
   * See [[ZStream.managed]]
   */
  final def managed[E, A](managed: Managed[E, A]): Stream[E, A] =
    ZStream.managed(managed)

  /**
   * See [[ZStream.mergeAll]]
   */
  final def mergeAll[E, A](n: Int, outputBuffer: Int = 16)(
    streams: Stream[E, A]*
  ): Stream[E, A] =
    ZStream.mergeAll[Any, E, A](n, outputBuffer)(streams: _*)

  /**
   * See [[ZStream.range]]
   */
  final def range(min: Int, max: Int): Stream[Nothing, Int] =
    ZStream.range(min, max)

  /**
   * See [[ZStream.succeed]]
   */
  final def succeed[A](a: A): Stream[Nothing, A] =
    ZStream.succeed(a)

  /**
   * See [[ZStream.succeedLazy]]
   */
  final def succeedLazy[A](a: => A): Stream[Nothing, A] =
    ZStream.succeedLazy(a)

  /**
   * See [[ZStream.unfold]]
   */
  final def unfold[S, A](s: S)(f0: S => Option[(A, S)]): Stream[Nothing, A] =
    ZStream.unfold(s)(f0)

  /**
   * See [[ZStream.unfoldM]]
   */
  final def unfoldM[E, A, S](s: S)(f0: S => IO[E, Option[(A, S)]]): Stream[E, A] =
    ZStream.unfoldM(s)(f0)

  /**
   * See [[ZStream.unwrap]]
   */
  final def unwrap[E, A](fa: IO[E, Stream[E, A]]): Stream[E, A] =
    ZStream.unwrap(fa)
}
